/* * mksfs.c - Make a sfs (steganographic) filesystem.  *
 * Copyright (C) 1998 Carl van Schaik (carl@leg.uct.ac.za)
 *                    Paul Smeddle (psmeddle@cs.uct.ac.za)
 * SFS - [S]teganographic [F]ile [S]ystem
 *
 *
 * %Begin-Header%
 * This file may be redistributed under the terms of the GNU Public
 * License.
 * %End-Header%
 */

/* Usage: mksfs [options] device
 * 
 * mksfs by Carl van Schaik
 * ALL code except where noted, by Carl van Schaik
 */

/* We need to include a lot of things */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <termios.h>
#include <time.h>
#include <errno.h>
#include <mntent.h>
#include <malloc.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

/* Include VFS and SFS information */
#include <linux/fs.h>
#include <linux/sfs_fs.h>

/* Includes for libsfs */
#include <crypt_funcs.h>
#include <rand.h>  /* Strong Random Number Generator */
#include <sfslib.h>

/* Other needed information for comandline options */
char 	*program_name = "mksfs";
int disk_check = 1;
int sim_bad_blocks = 0;
int disk_randomize = 1;
int bluff_blks = SFS_PRE_ALLOC;


/* the unit of the dynamic bad-block linked list */
struct bad_list
{
	void * next;
	word32 block;
};

struct bad_list *bad_start; /* Start of the dynamic bad-block list */
struct bad_list *bad_item;  /* For scanning along the list */


/* add an entry to the bad block inode */
static void b_inode_addentry(struct sfs_inode inode, word32 blk)
{
	word32 disk_block[SFS_BLK/4];
	word32 pos;
	int    offs = offset/SFS_BLK;

	printf("Marking bad block %li\n",blk);
	set_block_used(blk);   /* Mark bad block as used */

	/* Inserting the bad blocks into the bad block inode */
	/* This is compilcated as it contains direct, single and doublely linked lists */

	/* Direct mapping blocks are easy */
	if (inode.i_reserved[0] < 96)
	  inode.i_direct[inode.i_reserved[0]] = blk;
	/* Singlely linked blocks */
	else /* If a block already exists for the bad block we are inserting */
	     /* we simply open the block and add it */
	  if ((((inode.i_reserved[0]-96) % (SFS_BLK/4)) != 0) &&
                ((inode.i_reserved[0]-96) < (80*(SFS_BLK/4))))
	    {
		read_data_block(inode.i_single[(inode.i_reserved[0]-96)
		  / (SFS_BLK/4)],(char*)disk_block);

		/* record the bad blocks position */
		disk_block[(inode.i_reserved[0]-96) % (SFS_BLK/4)] = blk;

		write_data_block(inode.i_single[(inode.i_reserved[0]-96)
		  / (SFS_BLK/4)],(char*)disk_block);
	    }
	else
	  /* we have to create a new singlely linked block */
	  if ((((inode.i_reserved[0]-96) % (SFS_BLK/4)) == 0) &&
		((inode.i_reserved[0]-96) < (80*(SFS_BLK/4))))
	    {
	      disk_block[0] = blk;
       	      read_block(offs, (char *)&sb);
	      if (sfs_verify(sb))         
	        die("superblock encryption test failed");         

	      if (sb.s_free_blocks == 0)
		die ("mark bad block : no disk space left");

	      /* Get a random position */
	      pos = genrand() % sb.s_block_total;
	      while (is_block_used(pos))
	        pos = (++pos) % sb.s_block_total;

	      /* Marks the bad blocks position */
	      inode.i_single[(inode.i_reserved[0]-96) / (SFS_BLK/4)] = pos;
	      /* Write the new block to disk setting its psotion used */
	      set_block_used(pos);
	      write_data_block(inode.i_single[(inode.i_reserved[0]-96)	
		/ (SFS_BLK/4)],(char*)disk_block);
	    }
	else
	  /* Into the doublely linked lists */
	  /* This will happen if we have more than 80MB of bad disk */
	  /* I will write this when I get more time */
	  die("too many bad blocks for me");
}


/* Make the bad block inode */
static void create_bad_inode()
{
	unsigned int i, pos;
	struct sfs_inode    inode;
	time_t t;
	int    offs = offset/SFS_BLK;
	struct bad_list *temp;

	printf("Creating bad block inode\n");
	t = time(NULL);
	/* Fill in the required details */
	inode.i_crypt_id_1 = INODE_CRYPT_ID_1;
	inode.i_crypt_id_2 = INODE_CRYPT_ID_2;
	inode.i_blocks     = 0;
	inode.i_type       = SFS_BAD_INO;
	inode.i_uid        = sb.s_default_uid;
	inode.i_gid        = sb.s_default_gid;
	inode.i_file_size  = 0;
	inode.i_ctime      = t;
	inode.i_atime      = t;
	inode.i_mtime      = t;
	inode.i_link_count = 1;
        inode.i_perms	   = 0000;
	for (i=0;i<3;i++)
	  inode.i_reserved[i] = genrand();

	/* reset to start of bad-block link list */
        bad_item = bad_start;
	inode.i_reserved[0] = 0;
	/* Process every recorded bad block */
        while (bad_item->next != 0)
          {
	    b_inode_addentry(inode, bad_item->block);
	    inode.i_reserved[0]++;
            bad_item = bad_item->next;

	    temp = bad_start;
	    bad_start = bad_start->next;
	    free(temp);
          }

	/* Check if disk is full */
	read_block(offs, (char *)&sb);
	if (sfs_verify(sb))
	  die("superblock encryption test failed");

	if (sb.s_free_blocks == 0)
	  die("disk full");

	/* Find a free position on the disk */
	pos = genrand() % sb.s_block_total;
	while (is_block_used(pos))
	  pos = (++pos) % sb.s_block_total;

	set_block_used(pos);
	write_data_block(pos, (char *)&inode);

	read_block(offs, (char *)&sb);
	if (sfs_verify(sb))
	  die("superblock encryption test failed");

	sb.s_bad_inode = pos;
	write_block(offs, (char *)&sb);
}

/* Make the root inode, and commit it to disk */
static void create_root_dir()
{
	int i;
	unsigned int pos, dpos;
	time_t t;
	struct sfs_inode    inode;
	struct sfs_dir_entry *de;
	char dirblock[SFS_BLK];
	int    offs = offset/SFS_BLK;

	printf("Writing root inode\n");
	t = time(NULL);

	inode.i_crypt_id_1 = INODE_CRYPT_ID_1;
        inode.i_crypt_id_2 = INODE_CRYPT_ID_2;
	inode.i_blocks     = 1;
	inode.i_type       = SFS_ROOT_INO;
	inode.i_uid        = sb.s_default_uid;
        inode.i_gid        = sb.s_default_gid;
	inode.i_file_size  = inode.i_blocks*SFS_BLK;
        inode.i_ctime      = t;
        inode.i_atime      = t;
        inode.i_mtime      = t;
	inode.i_link_count = 2;
        inode.i_perms      = 0755;
        for (i=0;i<3;i++)
          inode.i_reserved[i] = genrand();

	read_block(offs, (char *)&sb);
        if (sfs_verify(sb))
          die("superblock encryption test failed");

	/* Check if we have enough disk space */
	if (sb.s_free_blocks <= 1)
	  die("disk full");

	pos = genrand() % sb.s_block_total;
	while (is_block_used(pos))
	  pos = (++pos) % sb.s_block_total;

	/* Write the root inode to disk */
	set_block_used(pos);
        write_data_block(pos, (char *)&inode);

	/* Create the directory entry */
        memset(dirblock,0,SFS_BLK);
        de = (struct sfs_dir_entry *)(dirblock);

	/* Create the '.' directory */
        de->inode_p = pos;
	de->rec_len = 4+2+2+1; /* 32bits + 16 + 16 + `.' */
	de->name_len = 1;
	strcpy(de->name,".");

	/* Make ".." */
        de = (struct sfs_dir_entry *)(dirblock + de->rec_len);
        de->inode_p = pos;
        de->rec_len = 4+2+2+2; /* 32bits + 16 + 16 + `..' */
        de->name_len = 2;
        strcpy(de->name,"..");

	/* Update the superblock */
        read_block(offs, (char *)&sb);
        if (sfs_verify(sb))
          die("superblock encryption test failed");
	sb.s_root_ptr = pos;

	write_block(offs, (char *)&sb);

	/* Update the root inode and write the directory */
	dpos = genrand() % sb.s_block_total;
	while (is_block_used(dpos))
	  dpos = (++dpos) % sb.s_block_total;
	set_block_used(dpos);

	inode.i_direct[0] = dpos;
	inode.i_dir_entries = 2;
	write_data_block(pos, (char *)&inode);
	write_data_block(dpos, (char*)&dirblock);
}


/* Creates the group blocks on the disk */
static void create_groups()
{
	unsigned int group = 0;
	unsigned int pos, i;

	sb.s_free_blocks =          /* blocks used for superblocks/bitmaps */
	  ( sb.s_block_total - (sb.s_group_no*2 + sb.s_first_block));

	pos = offset/SFS_BLK;
	printf("Creating groups...\n");

	while (group < sb.s_group_no)
	  {
	    sb.s_this_group_no = group; /* Set curent group */
	    /* Clear the bitmap and mark the superblock and bitmap used */
	    memset(bitmap,0,SFS_BLK);
	    bitset(bitmap,0);
	    bitset(bitmap,1);

	    /* If this is the last group */
	    if (group == (sb.s_group_no-1))
	      {
		for(i=0;i<SFS_GROUP;i++)
		  { /* Mark blocks past the end of the disk used */
		    if ((i+pos) >= sb.s_block_total)
		    bitset(bitmap,i);
		  }
	      }

            write_block(pos, (char *)&sb);
	    write_block(pos+1,(char *)&bitmap);

	    printf("%i out of %i done\r",group+1,sb.s_group_no);
	    fflush(stdout);

	    group++;
	    pos = (offset/SFS_BLK) + (group*SFS_GROUP);
	  }
}

/* Writes random data to the entire disk surface */
static void randomize_platter()
{
	unsigned int i, j;
	char junk[SFS_BLK];
	word32 temp;

	if (lseek(device,offset,SEEK_SET) != offset)
	    die("seek to first data block failed");

        printf("Randomizing disk surface...\n");

	sgenrand(time(NULL)+three_way_key[1]);  /* set random seed */

	for (i=sb.s_first_block;i<sb.s_block_total;i++)
	{
	  /* Good enough random generator for us */
	  for(j=0;j<SFS_BLK/4;j++)
	     {
		temp = genrand();
		junk[j*4+0] = (char)(temp);
		junk[j*4+1] = (char)(temp >> 8);
		junk[j*4+2] = (char)(temp >> 16);
		junk[j*4+3] = (char)(temp >> 24);
	      }

	if ((i % 1024) == 0)
	  {
	    printf("%i%% or %i MB done\r",(i*100 / sb.s_block_total),i/1024);
            fflush(stdout); 
	  }

	  /* Write the junk to disk */
	  if (SFS_BLK != write(device,junk,SFS_BLK))
	    printf ("Hit bad block\n");
	}
}

char* bitarray;

/* (Fast) Checks if a specified block is used or not */
/* Optimised pointer version of is_block_used */
/* Gives +- 40000% speed increase (uses no disk accesses) */
int f_is_block_used(unsigned int blk)
{
	int offs = offset/SFS_BLK;
	unsigned int i,o;

	i = (blk-offs) / SFS_GROUP;
	o = (blk-offs) % SFS_GROUP;

	if (i > sb.s_group_no)
	  return (1);
	if (blk > sb.s_block_total)
	  return (1);

	return (is_bit_set(bitarray + i*SFS_BLK, o));
}

/* (fast) Sets a block marked used on the disk */
/* Optimised set_block_used routine */
void f_set_block_used(unsigned int blk)
{
	int    offs = offset/SFS_BLK;
	unsigned int i,o;

	i = (blk-offs) / SFS_GROUP;
	o = (blk-offs) % SFS_GROUP;

	if (i > sb.s_group_no)
	  die("tried to read past end of fs");

	if (is_bit_set(bitarray + i*SFS_BLK, o))
	  die("tried to set bit that already set, corruption warning!");

	bitset(bitarray + i*SFS_BLK, o);

	sb.s_free_blocks--;
}


/* Create random used blocks to cover information */
static void create_noise()
{
	unsigned int offs = offset/SFS_BLK;
	unsigned int i,t, pos;

	read_block(offs, (char *)&sb);
	if (sfs_verify(sb))
	  die("superblock encryption test failed");

	t = (sb.s_free_blocks/100)*bluff_blks;
	printf("Filling %i%% of disk with nothing\n",bluff_blks);

	/* Try allocate memory for a disk buffer */
	bitarray = (char*)malloc(sb.s_group_no*SFS_BLK);
	if (bitarray == NULL)
	  {
	    printf("Not enough memory to buffer bitmaps\n");
	    printf("Performing non-cached operation (this may take a while)\n");
	    for (i=0;i<t;i++)
	      {
		printf("%i or %i \r",i,t);
		fflush(stdout);
		pos = genrand() % sb.s_block_total;
		while (is_block_used(pos))
		  pos = (++pos) % sb.s_block_total;

		set_block_used(pos);
              }
	  }
	else
	  {
		/* Read in the buffer */
		for (i=0;i<sb.s_group_no;i++)
		  read_block(offset/SFS_BLK+i*SFS_GROUP+1,
			(char*)&bitarray[i*SFS_BLK]);

		for (i=0;i<t;i++)
		  {
		    pos = genrand() % sb.s_block_total;
		    while (f_is_block_used(pos))
		      pos = (++pos) % sb.s_block_total;

		    f_set_block_used(pos);
		  }

		/* Write buffer to disk */
		for (i=0;i<sb.s_group_no;i++)
		  write_block(offset/SFS_BLK+i*SFS_GROUP+1,
			(char*)&bitarray[i*SFS_BLK]);

		/* This was used for testing */
		/* for (i=0;i<SFS_BLK;i++)
		  printf("%i",f_is_block_used(i));
		printf("\n");*/

		free(bitarray);
	  }

	/* Finnish off by updating superblocks */
	write_block(offs, (char *)&sb);
	sync_super_blocks();
}


/* Initialise the super block */
static void init_super_block()
{
	int i;

	/* Fill in the nessasary information */
        sb.s_crypt_id_1   = SUPER_CRYPT_ID_1;
        sb.s_crypt_id_2   = SUPER_CRYPT_ID_2;
	sb.s_block_total  = get_size();
	sb.s_magic_number = SFS_SUPER_MAGIC;
	sb.s_first_block  = SFS_BOOT_SECT_SIZE / SFS_BLK;
	sb.s_free_blocks  = sb.s_block_total-sb.s_first_block;
	sb.s_default_uid  = 0; /* This */
	sb.s_default_gid  = 0; /* and this will be done at a later stage */
	sb.s_root_ptr     = 0;
	sb.s_bad_inode    = 0;
	for (i=0; i<15; i++)
	  sb.s_vol_name[i] = volume_label[i];
	sb.s_vol_name[15] = 0; /* Terminate the string */
	sb.s_group_no     = sb.s_block_total / SFS_GROUP;

	if ((sb.s_block_total % SFS_GROUP) > 0)
	  sb.s_group_no++;


	sb.s_this_group_no = 0;

	for (i=0;i<240;i++)
	  sb.s_reserved[i] = genrand();
}


/*
 * Perform a test of a block; return the number of
 * blocks readable/writeable.
 */
long do_check(char * buffer, int try, unsigned int current_block)
{
        long got;

        /* Seek to the correct loc. */
        if (lseek(device, current_block * SFS_BLK, SEEK_SET) !=
                          current_block * SFS_BLK ) {
                 die("seek failed during testing of blocks");
        }


        /* Try the read */
	if (disk_check)
          got = read(device, buffer, try * SFS_BLK);
	else
	  got = try*SFS_BLK;
        if (got < 0) got = 0;
        if (got & (SFS_BLK - 1 )) {
                printf("Weird values in do_check: probably bugs\n");
        }
        got /= SFS_BLK;

	if (sim_bad_blocks)
	  if ((genrand()/1024/1024) < 16)
	    return 0;
        return got;
}


/* Marks a block bad by inserting to the end of the list*/
static void mark_block(unsigned int blk)
{
	struct bad_list *temp;

	bad_item = bad_start;
	while (bad_item->next != 0)
	    bad_item = bad_item->next;

	bad_item->block = blk;
	temp = malloc (8);
	temp->next  = 0;
	bad_item->next = temp;
	bad_item = bad_item->next;
}


/* Scans disk for bad blocks, and adds them to bad block queue */
static void check_blocks()
{
	int try, got ,badblocks = 0;
	static char buffer[SFS_BLK * 16];
	unsigned int current=0;

	printf("Testing disk surface\n");

	sgenrand(time(NULL));  /* set seed to time */
	while (current < sb.s_block_total) {
	        if (lseek(device,current*SFS_BLK,SEEK_SET) !=
	          current*SFS_BLK)
	                die("seek failed in check_blocks");
	        try = 16;
	        if (current + try >= sb.s_block_total)
	                try = sb.s_block_total-current;
	        got = do_check(buffer, try, current);
	        current += got;

	        if ((current % 1024) == 0)
	          {
	            printf("%i%% or %i MB done\r",(current*100 / sb.s_block_total),current/1024);
	            fflush(stdout);
	          }

	        if (got == try)
	                continue;
	        if (((current-(sb.s_first_block)) % (SFS_GROUP)) < 2)
	                die("bad blocks in superblock area: cannot make fs");
		else
		  printf("Found bad block at %i\n",current);
	        mark_block(current);
	        badblocks++;
	        current++;
	}
	if (badblocks)
	        printf("%d bad blocks found\n",badblocks);
}


/* Ask the user weather he is really sure */
static int ask_user()
{
	char k;

        printf("Continuing will erase all information on the disk, are you sure? ");
        fflush(stdout);

        k = getchar();

        if ((k != 'y') && (k != 'Y'))
        {
          printf("\nNo changes were made to %s\n",device_name);
	  clean_up();
          close(device);
          return (1);
        }

	return(0);
}


/* Displays the programs options
 */
static void show_options()
{
	printf("USAGE:  mksfs [options] <dev>\n");
	printf("	Options : -V name : volume-name\n");
	printf("		  -k      : display encryption keys\n");
	printf("		  -i      : show fs information on completion\n");
	printf("		  -c <num>: percent of free disk space to use for cover\n");
	printf("			    (default is %i%% of disk free)\n",bluff_blks);
	printf("	-- options only to be used for testing --\n");
	printf("		  -f      : force no bad block check\n");
	printf("		  -n      : force no disk randomize\n");
	printf("		  -s      : simulate bad blocks\n");
}


/* This checks if all the options were gathered properly */
static int check_options()
{
	if (got_device != 1)
	  return(0);

	return(1);
}

/* Process command line arguments */
static void PRS(int argc, char *argv[])
{
	char k,l;

	printf("[0;1;37;33m");
	printf("mksfs v0.0, Steganographic File System\n");
	printf("[0;1;37;00m");
	printf("(C) 1998 Paul Smeddle (psmeddle@cs.uct.ac.za)\n");
	printf("         Carl van Schaik (carl@leg.uct.ac.za)\n");

	/* clear the first variable (the program name) */
	argc--;
	argv++;

	if (argc == 0) {
		show_options();
		exit(1);
		}

	while (argc--)
	{
	  if (argv[0][0] == '-')
	    {
top:
	      k = (*(++argv[0])); 
	      k = argv[0][0];
	      l = 1;
	      switch(k)
		{
		  case 'V' :
			{
			  if (!argc--)
			    { show_options(); break; }
			  argv++;
			  volume_label = *argv;
			  printf("Setting volume label to %s\n",volume_label);
			  l--;
			  break;
			}
		  case 'c' :
			{
			  if (!argc--)
			    { show_options(); break; }
			  argv++;
			  bluff_blks = atoi(*argv);
			  if (bluff_blks >= 80)
			    {
			      printf("That is too much space to waste : %i%%\n",bluff_blks);
			      exit(0);
			    }
			  l--;
			  break;
			}
		  case 'k' :
			{
			  wants_key = 1;
			  break;
		 	}
		  case 'i' :
			{
			  show_stats = 1;
			  break;
			}
                  case 'f' :
                        {
                          disk_check = 0;
                          break;
                        } 
		  case 's' :
			{
			  sim_bad_blocks = 1;
			  break;
			}
		  case 'n' :
			{
			  disk_randomize = 0;
			  break;
			}
		  default :
			{
			  printf("Invalid Option : %c\n",k);
			  show_options();
			  exit(1);
			}
		}
		if ((l) && (argv[0][1] != '\0'))
		  { goto top; }
	    }
	  else
	    {
	      device_name = *argv;
	      got_device++;
	     }
	  argv++;
	}

  if (!check_options())
    {
      show_options();
      exit(1);
    }
}


/* main procedure */
int main (int argc, char *argv[])
{
	PRS(argc, argv);  /* Process command line options */

	init_3way();      /* Initialise the 3-Way encryption cipher */

	/* Open read/write, with disk sync */
	device = open(device_name,O_RDWR|O_SYNC);
	if(device < 0) {
	  perror(device_name);
	  exit(errno);
	}


	/* Check to see if device is valid etc */
	/* And if the user really want to procede */
	if ((check_device() || ask_user()))
	{
	  close(device);
	  exit(0);
	}

	ask_for_key('n');    /* Ask the user for the Key */
        init_super_block();  /* Setup basic superblock info */

	bad_start = malloc(8);
	bad_start->next = 0;

	check_blocks();      /* Scan disk for bad blocks */
	if (disk_randomize)
	  randomize_platter(); /* Randomize disk surface */

	close(device);
	device = open(device_name,O_RDWR); /* Open read/write */

	create_groups();     /* Create groups on disk */
	create_root_dir();   /* Create the root inode */
	create_bad_inode();  /* Create the bad block inode */
	create_noise();      /* Marks a certain percentage of the disk used */

        read_block(SFS_BOOT_SECT_SIZE/SFS_BLK, (char *)&sb);
	if (sfs_verify(sb))
	  die("superblock encryption test failed");
	if (show_stats)
	  get_fs_info();

	sync_super_blocks();
	clean_up();
	close(device);
	printf("\nmksfs completed, and passed test\n");
	return 0;
}
