/* sfslib.c - Common functions shared by the steganographic
 *            filesystem
 *
 * Copyright (C) 1998 Carl van Schaik (carl@leg.uct.ac.za)
 *                    Paul Smeddle (psmeddle@cs.uct.ac.za)
 * SFS - [S]teganographic [F]ile [S]ystem
 *
 *
 * %Begin-Header%
 * This file may be redistributed under the terms of the GNU Public
 * License.
 * %End-Header%
 *
 * Note, the position of the "crypt_master" block is located by taking
 * (e_three_way_key[0] % (blocks on disk))
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <termios.h>
#include <time.h>
#include <errno.h>
#include <mntent.h>
#include <malloc.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <linux/fs.h>
#include <linux/sfs_fs.h>

#include "sfslib.h"

char    *device_name = NULL;

int got_device = 0;
int wants_key = 0;
int show_stats = 0;

char *volume_label = "SFS";
unsigned int offset = SFS_BOOT_SECT_SIZE;

/* Clean up inportant information */
void clean_up()
{
	int i = 0;
	char * sbc = (char*)&sb;

	for(i=0;i<4;i++)
	  {
	    idea_key[i] = genrand();
	    e_idea_key[i] = genrand();
	  }
	for(i=0;i<3;i++)
	  {
	    three_way_key[i] = genrand();
	    e_three_way_key[i] = genrand();
	  }

	for (i=0;i<SFS_BLK;i++)
	  {
	    sbc[i] = genrand();
	    bitmap[i] = genrand();
	  }
}

/* Exits with an error message */
void die(char* text)
{
	clean_up();
	close(device);
	printf("Error: %s\n",text);
	exit(0);
}


/*
 * Check to make certain that our new filesystem won't be created on
 * an already mounted partition.  Code adapted from mke2fs, Copyright
 * (C) 1994 Theodore Ts'o.  Also licensed under GPL.
 */
void check_mount(void)
{
        FILE * f;
        struct mntent * mnt;

        if ((f = setmntent (MOUNTED, "r")) == NULL)
                return;
        while ((mnt = getmntent (f)) != NULL)
                if (strcmp (device_name, mnt->mnt_fsname) == 0)
                        break;
        endmntent (f);
        if (!mnt)
                return;

	printf("%s is mounted, please re-consider your request\n",device_name);
	clean_up();
	close(device);
	exit(0);
}

/* Writes a single disk block at position blk  */
/* Uses 3-way encryption - for data structures */
void write_block(unsigned int blk, char * buffer)
{
	char enc_buf[SFS_BLK];
	memcpy(enc_buf,buffer,SFS_BLK);
	encrypt_3_way(three_way_key,enc_buf,SFS_BLK);

        if (blk*SFS_BLK != lseek(device, blk*SFS_BLK, SEEK_SET))
                die("seek failed in write_block");


        if (SFS_BLK != write(device, enc_buf, SFS_BLK)) {
	    printf("block write error at %i\n",blk);
            die("write failed in write_block");
	  }
}

/* Writes a single disk block at position blk  */
/* Using the user encryption password          */
/* Uses 3-way encryption - for data structures */
void write_e_block(unsigned int blk, char * buffer)
{
        char enc_buf[SFS_BLK];
        memcpy(enc_buf,buffer,SFS_BLK);
        encrypt_3_way(e_three_way_key,enc_buf,SFS_BLK);

        if (blk*SFS_BLK != lseek(device, blk*SFS_BLK, SEEK_SET))
                die("seek failed in write_block");


        if (SFS_BLK != write(device, enc_buf, SFS_BLK)) {
            printf("block write error at %i\n",blk);
            die("write failed in write_block");
          }
}

/* Writes a single data disk block at position blk */
/* Uses IDEA encryption for normal data            */
void write_data_block(unsigned int blk, char * buffer)
{
	char enc_buf[SFS_BLK];
	memcpy(enc_buf,buffer,SFS_BLK);
	encrypt_idea(idea_key,enc_buf,SFS_BLK);

	if (blk*SFS_BLK != lseek(device, blk*SFS_BLK, SEEK_SET))
	  die("seek failed in write_block");


	if (SFS_BLK != write(device, enc_buf, SFS_BLK)) {
	    printf("block write error at %i\n",blk);
	    die("write failed in write_block");
	  }
}

/* Writes a single data disk block at position blk */
/* Using the user encryption password          */
/* Uses IDEA encryption for normal data            */
void write_e_data_block(unsigned int blk, char * buffer)
{
        char enc_buf[SFS_BLK];
        memcpy(enc_buf,buffer,SFS_BLK);
        encrypt_idea(e_idea_key,enc_buf,SFS_BLK);

        if (blk*SFS_BLK != lseek(device, blk*SFS_BLK, SEEK_SET))
          die("seek failed in write_block");


        if (SFS_BLK != write(device, enc_buf, SFS_BLK)) {
            printf("block write error at %i\n",blk);
            die("write failed in write_block");
          }
}


/* Reads a single disk block at posistion blk into buffer */
/* Uses 3-way encryption - for data structures */
void read_block(unsigned int blk, char * buffer)
{
	char enc_buf[SFS_BLK];

        if (blk*SFS_BLK != lseek(device, blk*SFS_BLK, SEEK_SET))
                die("seek failed in read_block");

        if (SFS_BLK != read(device, enc_buf, SFS_BLK)) {
            printf("block read error at %i\n",blk);
            die("read failed in read_block");
          }

	decrypt_3_way(three_way_key,enc_buf,SFS_BLK);
	memcpy(buffer,enc_buf,SFS_BLK);
}

/* Reads a single disk block at posistion blk into buffer */
/* Using the user encryption password          */
/* Uses 3-way encryption - for data structures */
void read_e_block(unsigned int blk, char * buffer)
{
        char enc_buf[SFS_BLK];

        if (blk*SFS_BLK != lseek(device, blk*SFS_BLK, SEEK_SET))
                die("seek failed in read_block");

        if (SFS_BLK != read(device, enc_buf, SFS_BLK)) {
            printf("block read error at %i\n",blk);
            die("read failed in read_block");
          }

        decrypt_3_way(e_three_way_key,enc_buf,SFS_BLK);
        memcpy(buffer,enc_buf,SFS_BLK);
}

/* Reads a single disk block at posistion blk into buffer */
/* Uses IDEA encryption for normal data            */
void read_data_block(unsigned int blk, char * buffer)
{
	char enc_buf[SFS_BLK];

	if (blk*SFS_BLK != lseek(device, blk*SFS_BLK, SEEK_SET))
	  die("seek failed in read_block");

	if (SFS_BLK != read(device, enc_buf, SFS_BLK)) {
	    printf("block read error at %i\n",blk);
	    die("read failed in read_block");
	  }

	decrypt_idea(idea_key,enc_buf,SFS_BLK);
	memcpy(buffer,enc_buf,SFS_BLK);
}

/* Reads a single disk block at posistion blk into buffer */
/* Using the user encryption password          */
/* Uses IDEA encryption for normal data            */
void read_e_data_block(unsigned int blk, char * buffer)
{
        char enc_buf[SFS_BLK];

        if (blk*SFS_BLK != lseek(device, blk*SFS_BLK, SEEK_SET))
          die("seek failed in read_block");

        if (SFS_BLK != read(device, enc_buf, SFS_BLK)) {
            printf("block read error at %i\n",blk);
            die("read failed in read_block");
          }

        decrypt_idea(e_idea_key,enc_buf,SFS_BLK);
        memcpy(buffer,enc_buf,SFS_BLK);
}


/* Sets a bit in the bitmap on */
void bitset(char bitm[],int bit)
{
	if ((bit >= 8192) || (bit < 0))
	  return;

	bitm[bit/8] |= (1 << (bit % 8));
}

/* Sets a bit in the bitmap off */
void bitunset(char bitm[],int bit)
{
	if ((bit >= 8192) || (bit < 0))
	  return;

	bitset(bitm, bit);
	bitm[bit/8] = (bitm[bit/8] - (1 << (bit % 8)));
}


/* Checks whether a bit in bitmap is set or not */
int is_bit_set(char bitm[],int bit)
{
	int tst;

        if ((bit >= 8192) || (bit < 0))
          return (1);

	tst = bitm[bit/8];
	tst |= (1 << (bit % 8));

	return (tst == bitm[bit/8]);
}

/* Check for a valid super_block */
int sfs_verify(struct sfs_super_block SB)
{
	if ((SB.s_crypt_id_1 != SUPER_CRYPT_ID_1) ||
	    (SB.s_crypt_id_2 != SUPER_CRYPT_ID_2) ||
	    (SB.s_magic_number != SFS_SUPER_MAGIC))
	  return(1);

	return(0);
}


/* Check for a valid inode */
int sfs_verify_i(struct sfs_inode inode)
{
	if ((inode.i_crypt_id_1 != INODE_CRYPT_ID_1) ||
	    (inode.i_crypt_id_2 != INODE_CRYPT_ID_2))
	  return(1);

	return(0);
}


/* Update all the superblocks with the new one */
void update_super_block(struct sfs_super_block SB)
{
        unsigned int pos, group = 0;

        pos = offset/SFS_BLK;

        while (group < SB.s_group_no)
          {
            SB.s_this_group_no = group; /* Set curent group */
            printf("Updating %i out of %i\r",group+1,SB.s_group_no);
            fflush(stdout);

            write_block(pos, (char *)&SB);

            group++;
            pos = (offset/SFS_BLK) + (group*SFS_GROUP);
          }
}


/* Checks if a specified block is used or not */
int is_block_used(unsigned int blk)
{
	struct sfs_super_block SB;
	char   bitm[SFS_BLK];
	int offs = offset/SFS_BLK;
	unsigned int i,o;

        if (blk < offs)
           return(1);

	i = (blk-offs) / SFS_GROUP;
	o = (blk-offs) % SFS_GROUP;

	read_block(offs, (char *)&SB);
	if (sfs_verify(SB))
	  die("superblock encryption test failed");

        if (i > SB.s_group_no)
          return (1);
	if (blk > SB.s_block_total)
	  return (1);

	read_block(offs+i*SFS_GROUP+1, (char *)&bitm);

	return (is_bit_set(bitm, o));
}

/* Sets a block marked used on the disk */
void set_block_used(unsigned int blk)
{
	struct sfs_super_block SB;
	char   bitm[SFS_BLK];
	int    offs = offset/SFS_BLK;
	unsigned int i,o;

	i = (blk-offs) / SFS_GROUP;
	o = (blk-offs) % SFS_GROUP;

	read_block(offs, (char *)&SB);
        if (sfs_verify(SB))
          die("superblock encryption test failed");
	if (i > SB.s_group_no)
	  die("tried to read past end of fs");

	read_block(offs+i*SFS_GROUP+1, (char *)&bitm);

	if (is_bit_set(bitm, o))
	  die("tried to set bit that already set, corruption warning!");

	bitset(bitm, o);

	write_block(offs+i*SFS_GROUP+1, (char *)&bitm);

	SB.s_free_blocks--;

	write_block(offs, (char *)&SB);
}

/* Sync the superblocks */
void sync_super_blocks()
{
	struct sfs_super_block SB;
	int  offs = offset/SFS_BLK;

	read_block(offs, (char *)&SB);
	if (sfs_verify(SB))
	  die("superblock encryption test failed");

	update_super_block(SB);
}


/* Reads in a directory into a single buffer */
char* read_dir(struct sfs_inode ino)
{
   char *data;
   unsigned int place, p2, i, j;
   unsigned int block[256];
   
   if ((ino.i_type != SFS_ROOT_INO)
      && (ino.i_type != SFS_DIRECTORY_INO)
      && (ino.i_type != SFS_CRYPT_ROOT))
      {
         return NULL;
      }
      
   data = (char*)malloc(ino.i_blocks*SFS_BLK);
   if (data == NULL)
      return NULL;
      
   place = ino.i_blocks;
   if (place > 96)
      place = 96;
      
   for (i=0;i<place;i++)
      read_data_block(ino.i_direct[i],data+i*SFS_BLK);

   place = ino.i_blocks;
   if (place > 96)
   {
      place -= 96;
      if (place > 20480)
         place = 20480;
      for (i=0;i<(place / 256);i++)
      {
         read_data_block(ino.i_single[i],(char*)&block);
         p2 = place - i*256;
         if (p2 > 256)
            p2 = 256;
            
         for (j=0;j<p2;j++)
            read_data_block(block[j],data+(96+i*256+j)*SFS_BLK);
      }
   }

   return data;
}

/* Updates a directory from a buffer */
/* Assumes that the blocks have already been */
/* allocated on the device and that the inode */
/* structure is up to date with the data */
/* ie i_direct[i] points to where data should */
/* be written */
void update_dir(char* data, struct sfs_inode ino)
{
   unsigned int place, p2, i, j;
   unsigned int block[256];
   
   if ((ino.i_type != SFS_ROOT_INO)
      && (ino.i_type != SFS_DIRECTORY_INO)
      && (ino.i_type != SFS_CRYPT_ROOT))
        return;
      
   if (data == NULL)
      return;
      
   place = ino.i_blocks;
   if (place > 96)
      place = 96;
      
   for (i=0;i<place;i++)
      write_data_block(ino.i_direct[i],data+i*SFS_BLK);

   place = ino.i_blocks;
   if (place > 96)
   {
      place -= 96;
      if (place > 20480)
         place = 20480;
      for (i=0;i<(place / 256);i++)
      {
         read_data_block(ino.i_single[i],(char*)&block);
         p2 = place - i*256;
         if (p2 > 256)
            p2 = 256;
            
         for (j=0;j<p2;j++)
            write_data_block(block[j],data+(96+i*256+j)*SFS_BLK);
      }
   }
}

long valid_offset (int fd, int offset)
{
        char ch;

        if (lseek (fd, offset, 0) < 0)
                return 0;
        if (read (fd, &ch, 1) < 1)
                return 0;
        return 1;
}

int count_blocks (int fd)
{
        int high, low;

        low = 0;
        for (high = 1; valid_offset (fd, high); high *= 2)
                low = high;
        while (low < high - 1)
        {
                const int mid = (low + high) / 2;

                if (valid_offset (fd, mid))
                        low = mid;
                else
                        high = mid;
        }
        valid_offset (fd, 0);
        return (low + 1);
}

/* Return the size of the disk */
unsigned int get_size()
{
	unsigned int  size;

	if (ioctl(device, BLKGETSIZE, &size) >= 0) {
	  return (size  / 2);
	}
	else
	  return count_blocks(device)/SFS_BLK;
}


/* Check to see if a then given file is valid etc */
int check_device()
{
	struct stat statbuf;
	unsigned int size;

        if (fstat(device,&statbuf) < 0) {
          printf ("Unable to get stats for %s\n",device_name);
          return(1);
        }

	if (!S_ISBLK(statbuf.st_mode) && !S_ISREG(statbuf.st_mode))
	  {
	    printf("%s: only block and file devices are supported\n",device_name);
	    return(1);
	  }
	if (statbuf.st_rdev == 0x0300 || statbuf.st_rdev == 0x0340)
	  {
	    printf("Cannot make a filesystem on %s\n",device_name);
	    return(1);
	  }

	size = get_size();
	if (size == 0)
	{
	  printf("%s: device has zero lenght\n",device_name);
	  return (1);
	}
	else if (size < 64)
	{
	  printf("%s: device is too small\n",device_name);
	  return (1);
	}
	else
          printf("Device %s is %iMB\n",device_name, size/1024 );

	check_mount();

	return 0;
}

/* Get the users key and create data keys */
void ask_for_key(char which)
{
	char  check[128];
	char  *temp;
	word32 i = 0;

	printf("[0;1;37;36m");
	if (which == 'n')
	  printf("Please provide your file system password\nNote - [ALL] key presses are recorded");
	else
	  printf("Please provice your new encryption password");
	printf("[0;1;37;32m\n");
	temp = getpass("Enter Password : ");
	strcpy(text_key,temp);
	temp = getpass("Please Verify : ");
	strcpy(check,temp);

	printf("[0;1;37;00m");

	while ((text_key[i] != '\0') && (i < 128))
	  {
	    if (text_key[i] != check[i])
	      {
		printf("Keys do not match\n");
		clean_up();
		exit(0);
	      }
	    i++;
	  }

	if (check[i] != '\0')
	  {
	    printf("Keys do not match\n");
	    clean_up();
	    exit(0);
	  }

	if (i < 5)
	  {
	    printf("Key is two short\n");
	    printf("Key Options:\n");
	    printf("  More than 20 characters is recomened, 5 is minimum.\n");
	    printf("  Spaces, numbers and control characters allowed.\n");
	    printf("  eg : \"Nobody i`ll get {This} ~one~ $#!@\"\n");
	    printf("  Also, try keys like <F1> <Page Up> etc.\n");
	    clean_up();
	    exit(0);
	  }


	if (which == 'n')
	  {
	    make_idea_key(text_key, idea_key);
	    make_3way_key(text_key, three_way_key);
	  }
	  else
	  {
	    make_idea_key(text_key, e_idea_key);
	    make_3way_key(text_key, e_three_way_key);
	  }

	if (wants_key)
	  if (which == 'n')
	    {
	      printf("This is the access key\n");
	      printf("IDEA KEY = 0x%04x%04x%04x%04x\n3WAY KEY = 0x%08lx%08lx%08lx\n",
	      idea_key[0],idea_key[1],idea_key[2],idea_key[3],
	      three_way_key[0],three_way_key[1],three_way_key[2]);
	    }
	    else
	    {
	      printf("This is your private key\n");
              printf("IDEA KEY = 0x%04x%04x%04x%04x\n3WAY KEY = 0x%08lx%08lx%08lx\n",
              e_idea_key[0],e_idea_key[1],e_idea_key[2],e_idea_key[3],
              e_three_way_key[0],e_three_way_key[1],e_three_way_key[2]);
	    }


	if (idea_key[0] == e_idea_key[0])
        if (idea_key[1] == e_idea_key[1])
        if (idea_key[2] == e_idea_key[2])
        if (idea_key[3] == e_idea_key[3])
	  {
	    printf("Error: system and user key may not be the same\n");
	    clean_up();
	    exit(0);
	  }
}


/* Displays final fs information */
void get_fs_info()
{
	struct sfs_inode inode;
        read_block(offset/SFS_BLK, (char *)&sb);
        if (sfs_verify(sb))
          die("superblock encryption test failed");

	printf("---------latest fs details---------\n");
	printf("Volume Name  : %s\n",sb.s_vol_name);
	printf("Disk Size    : %i MB\n",sb.s_block_total/SFS_BLK);
	printf("Disk Free    : %i MB\n",sb.s_free_blocks/SFS_BLK);
	printf("Disk Blocks  : %i\n",sb.s_block_total);
	printf("Free Blocks  : %i\n",sb.s_free_blocks);
	printf("# of Groups  : %i\n",sb.s_group_no);

	read_data_block(sb.s_bad_inode,(char*)&inode);	
	if (sfs_verify_i(inode))
	  die("bad block inode encryption test failed");
	printf("Bad blocks   : %i\n",inode.i_reserved[0]);
}
