/* 
 * vs3 file system
 * (C) 1998 Paul Smeddle (psmeddle@cs.uct.ac.za)
 *          Carl v. Schaik (carl@groomlake.ml.org)
 */

/*
 * file.c - file reading and writing routines.
 * (C) 1998 P. Smeddle & C. van Schaik
 */

#define __KERNEL__
#include <linux/fs.h>
#include <linux/string.h>
#include <linux/vs3_fs.h>
#include <linux/sched.h>

/* Finds the device block corresponding to file-relative block "block" 
   only works for single pointers so far                              */

__u32 vs3_find_block( struct inode *i, __u32 block )
{
	__u32 blockno=0;
	struct vs3_inode *raw;
	struct buffer_head *bh, *bh2;
        __u32 *buf;
        
	bh = vs3_bread( i, i->i_ino, i->u.vs3_i.enc_mode );
	if( !bh ) {
		printk( "vs3fs_find_block: Unable to read inode block.\n" );
		return -1;
	}
	raw = (struct vs3_inode *)bh->b_data;
	if( (raw->i_crypt_id_1 != INODE_CRYPT_ID_1) || (raw->i_crypt_id_2 != INODE_CRYPT_ID_2) ) 
	{
		printk( "vs3fs_find_block: Unable to read inode data\n" );
		vs3_bforget( bh );
		return -1;
        }
        if (block >= raw->i_blocks)
        {
           vs3_bforget( bh );
           if (i->i_sb->u.vs3_sb.s_free_blocks < 3)  /* 3 for i_double (worst case) */
              return -1;
           return 0;
        }
	if( block < 96 ) {
		blockno = raw->i_direct[block];
	}
        else
        if (block-96 < 20480)
        {
           blockno = block - 96;
           bh2 = vs3_bread(i,raw->i_single[blockno/256],i->u.vs3_i.enc_mode);
           buf = (__u32 *)bh2->b_data;
           blockno = buf[blockno % 256];
           vs3_bforget(bh2);
        }
	vs3_bforget( bh );
	return blockno;
}


/* Performs read-ahead caching */
void do_read_ahead(struct inode *i, __u32 start, __u32 end )
{
        __u32 blockno=0, count;
        struct vs3_inode *raw;
        struct buffer_head *bh, *bh2;
        __u32 *buf;

        bh = vs3_bread( i, i->i_ino, i->u.vs3_i.enc_mode );
        if( !bh ) {
                printk( "vs3fs_read_ahead: Unable to read inode block.\n" );
                return;
        }
        raw = (struct vs3_inode *)bh->b_data;
        if( (raw->i_crypt_id_1 != INODE_CRYPT_ID_1) || (raw->i_crypt_id_2 != INODE_CRYPT_ID_2) )
        {
                printk( "vs3fs_read_ahead: Unable to read inode data\n" );
                vs3_bforget( bh );
                return;
        }

	for (count = start; count < end; count ++)
	{
           if (count >= raw->i_blocks)
           {
	      printk("vs3_read_ahead: tried to read past end of file\n");
              vs3_bforget( bh );
              return;
           }
           if( count < 96 ) {
                blockno = raw->i_direct[count];
           }
           else
           if (count-96 < 20480)
           {
              blockno = count - 96;
              bh2 = vs3_bread(i,raw->i_single[blockno/256],i->u.vs3_i.enc_mode);
              buf = (__u32 *)bh2->b_data;
              blockno = buf[blockno % 256];
              vs3_bforget(bh2);
           }
	   bh2 = bread( i->i_dev, blockno, VS3_BLK );
	   brelse(bh2);
	}
        vs3_bforget( bh );
}

/* Updates the inode block-mapping pointers to add in an allocated block*/  
void vs3_update_pointers( struct inode *i, __u32 block )
{
	__u32 blockno,blk;
	struct buffer_head *bh, *bh2;
	struct vs3_inode *raw;
        __u32 *buf;

	bh = vs3_bread( i, i->i_ino, i->u.vs3_i.enc_mode );
	if( !bh ) {
		printk( "vs3fs_up_ptrs: unable to read raw inode data\n" );
                return;
	}	 
	raw = (struct vs3_inode *)bh->b_data;
        if( (raw->i_crypt_id_1 != INODE_CRYPT_ID_1) || (raw->i_crypt_id_2 != INODE_CRYPT_ID_2) )
        {
                printk( "vs3fs_up_ptrs: Unable to read inode data\n" );
		vs3_bforget(bh);
                return;
        }
	if( i->i_blocks < 96 )  {
		raw->i_direct[i->i_blocks] = block;
	}
        else
        if (i->i_blocks-96 < 20480)
        {
           blockno = i->i_blocks-96;
           if ((blockno % 256) == 0)
           {
              blk = get_free_block(i->i_sb);
              set_block_used(i->i_sb,blk);
              raw->i_single[blockno/256]=blk;
           }
           bh2 = vs3_bread(i,raw->i_single[blockno/256],i->u.vs3_i.enc_mode);
           buf = (__u32 *)bh2->b_data;
           buf[blockno % 256] = block;
           vs3_bwrite(bh2, i, i->u.vs3_i.enc_mode);
        }
        i->i_blocks++;
        raw->i_blocks++;
	vs3_bwrite( bh, i, i->u.vs3_i.enc_mode );
} 

/* writes a device-block aligned absolute block to the device given a
   file-relative block, an offset into the block, and a length to write */

int write_block_vs3fs( struct inode *i, const char *buf, int len, __u32 block, int offs )
{
	__u32 blockno;
	struct buffer_head *bh;
	blockno = vs3_find_block( i, block );
	if( blockno == 0 ) {
		blockno = get_free_block( i->i_sb );
                set_block_used(i->i_sb,blockno);
		vs3_update_pointers( i, blockno );
	} else
	if( (int)blockno == -1 )
	{
		printk( "vs3fs_write_block: no free space left on device." );
		return -1;
	}
	bh = vs3_bread( i, blockno, i->u.vs3_i.enc_mode );
	if( !bh ) {
		printk( "vs3fs_write_block: unable to read file block.\n" );
		return -1; 
	}
	memcpy_fromfs( bh->b_data + offs, buf, len );
	vs3_bwrite( bh, i, i->u.vs3_i.enc_mode );
	return len;
}

/* reads a device-aligned absolute block from the device given a
   file-relative block, an offset into the block and a length to read */

int read_block_vs3fs( struct inode *i, char *buf, int len, __u32 block, int offs )
{
	__u32 blockno;
	struct buffer_head *bh;
	blockno = vs3_find_block( i, block );	
	if( (int)blockno == -1 ) return -1;
	bh = vs3_bread( i, blockno, i->u.vs3_i.enc_mode );
	if( !bh ) {
		printk( "vs3fs_read_block: Unable to read data block from device\n" );
		return -1;
	} 
	memcpy_tofs( buf, bh->b_data + offs, len );
	vs3_bforget( bh );
	return len;
}

/* Read a specified number of characters into a buffer from a file at a
specific offset. This is the file-read function pointed to in the VFS
in inode_operations->file_operations.read */

int vs3_file_read( struct inode *i, struct file *f, char *buf, int c )
{
	int read = 0, len, result;
	__u32 blockno, left, bufnum;
	int foff;
	off_t pos;
	pos = f->f_pos;
	if ( !i ) {
                printk( "vs3fs_file_read: inode = NULL\n" );
                return -1;
        }
        if( !S_ISREG( i->i_mode ) ) {
        	printk( "vs3fs_file_read: not a regular file.\n" );
        	return -1;
        }

        foff = pos % VS3_BLK;

	left = i->i_size - f->f_pos;
	if (left > c)
           left = c;


        blockno = pos / VS3_BLK;
	bufnum = (blockno + 16) - ((blockno+16) % 16);
        bufnum = (bufnum > (i->i_size/VS3_BLK)) ? (bufnum) : (i->i_size/VS3_BLK);
//	if (!(blockno % 16))
//	   do_read_ahead(i, blockno, bufnum);

        while( read < left ) {
		if( foff )
                {
                   if( left - read >= VS3_BLK - foff )
                      len = VS3_BLK - foff;
                   else
                      len = left - read;
                }
        	else
                {
                   if( left - read >= VS3_BLK )
                      len = VS3_BLK;
                   else
                      len = left - read;
                }
		result = read_block_vs3fs( i, buf + read, len, blockno, foff );
		if( result != -1 ) read += result;
		else { 
			read = -1;
			break;
		}
		blockno++;		
		foff = 0;
        }
        if( read != -1 ) pos += read;
	if (!IS_RDONLY (i)) {
           i->i_atime = CURRENT_TIME;
           i->i_dirt = 1;
	}
        f->f_pos = pos;
        return read;	
}

/* write a specified number of characters from a buffer to a file at a
   certain offset. This is the function pointed to in the VFS by
   inode_operations->file_operations.write */

int vs3_file_write( struct inode *i, struct file *f, const char *buf, int c )
{
	int written = 0, len, result;
	off_t pos, foff;
	__u32 blockno;

	if ( !i ) {
                printk( "vs3fs_file_write: inode = NULL\n" );
                return -1;
        }
        if( !S_ISREG( i->i_mode ) ) {
        	printk( "vs3fs_file_write: not a regular file\n" );
        	return -1;
        }
        if( f->f_flags & O_APPEND ) 
		pos = i->i_size;
	else 
		pos = f->f_pos;
        foff = pos % VS3_BLK;
        blockno = pos / VS3_BLK;
        while( written < c ) {
        	if( foff )
                {
                   if( c - written > VS3_BLK - foff )
                      len = VS3_BLK - foff;
                   else
                      len = c - written;
                }
        	else
                {
                   if( c - written > VS3_BLK )
                      len = VS3_BLK;
                   else
                      len = c - written;
                }
		result = write_block_vs3fs( i, buf + written, len, blockno, foff );
		if( result != -1 ) written += result;
                else
                { 
                   written = -1;
                   break;
		}
		blockno++;
		foff = 0;		
        }
        if( written != -1 ) pos += written;
        if (pos > i->i_size)
                        i->i_size = pos;        
        i->i_mtime = i->i_ctime = CURRENT_TIME;
        f->f_pos = pos;
        i->i_dirt = 1;

        return written;
}

