/*
 *  linux/include/linux/inode.c
 *
 * The Steganographic Filesystem (vs3fs)
 *
 *   Copyright (C) 1998
 *   Paul Smeddle (psmeddle@cs.uct.ac.za)
 *   Carl van Schaik (carl@leg.uct.ac.za)
 *   University of Cape Town, South Africa
 *
 * Portions (C) from ext2 / msdos / minix filesystems
 */

#define __KERNEL__
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/fcntl.h>
#include <linux/locks.h>
#include <linux/vs3_fs.h>

#include <linux/errno.h>


struct inode * vs3_new_inode(const struct inode * dir)
{
	struct super_block * sb;
	struct buffer_head * bh;
	struct inode * inode;
	struct vs3_inode * vino;
	__u32 i;

	if (!dir || !(inode = get_empty_inode()))
		return NULL;
	sb = dir->i_sb;
	inode->i_sb = sb;
	inode->i_flags = inode->i_sb->s_flags;

        if (inode->i_sb->u.vs3_sb.s_free_blocks == 0)
           {
              printk("vs3: disk full\n");
              return NULL;
           }
        
	inode->i_count = 1;
	inode->i_nlink = 1;
	inode->i_dev = sb->s_dev;
	inode->i_uid = current->fsuid;
	inode->i_gid = (dir->i_mode & S_ISGID) ? dir->i_gid : current->fsgid;
	inode->i_dirt = 1;
	inode->i_mtime = inode->i_atime = inode->i_ctime = CURRENT_TIME;
	inode->i_op = NULL;
        inode->i_size = 0;
	inode->i_blocks = 0;
        inode->i_blksize = VS3_BLK;
        inode->i_ino = get_free_block(sb);

	inode->u.vs3_i.enc_mode = dir->u.vs3_i.enc_mode;

	insert_inode_hash(inode);

	bh = vs3_bread(inode ,inode->i_ino, inode->u.vs3_i.enc_mode);

 	vino = (struct vs3_inode *)bh->b_data;
/* ---- create phisical record ---- */
        vino->i_crypt_id_1  = INODE_CRYPT_ID_1;
        vino->i_crypt_id_2  = INODE_CRYPT_ID_2;
        vino->i_blocks      = 0;
        vino->i_uid = inode->i_uid;
 	vino->i_gid = inode->i_gid;
        vino->i_mtime = inode->i_mtime;
	vino->i_atime = inode->i_atime;
	vino->i_ctime = inode->i_ctime;
        vino->i_file_size   = 0;
        vino->i_link_count  = 1;
	vino->i_dir_entries = 0;
	vino->i_type        = VS3_NORMAL_INO;
        vino->i_perms       = 0755;
        for (i=0;i<3;i++)
          vino->i_reserved[i] = genrand();

        set_block_used(sb, inode->i_ino);

	vs3_bwrite(bh, inode, inode->u.vs3_i.enc_mode);
	return inode;
}


void vs3_free_inode(struct inode * inode)
{
	if (!inode)
		return;
	if (!inode->i_dev) {
		printk("vs3_free_inode: inode has no device\n");
		return;
	}
	if (inode->i_count != 1) {
		printk("vs3_free_inode: inode has count=%ld\n",inode->i_count);
		return;
	}
	if (inode->i_nlink) {
		printk("vs3_free_inode: inode has nlink=%d\n",inode->i_nlink);
		return;
	}
	if (!inode->i_sb) {
		printk("vs3_free_inode: inode on nonexistent device\n");
		return;
	}
	if (inode->i_ino < 1 || inode->i_ino > inode->i_sb->u.vs3_sb.s_block_total) {
		printk("vs3_free_inode: inode 0 or nonexistent inode\n");
		return;
	}
	clear_inode(inode);
}


int vs3_sync_inode(struct inode * inode)
{
        int err = 0;
	struct buffer_head * bh;

	vs3_write_inode(inode);
        bh = get_hash_table(inode->i_dev,inode->i_ino,VS3_BLK);
        if (bh && buffer_dirty(bh))
        {
                ll_rw_block(WRITE, 1, &bh);
                wait_on_buffer(bh);
                if (buffer_req(bh) && !buffer_uptodate(bh))
                {
                        printk ("IO error syncing minix inode ["
                                "%s:%08lx]\n",
                                kdevname(inode->i_dev), inode->i_ino);
                        err = -1;
                }
        }
        else if (!bh)
                err = -1;
        brelse (bh);

        return err;
}

/* How do we do this??? Can't cause all data in encrypted??
int vs3_bmap(struct inode * inode,int block)
{
        int i;

        if (block<1) {
                printk("vs3_bmap: block<1");
                return 0;
        }
        if (block >= (1024*1024*1024*4) {
                printk("vs3_bmap: block>big");
                return 0;
        }
        if ((block-1)%VS3_BLK < 2)
                return V1_inode_bmap(inode,block);
        block -= 7;
        if (block < 512) {
                i = V1_inode_bmap(inode,7);
                if (!i)
                        return 0;
                return V1_block_bmap(bread(inode->i_dev,i,BLOCK_SIZE),block);
        }
        block -= 512;
        i = V1_inode_bmap(inode,8);
        if (!i)
                return 0;
        i = V1_block_bmap(bread(inode->i_dev,i,BLOCK_SIZE),block>>9);
        if (!i)
                return 0;
        return V1_block_bmap(bread(inode->i_dev,i,BLOCK_SIZE),block & 511);
}
*/
