/*
 *  linux/include/linux/vs3_fs.h
 *
 * The Steganographic Filesystem (vs3fs)
 *
 *   Copyright (C) 1998
 *   Paul Smeddle (psmeddle@cs.uct.ac.za)
 *   Carl van Schaik (carl@leg.uct.ac.za)
 *   University of Cape Town, South Africa
 *
 */

/*
 *  Note that when talking about the "encrypted" somthing
 *  we are talking about whatever the user has hidden
 *  with a password other than with the default password.
 *  All other things are encrypted with only the default
 *  password.
 */

/* vs3_fs.h - Carl van Schaik */

#ifndef _LINUX_VS3_FS_H
#define _LINUX_VS3_FS_H

#include <linux/types.h>

/*
 * Default values for "Inode Idetification" structure in inode structure
 */

#define VS3_BAD_INO       1    /* This is the bad inode inode  */
#define VS3_ROOT_INO      2    /* The default root inode       */
#define VS3_CRYPT_ROOT    3    /* The encrypted root inode that gets mapped
                                * onto the default root once he has the
                                * correct password
                                */
#define VS3_DIRECTORY_INO 10   /* Idetifies this as a directory */
#define VS3_FIFO_INO      11   /* First In First Out */
#define VS3_SOCK_INO      12   /* A socket */

#define VS3_CHRDEV_INO    20   /* Character device driver special file */
#define VS3_BLKDEV_INO    21   /* Block device driver file */

#define VS3_NORMAL_INO    80   /* Normal file */

/*
 * Filesystem specific numbers
 */

#define VS3_SUPER_MAGIC  0x2BAD
#define VS3_BOOT_SECT_SIZE 1024    /* The size of the boot sector */
#define VS3_BLK 1024               /* The standard block size */
#define VS3_GROUP (8*VS3_BLK)      /* The standard group size */
#define VS3_PRE_ALLOC 3            /* Default ammount of space to pre-allocate
				    * as used data   */
#define VS3_WCPRE_ALLOC 3          /* Default ammount of space to pre-allocate
				    * to winnowing and chaffing per password */

#define VS3_NAME_LEN   255         /* Maximum file name length */
#define VS3_BIDEA	0	   /* encrypt/decrypt buffer with idea */
#define VS3_B3WAY	1	   /* encrypt/decrypt buffer with 3-way */
#define VS3_BU3WAY      2          /* user encrypy/decrypt with 3way */
#define VS3_BUIDEA      3          /* user encrypy/decrypt with idea */

/*
 * Cryptographic Idetification Numbers
 */

#define SUPER_CRYPT_ID_1   0x5C8D  /* Basicly 2 random numbers */
#define SUPER_CRYPT_ID_2   0xDD82  /* Combined to make 64 bits */

#define INODE_CRYPT_ID_1   0x6708  /* Ditto */
#define INODE_CRYPT_ID_2   0xB825

#define PASS_ROOT_ID_1     0x9D7F /* This is the ID for the encrypted */
#define PASS_ROOT_ID_2     0x77E3 /* master structure (64bits)  */


/*
 * This is the actual structure of a inode on disk
 */

struct vs3_inode {
	__u32  i_crypt_id_1;  /* First 32bit ID */
	__u32  i_blocks;      /* Number of blocks that file contains */
	__u32  i_crypt_id_2;  /* Second 32bit ID */
        __u16  i_type;        /* Inode idetification */
	__u16  i_uid;         /* Inode's User ID */
	__u16  i_gid;         /* Inode's Group ID */
	__u32  i_file_size;   /* Size in bytes */
	time_t i_ctime;       /* Creation time */
	time_t i_mtime;       /* Modification Time */
	time_t i_atime;       /* Access Time */
	__u16  i_link_count;  /* Total links to this file */
	__u32  i_dir_entries; /* Number of entries in a directory */
        __u16  i_perms;       /* Inode File permissions */
/* 42 bytes above */

	__u32  i_reserved[3]; /* Reserved for future use */
        __u16  i_resvd;       /* To get 1024 bytes */
        /* i_reserved[0]         Number of bad blocks in bad block inode */

/* 960 bytes below */
	__u32  i_direct[96];  /* Direct Block Mappings = 96kb */
	__u32  i_single[80];  /* Single Tree Mappings += 20MB */
	__u32  i_double[64];  /* Double Tree Mappings += 4GB */
/* Explanation of Block Mappings in v3fs */
/* Direct - Pointers directly to a blocks containing data */
/* Single - Pointers to 1024 byte blocks, which each have */
/*          a list of pointers to blocks containing data */
/* Double - Same a Single except that it is a tree with */
/*          two levels before you get the data */
/* This allows one inode to address upto 4GB of data */

};/* Make sure this structure is 1024 bytes / 8192 bits */


/*
 * The vs3_crypt_master is the master descriptor for each
 * password other than the default password. It contains
 * a pointer to the root, and contains a tree structure that
 * points to the bitmaps of each group. These do not have
 * to be located in the same group that they describe.
 *
 * The only way to get to this structure to to use the
 * password with a hashing function. It's location is
 * NEVER stored on the disk it's self.
 *
 * This is the only way we have found to overcome the problem
 * of knowing where data is stored. Thus, when entering a
 * new password into the system, you may have to try several
 * passwords (if the drive is full) to find one that maps to
 * a blank block. This is a nessasary evil. You may provide a
 * program that automatically generates a password for you that
 * is a derivative of the one the user wants.
 */

struct vs3_crypt_master {
        __u32  cm_crypt_1;    /* First Cryptographic ID */
        __u32  cm_root_ptr;   /* Pointer to encrypted root node */
        __u32  cm_crypt_2;    /* Second Crytographic ID */
	__u32  cm_wc_blocks; /* Number of winnoing / chaffing blocks */

	__u32  cm_reserved[12];  /* Reserved */

/* 960 bytes below */
        __u32  i_direct[96];  /* Direct Block Mappings = 96kb */
        __u32  i_single[80];  /* Single Tree Mappings += 2MB */
        __u32  i_double[64];  /* Double Tree Mappings += 4GB */
/* These are mappings to the bitmaps for each group which
 * are an addition to the standard bitmaps and show which
 * data is being used as winnowing and chaffing information
 */

}; /* Make sure this is 1024 bytes */

/*
 * Structure of the super block on the disk
 */

struct vs3_super_block {
	__u32  s_crypt_id_1;    /* First Cryptographic ID */
	__u32  s_free_blocks;   /* Count of total free blocks */
	__u32  s_crypt_id_2;    /* Second Cryptographic ID */
	__u32  s_block_total;   /* Total number of blocks on disk */
	__u16  s_magic_number;  /* File system magic number = 0x2BAD */
	__u32  s_first_block;   /* Pointer to first data block */
	__u32  s_default_uid;   /* Default User ID */
	__u32  s_default_gid;   /* Default Group ID */
	__u32  s_root_ptr;      /* Pointer to root inode */
	__u32  s_bad_inode;     /* Pointer to bad block inode */
	char   s_vol_name[16];  /* The system's volume name */
	__u32  s_group_no;      /* Number of groups in filesystem */
	__u32  s_this_group_no; /* This superblock's group number */

  /* 62 bytes = 496 bits so far */

  /* 962 bytes free to play with here */

	__u32  s_reserved[239];
	__u16  s_reserved1;
	

}; /* Make sure this structure is 1024 bytes / 8192 bits */


/* Still more to come below once I figure it out */

struct vs3_dir_entry {
        __u32   inode_p;                /* Pointer to Inode */
        __u16   rec_len;                /* Directory entry length */
        __u16   name_len;               /* Name length */
        char    name[VS3_NAME_LEN];     /* File name */
};

#ifdef __KERNEL__ /* if compiling the kernel */

/* Current random offset */
__u32 vs3_rand_pos;

/* Things needed for encryption */

#define Idea_nofRound    8  /* number of rounds              */
#define Idea_userKeyLen  8  /* user key length (8 or larger) */

/* Do not change the lines below.  */

#define Idea_dataLen                     4  /* plain-/ciphertext block length*/
#define Idea_keyLen (Idea_nofRound * 6 + 4) /* en-/decryption key length   */

#define Idea_dataSize       (Idea_dataLen * 2) /* 8 bytes = 64 bits    */
#define Idea_userKeySize (Idea_userKeyLen * 2) /* 16 bytes = 128 bits  */
#define Idea_keySize         (Idea_keyLen * 2) /* 104 bytes = 832 bits */


typedef __u16 Idea_Data[Idea_dataLen];
typedef __u16 Idea_UserKey[Idea_userKeyLen];
typedef __u16 Idea_Key[Idea_keyLen];

typedef __u32 __u96[3];

extern int encrypt_3_way(__u96, __u8*, __u32);
extern int decrypt_3_way(__u96, __u8*, __u32);
extern void init_3way();

extern int encrypt_idea(Idea_UserKey, __u8*, __u32);
extern int decrypt_idea(Idea_UserKey, __u8*, __u32);
extern void Idea_ExpandUserKey (Idea_UserKey userKey, Idea_Key key);
extern void make_3way_key(char* text_key, __u96 three_way_key);
extern void make_idea_key(char* text_key, Idea_UserKey idea_user_key);
extern void init_3way();

/* Global superblock pointer link list */

/*struct super_block */

/*
 * Function prototypes
 */

extern struct super_block *vs3_read_super( struct super_block *sb, void *data, int silent );
extern void vs3_put_inode( struct inode *i );
extern void vs3_read_inode( struct inode *i );
extern void vs3_write_inode( struct inode *i );
extern void vs3_put_super( struct super_block *sb );
extern void vs3_write_super( struct super_block *sb );
extern void vs3_stat_fs( struct super_block *sb, struct statfs *buf, int bufsiz );
extern int  vs3_remount( struct super_block *sb, int *i, char *data );
extern int  vs3_user_mount( struct super_block *sb, void *data );
extern int  vs3_user_unmount( struct super_block *sb );

extern void vs3_truncate( struct inode *i );
extern int vs3_file_read( struct inode *i, struct file *f, char *buf, int c );
extern int vs3_file_write( struct inode *i, struct file *f, const char *buf, int c );
extern int vs3_sync_file( struct inode *i, struct file *f );
extern int vs3_sync_supers( struct super_block *sb, int wait);

extern char* vs3_read_dir_low(struct inode *dir);
extern int vs3_dir_read( struct inode *i, struct file *f, char *buf, int c );
extern int vs3_readdir( struct inode *i, struct file *f, void * dirent, filldir_t filldir );
extern int vs3_stub_read( struct inode *i, struct file *f, char *buf, int len );
extern int vs3_stub_write( struct inode *i, struct file *f, const char *buf, int len );
extern struct buffer_head *vs3_bread( struct inode *i, unsigned int blocknum, int dec );
extern int vs3_mknod(struct inode * dir, const char * name, int len, int mode, int rdev);
extern int vs3_create(struct inode * dir, const char * name, int len, int mode,
	struct inode ** result);
extern int vs3_lookup(struct inode * dir,const char * name, int len, struct inode **result ); 

extern int vs3_mkdir(struct inode * dir, const char * name, int len, int mode);
extern int vs3_rmdir(struct inode * dir, const char * name, int len);
extern int vs3_unlink(struct inode * dir, const char * name, int len);
extern int vs3_rename(struct inode * old_dir, const char * old_name, int old_len,
        struct inode * new_dir, const char * new_name, int new_len,
        int must_be_dir);

extern struct inode * vs3_new_inode(const struct inode * dir);
extern void vs3_free_inode(struct inode * inode);
extern void vs3_bforget ( struct buffer_head *bh );
extern void vs3_bwrite( struct buffer_head *bh, struct inode *i, int enc );

extern int is_block_used(struct super_block * sb, __u32 blk);
extern void set_block_used(struct super_block * sb, __u32 blk);
extern int set_block_free(struct super_block * sb, __u32 blk);
extern __u32 get_free_block(struct super_block * sb);
extern __u32 vs3_find_block( struct inode *i, __u32 block );
extern void vs3_update_pointers( struct inode *i, __u32 block );


extern int vs3_sync_inode(struct inode * inode);

extern void sgenrand(unsigned long seed);
extern unsigned long genrand();


extern struct inode_operations vs3_file_inode_operations;
extern struct inode_operations vs3_dir_inode_operations;

#endif  /* __KERNEL__ */

#endif  /* _LINUX_VS3_FS_H */
