/************************************************************************\
*                                                                        *
*  C specification of the Threeway Block Cipher                          *
*                                                                        *
*  3-Way is an iterated block cipher by Joan Daemen. It uses 96 bits for *
*  its keyspace and block size. There is no known successful             *
*  cryptanalysis against 3-way, and it is free for use.                  *
*                                                                        *
*  Written by Carl van Schaik for the                                    *
*  Steganographic Filesystem (C) 1998 Carl van Schaik                    *
*                                                                        *
\************************************************************************/
#define __KERNEL__
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/locks.h>
#include <linux/vs3_fs.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>

#define   STRT_E   0x0b0b /* round constant of first encryption round */ 
#define   STRT_D   0xb1b1 /* round constant of first decryption round */
#define     NMBR       11 /* number of rounds is 11                   */

__u32 rcon_e[NMBR+1];  /* round constants for encryption */
__u32 rcon_d[NMBR+1];  /* round constants for decryption */

void inline mu(__u32 *a)       /* inverts the order of the bits of a */
{
  __u32 i;
  __u96 b;
  
  b[0] = b[1] = b[2] = 0;
  for( i=0; i<32; i++ )
    {
      b[0] <<= 1; b[1] <<= 1; b[2] <<= 1;
      if(a[0]&1) b[2] |= 1;
      if(a[1]&1) b[1] |= 1;
      if(a[2]&1) b[0] |= 1;
      a[0] >>= 1; a[1] >>= 1; a[2] >>= 1;
    }
  
  a[0] = b[0]; a[1] = b[1]; a[2] = b[2];
}

void inline gamma(__u32 *a)   /* the nonlinear step */
{
  __u96 b;

  b[0] = a[0] ^ (a[1]|(~a[2])); 
  b[1] = a[1] ^ (a[2]|(~a[0])); 
  b[2] = a[2] ^ (a[0]|(~a[1])); 
  
  a[0] = b[0]; a[1] = b[1]; a[2] = b[2];
}


void inline theta(__u32 *a)    /* the linear step */
{
  __u32 b[3];

  b[0] = a[0] ^  (a[0]>>16) ^ (a[1]<<16) ^     (a[1]>>16) ^ (a[2]<<16) ^
                 (a[1]>>24) ^ (a[2]<<8)  ^     (a[2]>>8)  ^ (a[0]<<24) ^
                 (a[2]>>16) ^ (a[0]<<16) ^     (a[2]>>24) ^ (a[0]<<8) ;
  b[1] = a[1] ^  (a[1]>>16) ^ (a[2]<<16) ^     (a[2]>>16) ^ (a[0]<<16) ^
                 (a[2]>>24) ^ (a[0]<<8)  ^     (a[0]>>8)  ^ (a[1]<<24) ^
                 (a[0]>>16) ^ (a[1]<<16) ^     (a[0]>>24) ^ (a[1]<<8) ;
  b[2] = a[2] ^  (a[2]>>16) ^ (a[0]<<16) ^     (a[0]>>16) ^ (a[1]<<16) ^
                 (a[0]>>24) ^ (a[1]<<8)  ^     (a[1]>>8)  ^ (a[2]<<24) ^
                 (a[1]>>16) ^ (a[2]<<16) ^     (a[1]>>24) ^ (a[2]<<8) ;

  a[0] = b[0]; a[1] = b[1]; a[2] = b[2];
}

void inline pi_1(__u32 *a)   
{
  a[0] = (a[0]>>10) ^ (a[0]<<22);  
  a[2] = (a[2]<<1)  ^ (a[2]>>31);
}

void inline pi_2(__u32 *a)   
{
  a[0] = (a[0]<<1)  ^ (a[0]>>31);
  a[2] = (a[2]>>10) ^ (a[2]<<22);
}

void inline rho(__u32 *a)    /* the round function       */
{
  theta(a); 
  pi_1(a); 
  gamma(a); 
  pi_2(a);
}

void inline rndcon_gen(__u32 strt,__u32 *rtab)
{                           /* generates the round constants */
  __u32 i;

  for(i=0; i<=NMBR; i++ )
    {
      rtab[i] = strt;
      strt <<= 1; 
      if( strt&0x10000 ) strt ^= 0x11011;
    }
}

void inline encrypt(__u32 *a, __u32 *key) /* does raw 3-way encryption */
{
  __u32 i;

  for( i=0; i<NMBR; i++ )   
    {
      a[0] ^= key[0] ^ (rcon_e[i]<<16); 
      a[1] ^= key[1]; 
      a[2] ^= key[2] ^ rcon_e[i];
      rho(a);
    }
  a[0] ^= key[0] ^ (rcon_e[NMBR]<<16); 
  a[1] ^= key[1]; 
  a[2] ^= key[2] ^ rcon_e[NMBR];
  theta(a);
}


void inline decrypt(__u32 *a, __u32 *key) /* does raw 3-way decryption */
{             
  __u32 i;
  __u96 keyi;          /* the `inverse' key             */

  keyi[0] = key[0];
  keyi[1] = key[1];
  keyi[2] = key[2]; 

  theta(keyi);
  mu(keyi);

  mu(a);
  for( i=0; i<NMBR; i++ )
    {
      a[0] ^= keyi[0] ^ (rcon_d[i]<<16); 
      a[1] ^= keyi[1]; 
      a[2] ^= keyi[2] ^ rcon_d[i];
      rho(a);
    }
  a[0] ^= keyi[0] ^ (rcon_d[NMBR]<<16); 
  a[1] ^= keyi[1]; 
  a[2] ^= keyi[2] ^ rcon_d[NMBR];
  theta(a);
  mu(a);
}

/* Initialise all round constants */
void init_3way()
{
  rndcon_gen(STRT_E,rcon_e); 
  rndcon_gen(STRT_D,rcon_d); 
}

/* Encrypts a given buffer of lenght len */
int encrypt_3_way(__u96 key, __u8* buffer, __u32 len)
{
  __u32 a=0,i,k,magic=0x12345678;
  __u96 vector, vector2;
  __u8 b;

  vector2[0]=vector2[1]=vector2[2]=0;
  if (len >= 12)
    {
      __u32 j = len/12;  /* 12 bytes = __u96 */
      for (i=0; i<j; i++)
	{
	  vector[0] = buffer[i*12+0] +       (buffer[i*12+1]<<8) +  /* each element in the buffer is 8-bit */
	             (buffer[i*12+2]<<16) +  (buffer[i*12+3]<<24);  /* we need to convert to 96-bit */
	  vector[1] = buffer[i*12+4] +       (buffer[i*12+5]<<8) +
	             (buffer[i*12+6]<<16) +  (buffer[i*12+7]<<24);
	  vector[2] = buffer[i*12+8] +       (buffer[i*12+9]<<8) +
	             (buffer[i*12+10]<<16) + (buffer[i*12+11]<<24);

	  magic *=3;

	  vector2[0] ^= vector[1] ^ magic;
	  vector2[1] ^= vector[2] ^ magic;
	  vector2[2] ^= vector[0] ^ magic;

	  encrypt(vector,key); /* do the encryption */

	  for (k=0; k<3; k++) /* convert the 96 bit vector back to the buffer */
	    {
	      a = vector[k];
	      b = a;
	      buffer[i*12+k*4] = b;
	      a >>= 8;
	      b = a;
	      buffer[i*12+1+k*4] = b;
	      a >>= 8;
	      b = a;
	      buffer[i*12+2+k*4] = b;
	      a >>= 8;
	      b = a;
	      buffer[i*12+3+k*4] = b;
	    }
	}
    }
  else
    {
      __u96 keya;
      keya[0] = key[0];
      keya[1] = key[1];
      keya[2] = key[2];

      theta(keya);
      mu(keya);

      vector2[0] = magic ^ (keya[0] << 7) ^ (keya[2] >> 2) ^ key[1];
      vector2[1] = magic ^ 0x98b43ae1 ^ (key[1] << 5) ^ (key[0] >> 7) ^ keya[2];
      vector2[2] = magic ^ 0x777df45a ^ (key[2] << 3) ^ (key[1] >> 3) ^ keya[0];
    }

  i = len / 12; /* there may be some left over bits if the buffer is not divisible by 12 */
  len %= 12;    /* we need to take them into account as well */

  if (len > 0)
    {
      vector[0] = vector[1] = vector[2] = len*10585737;

      vector[0] ^= buffer[i*12+0];
      if (len > 1)  vector[0] ^= (buffer[i*12+1]<<8);  /* make sure we don't buffer overrun */
      if (len > 2)  vector[0] ^= (buffer[i*12+2]<<16);
      if (len > 3)  vector[0] ^= (buffer[i*12+3]<<24);
      if (len > 4)  vector[1] ^= buffer[i*12+4];
      if (len > 5)  vector[1] ^= (buffer[i*12+5]<<8);
      if (len > 6)  vector[1] ^= (buffer[i*12+6]<<16);
      if (len > 7)  vector[1] ^= (buffer[i*12+7]<<24);
      if (len > 8)  vector[2] ^= buffer[i*12+8];
      if (len > 9)  vector[2] ^= (buffer[i*12+9]<<8);
      if (len > 10) vector[2] ^= (buffer[i*12+10]<<16);
      if (len > 11) vector[2] ^= (buffer[i*12+11]<<24);

      //encrypt(vector,key);
      vector[0] = (vector2[0] ^ vector[0]) ^ 0xa15d345a;
      vector[1] = (vector2[1] ^ vector[1]) ^ 0xbc12dc43;
      vector[2] = (vector2[2] ^ vector[2]) ^ 0x23a74342;

      for (k=0; k<3; k++)
	{
	  if ((k*4) < len)  /* make sure we don't buffer overrun */
	    {
	      a = vector[k];
	      b = a;
	      buffer[i*12+k*4] = b;
	    }
	  if ((k*4+1) < len)
	    {
	      a >>= 8;
	      b = a;
	      buffer[i*12+1+k*4] = b;
	    }
	  if ((k*4+2) < len)
	    {
	      a >>= 8;
	      b = a;
	      buffer[i*12+2+k*4] = b;
	    }
	  if ((k*4+3) < len)
	    {
	      a >>= 8;
	      b = a;
	      buffer[i*12+3+k*4] = b;
	    }
	}
    }

  return 0;
}

/* Decrypts a given buffer of lenght len */
int decrypt_3_way(__u96 key, unsigned char* buffer, __u32 len)
{
  __u32 a=0,i,k,magic=0x12345678;
  __u96 vector, vector2;
  __u8 b;

  vector2[0]=vector2[1]=vector2[2]=0;
  if (len >= 12)
    {
      __u32 j = len/12;  /* 12 bytes = __u96 */
      for (i=0; i<j; i++)
	{
	  vector[0] = buffer[i*12+0] +       (buffer[i*12+1]<<8) +  /* each element in the buffer is 8-bit */
	             (buffer[i*12+2]<<16) +  (buffer[i*12+3]<<24);  /* we need to convert to 96-bit */
	  vector[1] = buffer[i*12+4] +       (buffer[i*12+5]<<8) +
	             (buffer[i*12+6]<<16) +  (buffer[i*12+7]<<24);
	  vector[2] = buffer[i*12+8] +       (buffer[i*12+9]<<8) +
	             (buffer[i*12+10]<<16) + (buffer[i*12+11]<<24);

	  decrypt(vector,key); /* do the encryption */

	  magic *=3;

	  vector2[0] ^= vector[1] ^ magic;
	  vector2[1] ^= vector[2] ^ magic;
	  vector2[2] ^= vector[0] ^ magic;

	  for (k=0; k<3; k++) /* convert the 96 bit vector back to the buffer */
	    {
	      a = vector[k];
	      b = a;
	      buffer[i*12+k*4] = b;
	      a >>= 8;
	      b = a;
	      buffer[i*12+1+k*4] = b;
	      a >>= 8;
	      b = a;
	      buffer[i*12+2+k*4] = b;
	      a >>= 8;
	      b = a;
	      buffer[i*12+3+k*4] = b;
	    }
	}
    }
  else
    {
      __u96 keya;
      keya[0] = key[0];
      keya[1] = key[1];
      keya[2] = key[2];

      theta(keya);
      mu(keya);

      vector2[0] = magic ^ (keya[0] << 7) ^ (keya[2] >> 2) ^ key[1];
      vector2[1] = magic ^ 0x98b43ae1 ^ (key[1] << 5) ^ (key[0] >> 7) ^ keya[2];
      vector2[2] = magic ^ 0x777df45a ^ (key[2] << 3) ^ (key[1] >> 3) ^ keya[0];
    }

  i = len / 12; /* there may be some left over bits if the buffer is not divisible by 12 */
  len %= 12;    /* we need to take them into account as well */

  if (len > 0)
    {
      vector[0] = vector[1] = vector[2] = len*10585737;

      vector[0] ^= buffer[i*12+0];
      if (len > 1)  vector[0] ^= (buffer[i*12+1]<<8);  /* make sure we don't buffer overrun */
      if (len > 2)  vector[0] ^= (buffer[i*12+2]<<16);
      if (len > 3)  vector[0] ^= (buffer[i*12+3]<<24);
      if (len > 4)  vector[1] ^= buffer[i*12+4];
      if (len > 5)  vector[1] ^= (buffer[i*12+5]<<8);
      if (len > 6)  vector[1] ^= (buffer[i*12+6]<<16);
      if (len > 7)  vector[1] ^= (buffer[i*12+7]<<24);
      if (len > 8)  vector[2] ^= buffer[i*12+8];
      if (len > 9)  vector[2] ^= (buffer[i*12+9]<<8);
      if (len > 10) vector[2] ^= (buffer[i*12+10]<<16);
      if (len > 11) vector[2] ^= (buffer[i*12+11]<<24);

      //decrypt(vector,key);
      vector[0] = (vector2[0] ^ vector[0]) ^ 0xa15d345a;
      vector[1] = (vector2[1] ^ vector[1]) ^ 0xbc12dc43;
      vector[2] = (vector2[2] ^ vector[2]) ^ 0x23a74342;

      for (k=0; k<3; k++)
	{
	  if ((k*4) < len)  /* make sure we don't buffer overrun */
	    {
	      a = vector[k];
	      b = a;
	      buffer[i*12+k*4] = b;
	    }
	  if ((k*4+1) < len)
	    {
	      a >>= 8;
	      b = a;
	      buffer[i*12+1+k*4] = b;
	    }
	  if ((k*4+2) < len)
	    {
	      a >>= 8;
	      b = a;
	      buffer[i*12+2+k*4] = b;
	    }
	  if ((k*4+3) < len)
	    {
	      a >>= 8;
	      b = a;
	      buffer[i*12+3+k*4] = b;
	    }
	}
    }

  return 0;
}

void make_3way_key(char* text_key, __u96 t_w_k)
{
        unsigned int i = 0;

        t_w_k[0] = 1234;
        t_w_k[1] = 4321;
        t_w_k[2] = 4132;

        while ((text_key[i] != '\0') && (i < 128))
          {
            t_w_k[0] ^= (__u32)(text_key[(i*text_key[i]) % (i+1)])*i << ((i*text_key[i]) % 25);
            t_w_k[1] += (__u32)(text_key[(i^text_key[i]) % (i+1)])*i << ((i*text_key[i]) % 27);
            t_w_k[2] ^= (__u32)(text_key[i])*i << ((i*text_key[i]) % 24);
            i++;
          }
}

