/*
 *  linux/fs/vs3fs/diskio.c
 *
 * The Steganographic Filesystem (vs3fs)
 *
 * Copyright (C) 1998
 * Paul Smeddle (psmeddle@cs.uct.ac.za)
 * Carl van Schaik (carl@leg.uct.ac.za)
 * University of Cape Town, South Africa
 *
 *  from
 *
 *  libvs3 (C) 1998 C. van Schaik & P. Smeddle
 *             
*/

/*
 *  Note that when talking about the "encrypted" somthing
 *  we are talking about whatever the user has hidden
 *  with a password other than with the default password.
 *  All other things are encrypted with only the default
 *  password.
 */


#define __KERNEL__
#define MODULE
#include <linux/fs.h>
#include <linux/locks.h>
#include <linux/vs3_fs.h>
#include <linux/stat.h>
#include <linux/malloc.h>

/* ----------- Below - For disk reads ---------- */

/* synchronously reads a block from the disk. dec = 0: idea, dec = 1: 3-way */
/*                                            dec = 2; 3way, dec = 3l idea */
struct buffer_head *vs3_bread( struct inode *i, unsigned int blocknum, int dec )
{
   struct buffer_head *bh, *bh2;
   i->u.vs3_i.enc_mode = dec;

   bh2 = bread ( i->i_dev, blocknum, VS3_BLK );

   bh = (struct buffer_head *) kmalloc(sizeof(*bh),GFP_BUFFER);
   memcpy(bh,bh2,sizeof(*bh));
   bh->b_data = (char *) kmalloc(VS3_BLK, GFP_BUFFER);

   if ((bh == NULL) || (bh->b_data == NULL))
   {
      printk("vs3_bread: No more memory\n");
      return NULL;
   }
   if (!bh2 || buffer_uptodate(bh2))
   {
      memcpy(bh->b_data, bh2->b_data, VS3_BLK);
      switch (dec)
      {
	 case VS3_BIDEA:   decrypt_idea (i->i_sb->u.vs3_sb.idea_key, bh->b_data, VS3_BLK); break;
	 case VS3_B3WAY:   decrypt_3_way(i->i_sb->u.vs3_sb.three_way_key, bh->b_data, VS3_BLK); break;
	 case VS3_BU3WAY:  decrypt_3_way(i->i_sb->u.vs3_sb.e_three_way_key, bh->b_data, VS3_BLK); break;
	 case VS3_BUIDEA:  decrypt_idea (i->i_sb->u.vs3_sb.e_idea_key, bh->b_data, VS3_BLK); break;
	 default: printk ("vs3_bread: out of range cipher id\n"); break;
      }
      brelse(bh2);
      return bh;
   }
   ll_rw_block(READ, 1, &bh2);
   wait_on_buffer(bh2);
   if (buffer_uptodate(bh2))
   {
      memcpy(bh->b_data, bh2->b_data, VS3_BLK);
      switch (dec)
      {
	 case VS3_BIDEA:  decrypt_idea(i->i_sb->u.vs3_sb.idea_key,bh->b_data,VS3_BLK); break;
	 case VS3_B3WAY:  decrypt_3_way(i->i_sb->u.vs3_sb.three_way_key,bh->b_data,VS3_BLK); break;
	 case VS3_BU3WAY:  decrypt_3_way(i->i_sb->u.vs3_sb.e_three_way_key,bh->b_data,VS3_BLK); break;
	 case VS3_BUIDEA:  decrypt_idea(i->i_sb->u.vs3_sb.e_idea_key,bh->b_data,VS3_BLK); break;
	 default: printk ("vs3_bread: out of range cipher id\n"); break;
      }
      brelse(bh2);
      return bh;
   }

   printk("vs3_bread: buffer error... returning NULL\n");

   bforget(bh2);
   kfree(bh->b_data);
   kfree(bh);

   return NULL;
}


/* writes a block to the disc. enc = 0: idea, enc = 1: 3-way */
/* This HAS to be sync-ed as we need to makesure data is written */
/* before re-read (to prevent files been read unencrypted) */
void vs3_bwrite( struct buffer_head *bh2, struct inode *i, int enc )
{
   struct buffer_head *bh;

   if (bh2 == NULL)
	return;

   bh = bread ( bh2->b_dev, bh2->b_blocknr, VS3_BLK );

   memcpy(bh->b_data, bh2->b_data, VS3_BLK);

   switch (enc)
   {
      case VS3_BIDEA:  encrypt_idea(i->i_sb->u.vs3_sb.idea_key,bh->b_data,VS3_BLK); break;
      case VS3_B3WAY:  encrypt_3_way(i->i_sb->u.vs3_sb.three_way_key,bh->b_data,VS3_BLK); break;
      case VS3_BU3WAY:  encrypt_3_way(i->i_sb->u.vs3_sb.e_three_way_key,bh->b_data,VS3_BLK); break;
      case VS3_BUIDEA:  encrypt_idea(i->i_sb->u.vs3_sb.e_idea_key,bh->b_data,VS3_BLK); break;
      default: printk ("vs3_bwrite: out of range cipher id\n"); break;
   }
   mark_buffer_dirty(bh,1);
//   ll_rw_block(WRITE,1,&bh);
//   wait_on_buffer(bh);
   brelse(bh);

   kfree(bh2->b_data);
   kfree(bh2);
}

void vs3_bforget ( struct buffer_head *bh )
{
   if (bh != NULL)
   {
      kfree(bh->b_data);
      kfree(bh);
   }
}


/* Sets a bit in the bitmap on */
void inline bitset(char * bitm,int bit)
{
        if ((bit >= 8192) || (bit < 0))
          return;
        bitm[bit/8] |= (1 << (bit % 8));
}

/* Sets a bit in the bitmap off */
void inline bitclear(char * bitm,int bit)
{
        if ((bit >= 8192) || (bit < 0))
          return;

        bitset(bitm, bit);
        bitm[bit/8] = (bitm[bit/8] - (1 << (bit % 8)));
}


/* Checks whether a bit in bitmap is set or not */
int inline is_bit_set(char * bitm,int bit)
{
        int tst;

        if ((bit >= 8192) || (bit < 0))
          return (1);

        tst = bitm[bit/8];
        tst |= (1 << (bit % 8));

        return (tst == bitm[bit/8]);
}

int is_block_used(struct super_block * sb, __u32 blk)
{
   struct buffer_head *bh;
   struct inode ino;
   int offs = VS3_BOOT_SECT_SIZE/VS3_BLK;
   __u32 i,o;
   int result;

   if (blk < offs)
      return(1);

   i = (blk-offs) / VS3_GROUP;
   o = (blk-offs) % VS3_GROUP;

   if (i > sb->u.vs3_sb.s_group_no)
      return (1);
   if (blk > sb->u.vs3_sb.s_block_total)
      return (1);

   ino.i_dev = sb->s_dev;
   ino.i_sb = sb;
   bh = vs3_bread (&ino, offs+i*VS3_GROUP+1, VS3_B3WAY);
   result = is_bit_set(bh->b_data, o);
   vs3_bforget(bh);

   return result;
}

/* Sets a block marked used on the disk */
void set_block_used(struct super_block * sb, __u32 blk)
{
   struct buffer_head *bh;
   struct inode ino;
   int offs = VS3_BOOT_SECT_SIZE/VS3_BLK;
   unsigned int i,o;
   i = (blk-offs) / VS3_GROUP;
   o = (blk-offs) % VS3_GROUP;
   ino.i_dev = sb->s_dev;
   ino.i_sb = sb;
   bh = vs3_bread( &ino, offs+i*VS3_GROUP+1, VS3_B3WAY);
   if (is_bit_set(bh->b_data,o))
   {
      printk("vs3_set_block_used: tried to use block already in use\n");
      vs3_bforget(bh);
      return;
   }
   bitset(bh->b_data,o);
   vs3_bwrite(bh, &ino, VS3_B3WAY);

   sb->u.vs3_sb.s_free_blocks--;
   vs3_write_super(sb);
}

/* Sets a block marked free on the disk */
int set_block_free(struct super_block * sb, __u32 blk)
{
   struct buffer_head *bh;
   struct inode ino;
   int offs = VS3_BOOT_SECT_SIZE/VS3_BLK;
   unsigned int i,o;

   i = (blk-offs) / VS3_GROUP;
   o = (blk-offs) % VS3_GROUP;

   ino.i_dev = sb->s_dev;
   ino.i_sb = sb;

   bh = vs3_bread( &ino, offs+i*VS3_GROUP+1, VS3_B3WAY);

   if (!is_bit_set(bh->b_data,o))
      {
         printk("vs3: bit already not set");
	 vs3_bforget(bh);
         return -1;
      }

   bitclear(bh->b_data,o);

   vs3_bwrite(bh, &ino, VS3_B3WAY);

   sb->u.vs3_sb.s_free_blocks++;
   vs3_write_super(sb);
   return 0;
}

/*Gets a free block on the device */
__u32 get_free_block(struct super_block * sb)
{
   __u32 pos;

   vs3_rand_pos += genrand() % 256;
   pos = vs3_rand_pos % sb->u.vs3_sb.s_block_total;
   while (is_block_used(sb,pos))
      pos = (++pos) % sb->u.vs3_sb.s_block_total;

   return pos;
}

