/*
 *  linux/fs/vs3fs/fsync.c
 *
 * The Steganographic Filesystem (vs3fs)
 *
 *   Copyright (C) 1998
 *   Paul Smeddle (psmeddle@cs.uct.ac.za)
 *   Carl van Schaik (carl@leg.uct.ac.za)
 *   University of Cape Town, South Africa
 *
 * Portions (C) from ext2 / msdos / minix filesystems
 */

#define __KERNEL__
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/fcntl.h>
#include <linux/locks.h>

#include <linux/vs3_fs.h>

#include <asm/segment.h>
#include <asm/system.h>

#define blocksize BLOCK_SIZE

/*
 * The functions for vs3fs file synchronization.
 */
static int sync_block (struct inode * inode, __u32 block, int wait)
{
	struct buffer_head * bh;
	__u32 tmp;
	
	if (!block)
		return 0;
	tmp = block;
	bh = get_hash_table(inode->i_dev, block, blocksize);
	if (!bh)
		return 0;
	if (block != tmp) {
		brelse (bh);
		return 1;
	}
	if (wait && buffer_req(bh) && !buffer_uptodate(bh)) {
		brelse(bh);
		return -1;
	}
	if (wait || !buffer_uptodate(bh) || !buffer_dirty(bh))
	{
		brelse(bh);
		return 0;
	}
	ll_rw_block(WRITE, 1, &bh);
	bh->b_count--;
	return 0;
}

int vs3_sync_supers(struct super_block *sb, int wait)
{
	__u32 i;
	int rc, err = 0;
        __u32 pos = 1, group = 0;
	struct vs3_super_block * SB;
	struct buffer_head * bh;
	struct inode inode;

	SB = sb->u.vs3_sb.s_vs;
	inode.i_dev = sb->s_dev;

        while (group < SB->s_group_no)
          {
            SB->s_this_group_no = group; /* Set curent group */
	    bh = bread(sb->s_dev, pos, VS3_BLK);
	    memcpy(bh->b_data, (char*)SB, VS3_BLK);
	    encrypt_3_way(sb->u.vs3_sb.three_way_key,bh->b_data,VS3_BLK);
	    mark_buffer_dirty(bh,0);
	    brelse(bh); 

            group++;
            pos = 1 + group*VS3_GROUP;
          }
	SB->s_this_group_no = 0;

	/* sync super blocks */
	for (i = 0; i < sb->u.vs3_sb.s_vs->s_group_no; i++) {
		rc = sync_block (&inode, (i+1)*VS3_GROUP,wait);
		if (rc > 0)
			break;
		if (rc)
			err = rc;
	}

	/* sync bitmaps */
        for (i = 0; i < sb->u.vs3_sb.s_vs->s_group_no; i++) {
                rc = sync_block (&inode, (i+1)*VS3_GROUP+1,wait);
                if (rc > 0)
                        break;
                if (rc)
                        err = rc;
        }
	return err;
}


static int sync_file(struct inode *inode, int wait)
{
	int i;
	int rc = 0, err = 0;
	__u32 blk;

	for (i = 0; i < inode->i_blocks; i++)
	{
	   blk = vs3_find_block(inode, i);
	   rc = sync_block(inode,blk,wait);
	   if (rc > 0)
		break;
	   if (rc) err = rc;
	}
	return err;
}

/*
 * The function which is called for file synchronization.
 */
int vs3_sync_file(struct file * file, struct dentry *dentry)
{
	struct inode *inode = dentry->d_inode;
	int wait, err = 0;
	
	if (!(S_ISREG(inode->i_mode) || S_ISDIR(inode->i_mode) ||
	     S_ISLNK(inode->i_mode)))
		return -EINVAL;

	vs3_sync_inode(inode);

	for (wait=0; wait<=1; wait++)
	{
		err |= sync_file(inode, wait);
	}
	err |= vs3_sync_inode (inode);
	return (err < 0) ? -EIO : 0;
}

