/*
 *  linux/fs/vs3_fs/namei.c
 *
 * The Steganographic Filesystem (vs3fs)
 *
 *   Copyright (C) 1998
 *   Paul Smeddle (psmeddle@cs.uct.ac.za)
 *   Carl van Schaik (carl@leg.uct.ac.za)
 *   University of Cape Town, South Africa
 *
 * Portions (C) from ext2 / msdos / minix filesystems
 */

#define __KERNEL__
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/fcntl.h>

#include <linux/errno.h>
#include <linux/vs3_fs.h>

/*
 * Compares two strings, of length <len>
 *
 * NOTE! unlike strncmp, vs3_match returns 1 for success, 0 for failure.
 */
static inline int vs3_match (int len, const char * const name,
   struct vs3_dir_entry * de)
{
   if (!de || len > VS3_NAME_LEN)
      return 0;
   if (len != de->name_len)
      return 0;
   /* simple memory compare */
   return !memcmp(name, de->name, len);
}


/* Reads in a directory into a single buffer */
/* You need to free the buffer yourself when done */
char* vs3_read_dir_low(struct inode *dir)
{
   char *data;
   unsigned int i,place;
   struct buffer_head * bh;

   /* Arrrgh... ugly but it works, and makes life */
   /* a lot easier... allocate one BIG arrary */
   data = (char*)kmalloc(dir->i_blocks*VS3_BLK,GFP_KERNEL);
   if (data == NULL)
      return NULL;


/* ---- Read in the blocks into the buffer ---- */

   for (i=0;i<dir->i_blocks;i++) /* read in direct entries */
   {
      place = vs3_find_block(dir,i);
      if (place == 0)
        {
          printk("vs3_read_dir_low : ran past end of directory\n");
          break;
        }
      bh = vs3_bread(dir, place, dir->u.vs3_i.enc_mode);
      memcpy(data+i*VS3_BLK, bh->b_data, VS3_BLK);
      vs3_bforget(bh);
   }

   /* release the buffer */
   return data;
}



/*
 *	vs3_find_entry()
 *
 * finds an entry in the specified directory with the wanted name. It
 * returns the cache buffer in which the entry was found, and the entry
 * itself (as a parameter - res_dir). It does NOT read the inode of the
 * entry - you'll have to do that yourself if you want to.
 */
static unsigned int vs3_find_entry (struct inode * dir,
   const char * const name, int namelen)
{
   struct vs3_dir_entry * de;
   __u32 size;
   unsigned int pos, off;
   char * buffer;

   if (namelen > VS3_NAME_LEN)
      return 0;

   if (!dir)
      return 0;

   if (dir->i_sb->u.vs3_sb.user_key_in_use)
      if (dir->i_ino == dir->i_sb->u.vs3_sb.s_root_ptr)
	  if (!strcmp("secret",name))
	     return dir->i_sb->u.vs3_sb.s_crypt_master->cm_root_ptr;

   /* get the buffer containing directory entries */
   buffer = vs3_read_dir_low(dir);
   size = dir->u.vs3_i.i_dir_entries;
   if (buffer == NULL)
      return 0;

   de = (struct vs3_dir_entry *)(buffer);
   off = 0;

   /* process every directory entry */
   for (pos = 0; pos < size; pos++)
   {
      off = de->rec_len;
      if (!off)
	 return 0;
      if (vs3_match (namelen, name, de))
      {
	 pos = de->inode_p;
	 kfree(buffer);
	 return pos;
      }

      de = (struct vs3_dir_entry *)((char*)de + off);
   }
   /* don't want a memory leak :-) */
   kfree(buffer);
   return 0;
}


/* Looks for a files in a directory */
int vs3_lookup(struct inode * dir, struct dentry *dentry)
{
   __u32 i;
   struct inode *result = NULL;

   i = vs3_find_entry (dir,dentry->d_name.name,dentry->d_name.len); /* find the specifed file */

   if (i)
   {
      result = iget(dir->i_sb, i);

      if (!result)
          return -EACCES;
   }
   d_add(dentry, result);
   return 0;
}


/*
 * vs3_add_entry()
 *
 * adds a file entry to the specified directory
 */
static int vs3_add_entry(struct inode * dir,
   const char * name, int namelen, __u32 ppos)
   
{
   __u32 i, place, blks, next=0;
   struct vs3_dir_entry * de;
   char * buffer;

   if (!dir || !dir->i_sb)
      return -ENOENT;

   if (!namelen)
      return -ENOENT;

   if (namelen > VS3_NAME_LEN)
      return -ENAMETOOLONG;

/* -------- allocate memory -------- */
   buffer = (char*)kmalloc((dir->i_blocks+1)*VS3_BLK, GFP_KERNEL);
   if (buffer == NULL)
      return -ENOENT;

/* ------ Read in to the buffer ------- */
   for (i=0;i<dir->i_blocks;i++) /* read in direct entries */
   {
      struct buffer_head * bh;
      place = vs3_find_block(dir,i);
      if (place == 0)
	{
	  printk("vs3_add_entry : ran past end of directory");
          break;
	}
      bh = vs3_bread(dir, place, dir->u.vs3_i.enc_mode);
      memcpy(buffer+i*VS3_BLK, bh->b_data, VS3_BLK);
      vs3_bforget(bh);
   }

/* ------- Scan to end of buffer -------- */

   de = (struct vs3_dir_entry *)(buffer);
   blks = 0;
   for (i = 0; i < dir->u.vs3_i.i_dir_entries; i++)
   {
      if (vs3_match (namelen, name, de))
	 return -EEXIST;
      blks += de->rec_len;
      de = (struct vs3_dir_entry *)((char*)de + de->rec_len);
   }
   blks += (namelen+8);
   blks = blks / VS3_BLK;
   if (blks >= dir->i_blocks)
   {
      next=1;
      dir->i_blocks++;
   }

   /* set the new paramaters */
   de->name_len = namelen;
   de->rec_len = 8+namelen;
   de->inode_p = ppos;

   memcpy (de->name, name, namelen);

   dir->i_mtime = dir->i_ctime = CURRENT_TIME;
   dir->u.vs3_i.i_dir_entries++;
   mark_inode_dirty(dir);
 
/* ---------- Write back to the buffer ----------- */

   for (i=0;i<dir->i_blocks;i++) /* read in direct entries */
   {
      struct buffer_head * bh;
      if ((next) && (i+1 == dir->i_blocks))
      {
         blks = get_free_block( dir->i_sb );
         set_block_used(dir->i_sb,blks);
         dir->i_blocks--;
         vs3_update_pointers( dir, blks );
         dir->i_size += VS3_BLK;
	 mark_inode_dirty(dir);
      }
      place = vs3_find_block(dir,i);
      if( (int)place == -1 )
        {
           printk( "vs3fs_add_entry: no free space left on device.\n" );
           return -EACCES;
        }
      bh = vs3_bread(dir, place, dir->u.vs3_i.enc_mode);
      memcpy(bh->b_data, buffer+i*VS3_BLK, VS3_BLK);
      vs3_bwrite(bh, dir, dir->u.vs3_i.enc_mode);
   }

   kfree(buffer);
   return 0;
}


/* create a new file in a directory */
int vs3_mknod(struct inode * dir, struct dentry *dentry, int mode, int rdev)
{
   int error;
   __u32 i;
   struct inode * inode;

   if (!dir)
      return -ENOENT;

   /* check that it's not already there */
   i = vs3_find_entry (dir,dentry->d_name.name,dentry->d_name.len);
   if (i)
      return -EEXIST;

   /* get a new inode VFS's inode chaching system, */
   /* with some preset values */
   inode = vs3_new_inode(dir);
   if (!inode)
      return -ENOSPC; /* not enough space ? */

   /* setup some default settings */
   inode->i_uid = current->fsuid;
   inode->i_mode = mode;
   inode->i_op = NULL;

   /* setup the inode_operations pointers depending on the */
   /* type of files we are creating */
   if (S_ISREG(inode->i_mode))
   {
      inode->i_op = &vs3_file_inode_operations;
      inode->u.vs3_i.i_type = VS3_NORMAL_INO;
   }
   else if (S_ISDIR(inode->i_mode))
   {
      inode->i_op = &vs3_dir_inode_operations;
      if (dir->i_mode & S_ISGID)
	 inode->i_mode |= S_ISGID;
      inode->u.vs3_i.i_type = VS3_DIRECTORY_INO;
   }
   else if (S_ISLNK(inode->i_mode))
      return -EACCES; /* vs3fs has no implementation for symlinks yet */
   else if (S_ISCHR(inode->i_mode))
   {
      inode->i_op = &chrdev_inode_operations;
      inode->u.vs3_i.i_type = VS3_CHRDEV_INO;
   }
   else if (S_ISBLK(inode->i_mode))
   {
      inode->i_op = &blkdev_inode_operations;
      inode->u.vs3_i.i_type = VS3_BLKDEV_INO;
   }
   else if (S_ISFIFO(inode->i_mode))
   {
      init_fifo(inode);
      inode->u.vs3_i.i_type = VS3_FIFO_INO;
   }
   if (S_ISBLK(mode) || S_ISCHR(mode))
      inode->i_rdev = to_kdev_t(rdev);

   /* set file time stamps */
   inode->i_atime = inode->i_ctime = inode->i_atime = CURRENT_TIME;
   mark_inode_dirty(inode);

   /* try to add the entry to the directory */
   error = vs3_add_entry(dir, dentry->d_name.name, dentry->d_name.len, inode->i_ino);
   if (error) {
		 inode->i_nlink--;
		 mark_inode_dirty(inode);
		 iput(inode);
		 return error;
	      }
   d_instantiate(dentry, inode);
   return 0;
}


/* creates a new directory entry */
int vs3_create(struct inode * dir, struct dentry *dentry, int mode)
{
   struct inode * inode;
   int error;

   if (!dir)
      return -ENOENT;

   inode = vs3_new_inode(dir);
   if (!inode)
	return -ENOSPC;
   inode->i_op = &vs3_file_inode_operations;
   inode->i_mode = mode;
   inode->u.vs3_i.i_type = VS3_NORMAL_INO;
   mark_inode_dirty(inode);

   error = vs3_add_entry(dir, dentry->d_name.name, dentry->d_name.len, inode->i_ino);
   if (error) {
		 inode->i_nlink--;
		 mark_inode_dirty(inode);
		 iput(inode);
		 return error;
	      }

   d_instantiate(dentry, inode);
   return 0;
}


/* create a new sub directory in the current one */
int vs3_mkdir(struct inode * dir, struct dentry *dentry, int mode)
{
   struct inode * inode;
   struct buffer_head * bh, *dir_block;
   struct vs3_inode * ino;
   struct vs3_dir_entry * de;
   int error,i;
   __u32 tem;

   if (!dir || !dir->i_sb)
        return -EINVAL;
   /* check that it's not already there */
   i = vs3_find_entry (dir,dentry->d_name.name,dentry->d_name.len);
   if (i)
      return -EEXIST;
   if (dir->i_nlink > 64000)
         return -EMLINK;

   inode = vs3_new_inode(dir);
   if (!inode)
	  return -ENOSPC;

   /* setup the new inode */
   inode->i_op = &vs3_dir_inode_operations;
   inode->i_blocks = 1;
   inode->i_size = inode->i_blocks*VS3_BLK;
   inode->u.vs3_i.i_dir_entries = 2;
   inode->u.vs3_i.i_type=VS3_DIRECTORY_INO;
   mark_inode_dirty(inode);

   tem = get_free_block(inode->i_sb);
   set_block_used(inode->i_sb,tem);
   dir_block = vs3_bread(inode,tem,inode->u.vs3_i.enc_mode);

   if (!dir_block)
      {
         inode->i_nlink--;
         mark_inode_dirty(inode);
         iput(inode);
         return -ENOSPC;
      }
   de = (struct vs3_dir_entry *) dir_block->b_data;
   de->inode_p=inode->i_ino;
   strcpy(de->name,".");
   de->name_len=1;
   de->rec_len=10;
   de =(struct vs3_dir_entry *)(10 + (char*)dir_block->b_data);
   de->inode_p = dir->i_ino;
   strcpy(de->name,"..");
   de->name_len=2;
   de->rec_len=11;
   inode->i_nlink = 2;
   vs3_bwrite(dir_block, inode, inode->u.vs3_i.enc_mode);
   mark_inode_dirty(inode);

   bh = vs3_bread(inode, inode->i_ino, inode->u.vs3_i.enc_mode);
   ino = (struct vs3_inode *)bh->b_data;
   if( (ino->i_crypt_id_1 != INODE_CRYPT_ID_1) || (ino->i_crypt_id_2 != INODE_CRYPT_ID_2) )
   {
      vs3_bforget(bh);
      printk( "vs3_mkdir: Unable to read inode data : %li\n",inode->i_ino );
      return -EACCES;
   }
   ino->i_direct[0]=tem;
   ino->i_blocks = 1;
   vs3_bwrite(bh, inode, inode->u.vs3_i.enc_mode);

   inode->i_mode = S_IFDIR | (mode & 0777 & ~current->fs->umask);
   if (dir->i_mode & S_ISGID)
       inode->i_mode |= S_ISGID;
   mark_inode_dirty(inode);
   error = vs3_add_entry(dir, dentry->d_name.name, dentry->d_name.len, inode->i_ino);
   if (error) {
		 inode->i_nlink=0;
		 iput(inode);
		 return -ENOSPC;
	      }
   dir->i_nlink++;
   mark_inode_dirty(dir);
   d_instantiate(dentry, inode);
   return 0;
}


/*
 * routine to check that the specified directory is empty (for rmdir)
 */
static int empty_dir(struct inode * inode)
{
   __u32 block;
   struct buffer_head * bh;
   struct vs3_dir_entry * de;

   if (!inode || !inode->i_sb)
      return 1;
   block = 0;
   bh = NULL;

   if (inode->u.vs3_i.i_dir_entries < 2)
      goto bad_dir;

   block = vs3_find_block(inode,0);
   if ((int)block == -1)
      goto bad_dir;
   bh = vs3_bread(inode, block, inode->u.vs3_i.enc_mode);
   if (!bh)
      goto bad_dir;
   de = (struct vs3_dir_entry *) bh->b_data;
   if (!de->inode_p || !vs3_match(1,".",de))
      goto bad_dir;
   de = (struct vs3_dir_entry *) (bh->b_data + de->rec_len);
   if (!de->inode_p || !vs3_match(2,"..",de))
      if (de->inode_p) {
			  vs3_bforget(bh);
			  return 0;
		       }
   vs3_bforget(bh);
   if (inode->u.vs3_i.i_dir_entries > 2)
      return 0;
   return 1;

   bad_dir:
      if (bh)
         vs3_bforget(bh);
   printk("vs3_empty_dir: Bad directory on device %s\n",
      kdevname(inode->i_dev));
   return 1;
}


/* does the actual remove for rmdir */
void vs3_rm_entry(struct inode * dir, __u32 inode_p)
{
   char * buffer, *buffer2;
   __u32 off, i, pos, blks;
   struct vs3_dir_entry *de, *de2;

printk("rm_entry\n");
   buffer = vs3_read_dir_low(dir);
   buffer2 = vs3_read_dir_low(dir);

   if ((buffer == NULL) || (buffer2 == NULL))
   {
      printk("vs3_rm_dir: Error not enough memory to delete\n");
      if (buffer) kfree(buffer);
      if (buffer2) kfree(buffer2);
      return;
   }
   off = 0;
   de = (struct vs3_dir_entry *)(buffer);
   de2 = (struct vs3_dir_entry *)(buffer2);

   /* process every directory entry */
   for (pos = 0; pos < dir->u.vs3_i.i_dir_entries; pos++)
   {
      off = de->rec_len;
      if (!off)
         return;
      if (de->inode_p == inode_p)
	 de = (struct vs3_dir_entry *)((char*)de + off);

      de2->inode_p  = de->inode_p;
      de2->rec_len  = de->rec_len;
      de2->name_len = de->name_len;
      memcpy(de2->name,de->name,de->name_len);

      de2 = (struct vs3_dir_entry *)((char*)de2 + de2->rec_len);
      de = (struct vs3_dir_entry *)((char*)de + de->rec_len);
   }
   kfree(buffer);

   dir->u.vs3_i.i_dir_entries--;
   dir->i_mtime = dir->i_ctime = CURRENT_TIME;
   mark_inode_dirty(dir);

/* ------- Scan to end of buffer -------- */
   de = (struct vs3_dir_entry *)(buffer2);
   blks = 0;
   for (i = 0; i < dir->u.vs3_i.i_dir_entries; i++)
   {
      blks += de->rec_len;
      de = (struct vs3_dir_entry *)((char*)de + de->rec_len);
   }
 
   blks = blks / VS3_BLK + 1;
   if (blks != dir->i_blocks)
   {
      dir->i_blocks--;
      pos = vs3_find_block(dir,dir->i_blocks);
      set_block_free(dir->i_sb,pos);
      mark_inode_dirty(dir);
   }
 
   for (i=0;i<dir->i_blocks;i++) /* read in direct entries */
   {
      struct buffer_head * bh;
      pos = vs3_find_block(dir,i);
      if( (int)pos == -1 )
        {
           printk( "vs3fs_rm_entry: no free space left on device.\n" );
	   kfree(buffer2);
           return;
        }
      bh = vs3_bread(dir, pos, dir->u.vs3_i.enc_mode);
      memcpy(bh->b_data, buffer2+i*VS3_BLK, VS3_BLK);
      vs3_bwrite(bh, dir, dir->u.vs3_i.enc_mode);
   }

   kfree(buffer2);
}

/* Removes a directory entry from the current directory */
int vs3_rmdir(struct inode * dir,  struct dentry *dentry)
{
   __u32 i;
   int retval;
   struct inode * inode;

   inode = NULL;
   i = vs3_find_entry (dir,dentry->d_name.name,dentry->d_name.len);
   retval = -ENOENT;
   if (!i)
      goto end_rmdir;

   inode = iget(dir->i_sb,i);
   retval = -EPERM;

   if ((dir->i_mode & S_ISVTX) &&
      current->fsuid != inode->i_uid &&
      current->fsuid != dir->i_uid && !capable(CAP_FOWNER))
      goto end_rmdir;
   if (inode->i_dev != dir->i_dev)
      goto end_rmdir;
   if (inode->i_ino == dir->i_ino)	/* we may not delete ".", but "../dir" is ok */
      goto end_rmdir;
   if (!S_ISDIR(inode->i_mode))
   {
      retval = -ENOTDIR;
      goto end_rmdir;
   }

   if (dentry->d_count > 1)
                shrink_dcache_parent(dentry);

   if (!empty_dir(inode))
   {
      retval = -ENOTEMPTY;
      goto end_rmdir;
   }
   if (dentry->d_count > 1)
   {
      retval = -EBUSY;
      goto end_rmdir;
   }
   if (inode->i_nlink != 2)
      printk("vs3_rmdir: empty directory has nlink!=2 (%d)\n",inode->i_nlink);

   vs3_rm_entry(dir,i);

   dir->i_version = ++event;
   dir->i_ctime = dir->i_mtime = CURRENT_TIME;
   dir->i_nlink--;
   mark_inode_dirty(dir);
   retval = 0;

   d_delete(dentry);
   set_block_free(dir->i_sb, i);

end_rmdir:
   return retval;
}

int vs3_unlink(struct inode * dir,  struct dentry *dentry)
{
    int retval;
    __u32 i;
    struct inode * inode;

repeat:
    retval = -ENOENT;
    inode = NULL;
    i = vs3_find_entry(dir, dentry->d_name.name, dentry->d_name.len);

    inode = iget(dir->i_sb,i);

    retval = -EPERM;
    if (S_ISDIR(inode->i_mode))
        goto end_unlink;
    if (i != inode->i_ino) {
        current->counter = 0;
        schedule();
        goto repeat;
    }
    if ((dir->i_mode & S_ISVTX) && !fsuser() &&
            current->fsuid != inode->i_uid &&
            current->fsuid != dir->i_uid)
        goto end_unlink;
    if (!inode->i_nlink) {
        printk("vs3_unlink: Deleting nonexistent file\n");
        inode->i_nlink=1;
    }
    vs3_rm_entry(dir, i);

    dir->i_version = ++event;
    dir->i_ctime = dir->i_mtime = CURRENT_TIME;
    mark_inode_dirty(dir);
    retval = 0;

    inode->i_nlink--;
    inode->i_ctime = dir->i_ctime;
    mark_inode_dirty(inode);

    d_delete(dentry);

end_unlink:

    return retval;
}

#define PARENT_INO(buffer) \
(((struct vs3_dir_entry *) ((buffer)+ ((struct vs3_dir_entry *)(buffer))->rec_len))->inode_p)

/*
 * rename uses retrying to avoid race-conditions: at least they should be minimal.
 * it tries to allocate all the blocks, then sanity-checks, and if the sanity-
 * checks fail, it tries to restart itself again. Very practical - no changes
 * are done until we know everything works ok.. and then all the changes can be
 * done in one fell swoop when we have claimed all the buffers needed.
 *
 * Anybody can rename anything with this: the permission checks are left to the
 * higher-level routines.
 */
static int do_vs3_rename(struct inode * old_dir, struct dentry *old_dentry,
			 struct inode * new_dir, struct dentry *new_dentry)
{
        struct inode * old_inode, * new_inode;
        struct buffer_head * dir_bh;
        struct minix_sb_info * info;
	__u32 old_ptr, new_ptr;
        int retval;

        info = &old_dir->i_sb->u.minix_sb;
        goto start_up;
try_again:
        vs3_bforget(dir_bh);
        current->counter = 0;
        schedule();
start_up:
        old_inode = new_inode = NULL;
        dir_bh = NULL;
        old_ptr = vs3_find_entry(old_dir,old_dentry->d_name.name,old_dentry->d_name.len);
        retval = -ENOENT;
        if (!old_ptr)
                goto end_rename;
        old_inode = old_dentry->d_inode; /* don't cross mnt-points */
        if (!old_inode)
                goto end_rename;
        retval = -EPERM;
        if ((old_dir->i_mode & S_ISVTX) &&
            current->fsuid != old_inode->i_uid &&
            current->fsuid != old_dir->i_uid && !fsuser())
                goto end_rename;
	new_inode = new_dentry->d_inode;
        new_ptr = vs3_find_entry(new_dir,new_dentry->d_name.name,new_dentry->d_name.len);
        if (new_ptr) {
                if (!new_inode) {
                        new_ptr = 0;
                }
        }
        if (new_inode == old_inode) {
                retval = 0;
                goto end_rename;
        }
        if (new_inode && S_ISDIR(new_inode->i_mode)) {
                retval = -EISDIR;
                if (!S_ISDIR(old_inode->i_mode))
                        goto end_rename;
                retval = -EINVAL;
                if (is_subdir(new_dentry, old_dentry))
                        goto end_rename;
                retval = -ENOTEMPTY;
                if (!empty_dir(new_inode))
                        goto end_rename;
                retval = -EBUSY;
                if (new_inode->i_count > 1)
                        goto end_rename;
        }
        retval = -EPERM;
        if (new_inode && (new_dir->i_mode & S_ISVTX) &&
            current->fsuid != new_inode->i_uid &&
            current->fsuid != new_dir->i_uid && !fsuser())
                goto end_rename;
        if (S_ISDIR(old_inode->i_mode)) {
                retval = -ENOTDIR;
                if (new_inode && !S_ISDIR(new_inode->i_mode))
                        goto end_rename;
                retval = -EINVAL;
                if (is_subdir(new_dentry, old_dentry))
                        goto end_rename;
                retval = -EIO;
                dir_bh = vs3_bread(old_inode,old_inode->i_ino,old_inode->u.vs3_i.enc_mode);
                if (!dir_bh)
                        goto end_rename;
                if (PARENT_INO(dir_bh->b_data) != old_dir->i_ino)
                        goto end_rename;
                retval = -EMLINK;
                if (!new_inode && new_dir->i_nlink >= 256)
                        goto end_rename;
        }
        if (!new_inode) {
                retval = vs3_add_entry(new_dir,new_dentry->d_name.name,new_dentry->d_name.len,old_ptr);
                if (retval)
                        goto end_rename;
        }
/* sanity checking before doing the rename - avoid races */
        if (new_inode && (new_ptr != new_inode->i_ino))
                goto try_again;
        if (new_ptr && !new_inode)
                goto try_again;
        if (old_ptr != old_inode->i_ino)
                goto try_again;
/* ok, that's it */
        old_dir->i_ctime = old_dir->i_mtime = CURRENT_TIME;
        mark_inode_dirty(old_dir);
        old_dir->i_version = ++event;
        new_dir->i_ctime = new_dir->i_mtime = CURRENT_TIME;
        mark_inode_dirty(new_dir);
        new_dir->i_version = ++event;
	vs3_rm_entry(old_dir,old_ptr);
	
        if (new_inode)
	{
                new_inode->i_nlink--;
                new_inode->i_ctime = CURRENT_TIME;
                mark_inode_dirty(new_inode);
        }

        if (dir_bh) {
                PARENT_INO(dir_bh->b_data) = new_dir->i_ino;
                old_dir->i_nlink--;
                mark_inode_dirty(old_dir);
                if (new_inode) {
                        new_inode->i_nlink--;
                        mark_inode_dirty(new_inode);
                } else {
                        new_dir->i_nlink++;
                        mark_inode_dirty(new_dir);
                }
        }
        /* Update the dcache */
        d_move(old_dentry, new_dentry);
        retval = 0;
end_rename:
	if (dir_bh)
           vs3_bwrite(dir_bh, new_dir, new_dir->u.vs3_i.enc_mode);
        return retval;
}


/*
 * Ok, rename also locks out other renames, as they can change the parent of
 * a directory, and we don't want any races. Other races are checked for by
 * "do_rename()", which restarts if there are inconsistencies.
 *
 * Note that there is no race between different filesystems: it's only within
 * the same device that races occur: many renames can happen at once, as long
 * as they are on different partitions.
 */
int vs3_rename(struct inode * old_dir, struct dentry *old_dentry,
	       struct inode * new_dir, struct dentry *new_dentry)
{
        static struct wait_queue * wait = NULL;
        static int lock = 0;
        int result;

        while (lock)
                sleep_on(&wait);
        lock = 1;
        result = do_vs3_rename(old_dir, old_dentry,
			       new_dir, new_dentry);
        lock = 0;
        wake_up(&wait);
        return result;
}

