/*
 * /linux/fs/vs3_fs/vs3fs.c  
 *
 * The Steganographic Filesystem (vs3fs)
 *
 *   Copyright (C) 1998
 *   Paul Smeddle (psmeddle@cs.uct.ac.za)
 *   Carl van Schaik (carl@leg.uct.ac.za)
 *   University of Cape Town, South Africa	  
 */
 
/* 
 * This is the self-contained code for the kernel device driver.
 * This driver is only useful as a module.
 *
 * this code is derived in part from the various files in the fs/minix 
 * directory.
 *
 */
 
/* vs3fs.c - Paul Smeddle
 */
 
#define __KERNEL__
#define MODULE
#include <linux/module.h>
#include <asm/uaccess.h>
#include <linux/fs.h>
#include <linux/locks.h>
#include <linux/vs3_fs.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>

/*******VFS ROUTINE POINTERS**************************************************/

/* vs3fs's entry in the VFS's fs linked-list structure. See /proc/filesystems. */
static struct file_system_type vs3fs_type = { 
	"vs3fs",
	1,
	vs3_read_super,
	NULL 
};

/* pointers to superblock operations */
static struct super_operations vs3_sops = {
	vs3_read_inode,
	vs3_write_inode,
	NULL,
	vs3_delete_inode, 
	NULL,
	vs3_put_super,
	vs3_write_super,
	vs3_stat_fs,
	vs3_remount,
	NULL,
	NULL
};

/* pointers to default file operations */
static struct file_operations vs3_file_operations = {
	NULL,
	vs3_file_read,
	vs3_file_write,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	vs3_sync_file,
	NULL,
	NULL,
	NULL
};

/* pointers to default directory operations */
static struct file_operations vs3_directory_operations = {
	NULL,
	vs3_dir_read,
	NULL,
	vs3_readdir,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	vs3_sync_file,
	NULL,
	NULL,
	NULL
};

/* pointers to file inode operations */
static struct inode_operations vs3_file_inode_operations = {
	&vs3_file_operations,
	NULL,
	NULL,
	NULL,	
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	vs3_truncate,				
	NULL
};

/* pointers to directory inode operations */
static struct inode_operations vs3_dir_inode_operations = {
	&vs3_directory_operations,
        vs3_create,
	vs3_lookup,
	NULL,	
	vs3_unlink,
	NULL,
        vs3_mkdir,
        vs3_rmdir,
        vs3_mknod,
	vs3_rename,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	vs3_truncate,				
	NULL
};

/* procfs stub file operations */
struct file_operations vs3_stub_file_operations =
{
	NULL,
	vs3_stub_read,
	vs3_stub_write,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

/* procfs stub inode operations */
struct inode_operations vs3_stub_inode_operations =
{
	&vs3_stub_file_operations,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

/* inode value of /proc/vs3 (needed for unregistering later) */
static int vs3_stub_inode_value = 0;

/* boolean flip-flop for vs3_read_stub */
static int vs3_stub_readb = 0;

/* structure storing disk info */
struct vs3_list {
	struct super_block *sb;
	struct vs3_list *next;
};

struct vs3_list *vs3_list;

/*******SUPERBLOCK ROUTINES***************************************************/

/* VFS super-block read routine for the vs3fs. Taken largely from minix/inode.c */
struct super_block *vs3_read_super( struct super_block *sb, void *data, int silent )
{
	struct buffer_head *bh;
	struct vs3_super_block *vs;
	kdev_t dev = sb->s_dev;
        struct inode i, *root_inode;
	struct vs3_list *temp, *new_entry;

	MOD_INC_USE_COUNT;
	lock_super( sb );
	set_blocksize( dev, VS3_BLK );

/* get the keys */
   	make_idea_key(data, sb->u.vs3_sb.idea_key);
        make_3way_key(data, sb->u.vs3_sb.three_way_key);

        sgenrand((__u32)data + sb->u.vs3_sb.three_way_key[0]*sb->u.vs3_sb.three_way_key[2]);
	vs3_rand_pos = genrand();
        memset(data,0,strlen(data));

        i.i_sb = sb;
        i.i_dev = dev;

        if (!(bh = vs3_bread(&i, 1, VS3_B3WAY))) {
                        sb->s_dev = 0;
                        unlock_super( sb );
                        if( !silent )
                                printk("vs3fs: super-block read failure.\n");
                        MOD_DEC_USE_COUNT;
                        return NULL;
        }
        vs = (struct vs3_super_block *) bh->b_data;
	sb->u.vs3_sb.s_bh = bh;
	sb->u.vs3_sb.s_vs = vs;
  	sb->u.vs3_sb.s_free_blocks = vs->s_free_blocks;  
	sb->u.vs3_sb.s_block_total = vs->s_block_total;
	sb->u.vs3_sb.s_magic_number = vs->s_magic_number;
	sb->u.vs3_sb.s_first_block = vs->s_first_block;
	sb->u.vs3_sb.s_default_uid = vs->s_default_uid;
	sb->u.vs3_sb.s_default_gid = vs->s_default_gid;
	sb->u.vs3_sb.s_root_ptr = vs->s_root_ptr;
	sb->u.vs3_sb.s_bad_inode = vs->s_bad_inode;
	sb->u.vs3_sb.s_group_no = vs->s_group_no;
	sb->u.vs3_sb.user_key_in_use = 0;
	sb->u.vs3_sb.s_crypt_master =
	   (struct vs3_crypt_master *)kmalloc(sizeof(*sb->u.vs3_sb.s_crypt_master),GFP_KERNEL);
        sb->s_blocksize = VS3_BLK;
        sb->s_blocksize_bits = 10;
        sb->s_magic = vs->s_magic_number;
        if ( (sb->s_magic != VS3_SUPER_MAGIC) || 
             (vs->s_crypt_id_1 != SUPER_CRYPT_ID_1 ) || 
             (vs->s_crypt_id_2 != SUPER_CRYPT_ID_2 ) ) {
                unlock_super(sb);
                vs3_bforget( bh );
                sb->s_dev = 0;
	        if (!silent)
                    printk("vs3fs: no vs3fs filesystem found on %s.\n", kdevname( dev ) );
                MOD_DEC_USE_COUNT;
                return NULL;
        }
	unlock_super( sb );
	sb->s_dev = dev;
	sb->s_op = &vs3_sops;
	root_inode = iget( sb, vs->s_root_ptr );
	if ( !root_inode ) {
                vs3_bforget( bh );
                sb->s_dev = 0;
                if ( !silent )
                	printk("vs3fs: get root inode failed.\n");
                MOD_DEC_USE_COUNT;
                return NULL;
        }
        sb->s_root = d_alloc_root(root_inode, NULL);

	/* Insert fs as end of list */
	temp = vs3_list;
	while (temp->next != NULL)
	   temp = temp->next;
	new_entry = (struct vs3_list *)kmalloc(sizeof(*vs3_list),GFP_KERNEL);
	new_entry->sb = sb;
	new_entry->next = NULL;
	temp->next = new_entry;

/*        if (!vs3_user_mount( sb, "stego" ))
           printk("user mount good\n"); */
/*	check root file-system here */
        return sb;
}

/* create statfs structure (for df and suchlike). largely taken from fs/minix/inode.c */
int vs3_stat_fs( struct super_block *sb, struct statfs *buf, int bufsiz )
{
	struct statfs tmp;
	
        tmp.f_type = sb->s_magic;
        tmp.f_bsize = sb->s_blocksize;
        tmp.f_blocks = sb->u.vs3_sb.s_block_total;
        tmp.f_bfree = sb->u.vs3_sb.s_free_blocks;
        tmp.f_bavail = tmp.f_bfree;
        tmp.f_files = 0; // sb->u.minix_sb.s_ninodes;
        tmp.f_ffree = 0; // minix_count_free_inodes(sb);
        tmp.f_namelen = VS3_NAME_LEN;
        return copy_to_user( buf, &tmp, bufsiz )  ? -EFAULT : 0; 
}

/* put superblock routine (unmount) */
void vs3_put_super( struct super_block *sb )
{
	struct vs3_list *temp, *old_entry;
	struct inode i;
	int wait=0;
	i.i_sb = sb;
	if (vs3_user_unmount(sb))
	   printk("vs3fs: User filesystem unmount failed\n");

	/* Removed fs from the list */
	temp = vs3_list;
        while ((temp->next->sb != sb) && (temp->next != NULL))
           temp = temp->next;
	old_entry = temp->next;
	temp->next = old_entry->next;
	kfree(old_entry);

	vs3_sync_supers(sb,wait);
        kfree(sb->u.vs3_sb.s_crypt_master);

    	sb->u.vs3_sb.idea_key[0] = 0;
   	sb->u.vs3_sb.idea_key[1] = 0;
   	sb->u.vs3_sb.idea_key[2] = 0;
   	sb->u.vs3_sb.idea_key[3] = 0;
        sb->u.vs3_sb.three_way_key[0] = 0;
        sb->u.vs3_sb.three_way_key[1] = 0;
        sb->u.vs3_sb.three_way_key[2] = 0;

	MOD_DEC_USE_COUNT;	
}

/* write superblock routine */
void vs3_write_super( struct super_block *sb )
{
	struct buffer_head *bh;
	struct vs3_super_block *vs;
	struct inode i;
        kdev_t dev = sb->s_dev;

        i.i_sb = sb;
        i.i_dev = dev;

	lock_super( sb );
	set_blocksize( dev, VS3_BLK );
	bh = sb->u.vs3_sb.s_bh;
        vs = (struct vs3_super_block *) bh->b_data;

  	vs->s_free_blocks = sb->u.vs3_sb.s_free_blocks;  
        vs->s_block_total = sb->u.vs3_sb.s_block_total;
        vs->s_magic_number = sb->u.vs3_sb.s_magic_number;
	vs->s_first_block = sb->u.vs3_sb.s_first_block;
        vs->s_default_uid = sb->u.vs3_sb.s_default_uid;
        vs->s_default_gid = sb->u.vs3_sb.s_default_gid;
	vs->s_root_ptr = sb->u.vs3_sb.s_root_ptr;
	vs->s_bad_inode = sb->u.vs3_sb.s_bad_inode;
	vs->s_group_no = sb->u.vs3_sb.s_group_no;
        sb->s_magic = vs->s_magic_number;
        if ( (sb->s_magic != VS3_SUPER_MAGIC) || 
             (vs->s_crypt_id_1 != SUPER_CRYPT_ID_1 ) || 
             (vs->s_crypt_id_2 != SUPER_CRYPT_ID_2 ) ) {
                sb->s_dev = 0;
                unlock_super(sb);
                vs3_bforget( bh );
                panic("vs3 : Superblock verify error");
        }

	vs3_bwrite( bh, &i, VS3_B3WAY );
	bh = vs3_bread( &i, 1, VS3_B3WAY );
        sb->u.vs3_sb.s_bh = bh;
        vs = (struct vs3_super_block *) bh->b_data;

        if ( (sb->s_magic != VS3_SUPER_MAGIC) ||
             (vs->s_crypt_id_1 != SUPER_CRYPT_ID_1 ) ||
             (vs->s_crypt_id_2 != SUPER_CRYPT_ID_2 ) ) {
                sb->s_dev = 0;
                unlock_super(sb);
                vs3_bforget( bh );
                printk("vs3fs: error reading superblock" );
                MOD_DEC_USE_COUNT;
                return;
        }
        unlock_super( sb );
	sb->s_dirt=0;
}

/* remount the filesystem */
int vs3_remount( struct super_block *sb, int *i, char *data )
{
	return 0; // always allow remounts unchanged
}


/* Mount a user password into the filesystem */
int vs3_user_mount( struct super_block *sb, void *data )
{
        struct buffer_head *bh;
        struct vs3_crypt_master *cm;
	struct inode i;
	__u32 pos;

        make_idea_key(data, sb->u.vs3_sb.e_idea_key);
        make_3way_key(data, sb->u.vs3_sb.e_three_way_key);

        pos = ((sb->u.vs3_sb.e_three_way_key[0]+sb->u.vs3_sb.e_three_way_key[1]+
                sb->u.vs3_sb.e_three_way_key[2]) % sb->u.vs3_sb.s_block_total);

        i.i_sb = sb;
        i.i_dev = sb->s_dev;
	bh = vs3_bread(&i, pos, VS3_BU3WAY);
	cm = (struct vs3_crypt_master *)bh->b_data;

        if  ((cm->cm_crypt_1 != SUPER_CRYPT_ID_1 ) ||
             (cm->cm_crypt_2 != SUPER_CRYPT_ID_2 ) ) {
                vs3_bforget( bh );
                printk("No files exist for given password\n");
                return -1;
        }

	sb->u.vs3_sb.s_crypt_master = cm;
	sb->u.vs3_sb.s_cm_bh = bh;
	sb->u.vs3_sb.user_key_in_use = 1;

	return 0;
}

/* UnMount a user password from the filesystem */
int vs3_user_unmount( struct super_block *sb )
{
	struct buffer_head *bh;
	struct vs3_crypt_master *cm;
	struct inode i;

	cm = sb->u.vs3_sb.s_crypt_master;
	bh = sb->u.vs3_sb.s_cm_bh;

        i.i_sb = sb;
        i.i_dev = sb->s_dev;

	vs3_bwrite(bh, &i, VS3_BU3WAY);	

        sb->u.vs3_sb.user_key_in_use = 0;
        sb->u.vs3_sb.e_idea_key[0] = 0;
        sb->u.vs3_sb.e_idea_key[1] = 0;
        sb->u.vs3_sb.e_idea_key[2] = 0;
        sb->u.vs3_sb.e_idea_key[3] = 0;
        sb->u.vs3_sb.e_three_way_key[0] = 0;
        sb->u.vs3_sb.e_three_way_key[1] = 0;
        sb->u.vs3_sb.e_three_way_key[2] = 0;

	return 0;
}

/*******INODE ROUTINES********************************************************/

/* inode put routine ( deletes an inode ) */
void vs3_delete_inode( struct inode *i )
{
	i->i_size = 0;
	vs3_truncate(i);
	vs3_free_inode(i);
}

/* inode read routine */
extern void vs3_read_inode( struct inode *i )
{
	struct buffer_head *bh;
	struct vs3_inode *raw_inode;
	unsigned int inode_ptr;
	inode_ptr = i->i_ino;
	i->i_op = NULL;
	i->i_mode = 0;
	if( !inode_ptr || (inode_ptr > i->i_sb->u.vs3_sb.s_block_total) ) {
		printk( "vs3_read_inode: Bad inode pointer on device. Out of range %d. dev %s\n",
			inode_ptr, kdevname( i->i_dev ) );
		return;			
	}

        if (i->i_sb->u.vs3_sb.user_key_in_use) {
           if (i->i_ino == i->i_sb->u.vs3_sb.s_crypt_master->cm_root_ptr)
	      i->u.vs3_i.enc_mode = VS3_BUIDEA;
	}
	else
	   i->u.vs3_i.enc_mode = VS3_BIDEA;
		

	if( ! ( bh = vs3_bread( i, inode_ptr , i->u.vs3_i.enc_mode ) ) ) {
		printk( "vs3_read_inode: Unable to read inode %d from dev. %s\n", inode_ptr,
			kdevname( i->i_dev ) );
		return;
	}		   		
	raw_inode = (struct vs3_inode *)bh->b_data;
	if( (raw_inode->i_crypt_id_1 != INODE_CRYPT_ID_1 ) || 
	    (raw_inode->i_crypt_id_2 != INODE_CRYPT_ID_2 ) ) {
		vs3_bforget(bh);
                if (i->i_sb->u.vs3_sb.user_key_in_use) {
		   i->u.vs3_i.enc_mode = VS3_BUIDEA;
                   bh = vs3_bread( i, inode_ptr , VS3_BUIDEA );
                   if ((raw_inode->i_crypt_id_1 != INODE_CRYPT_ID_1 ) ||
                      (raw_inode->i_crypt_id_2 != INODE_CRYPT_ID_2 ) ) {
                           vs3_bforget(bh);
                           printk( "vs3_read_inode : Invalid inode : %i\n" , inode_ptr);
                           return;
                        }
                }
                else
                {
                   printk( "vs3_read_inode : Invalid inode : %i\n" , inode_ptr);
                   return;
                }
	}	 
	switch( raw_inode->i_type )
	{
		case VS3_NORMAL_INO : i->i_mode = S_IFMT & S_IFREG; break;
		case VS3_ROOT_INO : i->i_mode = S_IFMT & S_IFDIR; break;
		case VS3_CRYPT_ROOT : i->i_mode = S_IFMT & S_IFDIR; break;
		case VS3_DIRECTORY_INO : i->i_mode = S_IFMT & S_IFDIR; break;
		case VS3_FIFO_INO : i->i_mode = S_IFMT & S_IFIFO; break;
		case VS3_CHRDEV_INO : i->i_mode = S_IFMT & S_IFCHR; break;
		case VS3_BLKDEV_INO : i->i_mode = S_IFMT & S_IFBLK; break;
	}
	i->i_mode |= (raw_inode->i_perms&0777);
	i->i_uid = raw_inode->i_uid;
	i->i_gid = raw_inode->i_gid;
	i->i_nlink = raw_inode->i_link_count;
	i->i_size = raw_inode->i_file_size;
	i->i_mtime = raw_inode->i_mtime;
	i->i_atime = raw_inode->i_atime;
	i->i_ctime = raw_inode->i_ctime;
	i->i_blocks = raw_inode->i_blocks;
        i->i_blksize = VS3_BLK;
	i->u.vs3_i.i_type = raw_inode->i_type;
        i->u.vs3_i.i_dir_entries = raw_inode->i_dir_entries;
	
/*	if (S_ISCHR( i->i_mode ) || S_ISBLK( ie->i_mode ) )
	                i->i_rdev = to_kdev_t( raw_inode->i_zone[0] );
*/
        vs3_bforget( bh );
	switch( i->u.vs3_i.i_type )
	{
		case VS3_ROOT_INO : i->i_op = &vs3_dir_inode_operations; break;
		case VS3_CRYPT_ROOT : i->i_op = &vs3_dir_inode_operations; break;
		case VS3_DIRECTORY_INO : i->i_op = &vs3_dir_inode_operations; break;
		case VS3_FIFO_INO : init_fifo( i ); break;
		case VS3_CHRDEV_INO : i->i_op = &chrdev_inode_operations; break;
		case VS3_BLKDEV_INO : i->i_op = &blkdev_inode_operations; break;
		case VS3_NORMAL_INO : i->i_op = &vs3_file_inode_operations; break;
		default : printk( "vs3_read_inode: bad inode type.\n" ); break;
	}
}

/* inode write routine */
extern void vs3_write_inode( struct inode *i )
{
	struct buffer_head *bh;
	struct vs3_inode *raw_inode;
	unsigned int inode_ptr;

	inode_ptr = i->i_ino;
	if( !inode_ptr || (inode_ptr > i->i_sb->u.vs3_sb.s_block_total) ) {
		printk( "vs3_write_inode: Bad inode pointer on device. Out of range %d. dev %s\n",
			inode_ptr, kdevname( i->i_dev ) );
		return;			
	}
	if( ! ( bh = vs3_bread( i, inode_ptr , i->u.vs3_i.enc_mode ) ) ) {
		printk( "vs3_write_inode: Unable to read inode %d from dev. %s\n", inode_ptr,
			kdevname( i->i_dev ) );
		return;
	}		   		
	raw_inode = (struct vs3_inode *)bh->b_data;
	if( (raw_inode->i_crypt_id_1 != INODE_CRYPT_ID_1 ) || 
	    (raw_inode->i_crypt_id_2 != INODE_CRYPT_ID_2 ) )
	{
	    	printk( "vs3_write_inode : Invalid inode : %i\n", inode_ptr);
	        vs3_bforget( bh );
		return;
	}
	raw_inode->i_uid = i->i_uid;
	raw_inode->i_gid = i->i_gid;
	raw_inode->i_link_count = i->i_nlink;
	raw_inode->i_file_size = i->i_size;
	raw_inode->i_mtime = i->i_mtime;
	raw_inode->i_atime = i->i_atime;
	raw_inode->i_ctime = i->i_ctime;
	raw_inode->i_type = i->u.vs3_i.i_type;
        raw_inode->i_dir_entries = i->u.vs3_i.i_dir_entries;
        raw_inode->i_perms = i->i_mode&0777;
        raw_inode->i_type = i->u.vs3_i.i_type;

	raw_inode->i_blocks = i->i_blocks;
	vs3_bwrite( bh, i, i->u.vs3_i.enc_mode );
}


/****** DIRECTORY ROUTINES ***************************************************/

int vs3_dir_read( struct file *f, char *buf, size_t c, loff_t *ppos )
{
        return -EISDIR;
}

/*int vs3_readdir( struct inode *i, struct file *f, void * dirent, filldir_t filldir )
{
	if (!i || !i->i_sb || !S_ISDIR(i->i_mode))
	                return -EBADF;
}*/       
        


/*******PROC ROUTINES*********************************************************/

int vs3_stub_read( struct file *f, char *buf, size_t len, loff_t *ppos )
{
	char tmp[1024];
	struct vs3_list *temp = vs3_list;

	len = 0;
	if( vs3_stub_readb == 0 ) {
		while (temp->next != NULL)
		{
			char state[25];
			temp = temp->next;
			if (temp->sb->u.vs3_sb.user_key_in_use)
				strcpy(state,"| Password in use");
			else
				strcpy(state,"| Password not in use");
			len += sprintf( tmp+len, "%s       %s\n", kdevname(temp->sb->s_dev), state ); 
		}
		copy_to_user( buf, tmp, len );
		vs3_stub_readb = 1;
		return len;
	} else {
		vs3_stub_readb = 0;
		return 0;
	}
}

int vs3_stub_write( struct file *f, const char *buf, size_t len, loff_t *ppos )
{
	char tmp[1024];
	char *dev;
        struct vs3_list *temp = vs3_list;

	memset( tmp, '\0', sizeof( tmp ) );
	copy_from_user( &tmp, buf, (len < 255 ? len : 255 ) ); 
	
	dev = strtok(tmp," ");
        while (temp->next != NULL)
        {
                temp = temp->next;
                if (!strcmp(kdevname(temp->sb->s_dev), dev ))
		{
			dev = strtok(NULL," ");
			if ((dev[0] == 'U') && (temp->sb->u.vs3_sb.user_key_in_use))
			{
			   if (vs3_user_unmount( temp->sb ))
				printk("vs3_user_unmount: failed\n");
			}
			else if ((dev[0] == 'M') && (!temp->sb->u.vs3_sb.user_key_in_use))
			{
				dev = strtok(NULL," ");
				vs3_user_mount( temp->sb, dev );
			}
		}
        }

	return len;
}

/*******MODULE ROUTINES*******************************************************/

/* module initialisation routine */
int init_module( void )
{
	int status;
	struct proc_dir_entry *p;

	status = register_filesystem( &vs3fs_type );
	if( status == 0 )
	   printk( "vs3fs module loaded. (c) 1998 P.Smeddle, C. van Schaik.\n" );
	else
	   printk( "vs3fs module load failed.\n" );
	p = create_proc_entry( "vs3", S_IFREG | S_IRUGO | S_IWUGO, &proc_root );
	if ( !p )
	{
           printk( "vs3fs module load failed.\n" );
           return -ENOMEM;
	}

	p->ops = &vs3_stub_inode_operations;
	vs3_stub_inode_value = p->low_ino;
	printk( "/proc/vs3 stub loaded.\n" );
   	init_3way();

	vs3_list = (struct vs3_list *)kmalloc(sizeof(*vs3_list),GFP_KERNEL);
	vs3_list->sb = NULL;
	vs3_list->next = NULL;

	return( status );		
}

/* module cleanup routine */
void cleanup_module( void )
{
	kfree(vs3_list);
	unregister_filesystem( &vs3fs_type );
	if( vs3_stub_inode_value ) proc_unregister( &proc_root, vs3_stub_inode_value );
}
