/******************************************************************************/
/*                                                                            */
/* I N T E R N A T I O N A L  D A T A  E N C R Y P T I O N  A L G O R I T H M */
/*                                                                            */
/* Modified for the Steganographic Filesystem by                              */
/*   Carl van Schaik                                                          */
/******************************************************************************/
/* Original Program                                                           */
/* Author:       Richard De Moliner (demoliner@isi.ee.ethz.ch)                */
/*               Signal and Information Processing Laboratory                 */
/*               Swiss Federal Institute of Technology                        */
/*               CH-8092 Zuerich, Switzerland                                 */
/* Created:      April 23, 1992                                               */
/******************************************************************************/
#include "idea.h"

#define mulMod        0x10001 /* 2**16 + 1                                    */
#define ones           0xFFFF /* 2**16 - 1                                    */

/******************************************************************************/
/* Multiplication in the multiplicative group, a = a * b                      */
/* pre:  0 <= a <= 0xFFFF.                                                    */
/*       0 <= b <= 0xFFFF.                                                    */
/* post: 'a' and 'b' have been modified.                                      */
/*       a = a * b; where '*' is multiplication in the multiplicative group.  */
/* note: This implementation of '*' is not complete. To bee complete the      */
/*       result has to bee masked (MUL(a, b); a &= ones;).                    */

#define Mul(a, b)                                                              \
  if (a == 0) a = mulMod - b;                                                  \
  else if (b == 0) a = mulMod - a;                                             \
  else {                                                                       \
    a *= b;                                                                    \
    if ((a & ones) >= (b = a >> 16)) a -= b;                                   \
    else a += mulMod - b;                                                      \
  } /* Mul */

/******************************************************************************/
/* Encryption and decryption algorithm IDEA. Depending on the value of 'key'  */
/* 'Idea_Crypt' either encrypts or decrypts 'dataIn'. The result is stored    */
/* in 'dataOut'.                                                              */
/* pre:  'dataIn'  contains the plain/cipher-text block.                      */
/*       'key'     contains the encryption/decryption key.                    */
/* post: 'dataOut' contains the cipher/plain-text block.                      */

#ifdef ANSI_C
  void Idea_Crypt (Idea_Data dataIn, Idea_Data dataOut, Idea_Key key)
#else
  Idea_Crypt (dataIn, dataOut, key)
  Idea_Data dataIn;
  Idea_Data dataOut;
  Idea_Key key;
#endif

{ register word32 x0, x1, x2, x3, t0, t1, t2;
  int round;

  x0 = (word32)*dataIn++; x1 = (word32)*dataIn++;
  x2 = (word32)*dataIn++; x3 = (word32)*dataIn;
  for (round = Idea_nofRound; round > 0; round--) {
    t1 = (word32)*key++;
    x1 += (word32)*key++;
    x2 += (word32)*key++; x2 &= ones;
    t2 = (word32)*key++;
    Mul(x0, t1); x0 &= ones;
    Mul(x3, t2);
    t0 = (word32)*key++;
    t1 = x0 ^ x2;
    Mul(t0, t1); t0 &= ones;
    t1 = (word32)*key++;
    t2 = (x1 ^ x3) + t0 & ones;
    Mul(t1, t2); t1 &= ones;
    t0 += t1;
    x0 ^= t1; x3 ^= t0; x3 &= ones;
    t0 ^= x1; x1 = x2 ^ t1; x2 = t0;
  }
  t0 = (word32)*key++;
  Mul(x0, t0);
  *dataOut++ = (word16)(x0 & ones);
  *dataOut++ = (word16)((word32)*key++ + x2 & ones);
  *dataOut++ = (word16)((word32)*key++ + x1 & ones);
  t0 = (word32)*key;
  Mul(x3, t0);
  *dataOut = (word16)(x3 & ones);
} /* Idea_Crypt */

/******************************************************************************/
/* Multiplicative Inverse by Extended Stein Greatest Common Divisor Algorithm.*/
/* pre:  0 <= x <= 0xFFFF.                                                    */
/* post: x * MulInv(x) == 1, where '*' is multiplication in the               */
/*                           multiplicative group.                            */

#ifdef ANSI_C
  static word16 MulInv (word16 x)
#else
  static word16 MulInv (x)
  word16 x;
#endif

{ register int32 n1, n2, N, a1, a2, b1, b2;

  if (x <= 1) return x;
  n1 = N = (int32)x; n2 = mulMod;
  a1 = b2 = 1; a2 = b1 = 0;
  do {
    while ((n1 & 1) == 0) {
      if (a1 & 1)
        if (a1 < 0) { a1 += mulMod; b1 -= N; }
        else { a1 -= mulMod; b1 += N; }
      n1 >>= 1; a1 >>= 1; b1 >>= 1;
    }
    if (n1 < n2)
      do {
        n2 -= n1; a2 -= a1; b2 -= b1;
        if (n2 == 0) return (word16)(a1 < 0 ? a1 + mulMod : a1);
        while ((n2 & 1) == 0) {
          if (a2 & 1)
            if (a2 < 0) { a2 += mulMod; b2 -= N; }
            else { a2 -= mulMod; b2 += N; }
          n2 >>= 1; a2 >>= 1; b2 >>= 1;
        }
      } while (n1 <= n2);
    n1 -= n2; a1 -= a2; b1 -= b2;
  } while (n1);
  return (word16)(a2 < 0 ? a2 + mulMod : a2);
} /* MulInv */

/******************************************************************************/
/* Additive Inverse.                                                          */
/* pre:  0 <= x <= 0xFFFF.                                                    */
/* post: x + AddInv(x) == 0, where '+' is addition in the additive group.     */

#define AddInv(x)  (-x & ones)

/******************************************************************************/
/* Inverts a decryption/encrytion key to a encrytion/decryption key.          */
/* pre:  'key'    contains the encryption/decryption key.                     */
/* post: 'invKey' contains the decryption/encryption key.                     */

#ifdef ANSI_C
  void Idea_InvertKey (Idea_Key key, Idea_Key invKey)
#else
  Idea_InvertKey (key, invKey)
  Idea_Key key;
  Idea_Key invKey;
#endif

{ register word16 t, *in, *out;
  register int lo, hi, i;

  in = key; out = invKey;
  lo = 0; hi = 6 * Idea_nofRound;
  t = MulInv(in[lo]); out[lo++] = MulInv(in[hi]); out[hi++] = t;
  t = AddInv(in[lo]); out[lo++] = AddInv(in[hi]); out[hi++] = t;
  t = AddInv(in[lo]); out[lo++] = AddInv(in[hi]); out[hi++] = t;
  t = MulInv(in[lo]); out[lo++] = MulInv(in[hi]); out[hi] = t;
  for (i = (Idea_nofRound - 1) / 2 ; i != 0 ; i --) {
    t = in[lo]; out[lo++] = in[hi -= 5]; out[hi ++] = t;
    t = in[lo]; out[lo++] = in[hi]; out[hi] = t;
    t = MulInv(in[lo]); out[lo++] = MulInv(in[hi -= 5]); out[hi++] = t;
    t = AddInv(in[lo]); out[lo++] = AddInv(in[++hi]); out[hi--] = t;
    t = AddInv(in[lo]); out[lo++] = AddInv(in[hi]); out[hi++] = t;
    t = MulInv(in[lo]); out[lo++] = MulInv(in[++hi]); out[hi] = t;
  }
#if (Idea_nofRound % 2 == 0)
  t = in[lo]; out[lo++] = in[hi -= 5]; out[hi++] = t;
  t = in[lo]; out[lo++] = in[hi]; out[hi] = t;
  out[lo] = MulInv(in[lo]); lo++;
  t = AddInv(in[lo]); out[lo] = AddInv(in[lo + 1]); lo++; out[lo++] = t;
  out[lo] = MulInv(in[lo]);
#else
  out[lo] = in[lo]; lo++;
  out[lo] = in[lo];
#endif
} /* Idea_InvertKey */

/******************************************************************************/
/* Expands a user key of 128 bits to a full encryption key                    */
/* pre:  'userKey' contains the 128 bit user key                              */
/* post: 'key'     contains the encryption key                                */

#ifdef ANSI_C
  void Idea_ExpandUserKey (Idea_UserKey userKey, Idea_Key key)
#else
  Idea_ExpandUserKey (userKey, key)
  Idea_UserKey userKey;
  Idea_Key key;
#endif

{ register int i;

#if (Idea_keyLen <= Idea_userKeyLen)
  for (i = 0; i < Idea_keyLen; i++) key[i] = userKey[i];
#else
  for (i = 0; i < Idea_userKeyLen; i++) key[i] = userKey[i];
  for (i = Idea_userKeyLen; i < Idea_keyLen; i++)
    if ((i & 7) < 6)
      key[i] = (key[i - 7] & 127) << 9 | key[i - 6] >> 7;
    else if ((i & 7) == 6)
      key[i] = (key[i - 7] & 127) << 9 | key[i - 14] >> 7; 
    else
      key[i] = (key[i - 15] & 127) << 9 | key[i - 14] >> 7; 
#endif
} /* Idea_ExpandUserKey */


/******************************************************************************/
/* Encrypts a set of data of length len                                       */
/* pre: buffer                                                                */
/* post: buffer (modified)                                                    */
/* (C) Carl van Schaik (carl@leg.uct.ac.za)                                   */

int encrypt_idea(Idea_UserKey key, word8* buffer, word32 len)
{
  word32 i,k, magic=0x12345678;
  Idea_Data data,vector;
  Idea_Key iKey;
  word16 a;
  word8 b;

  Idea_ExpandUserKey(key, iKey);

  vector[0]=vector[1]=vector[2]=vector[3]=0;
  if (len >= 8)
    {
      word32 j = len/8;  /* 8 bytes = Idea_dataSize */
      for (i=0; i<j; i++)
	{
	  data[0] = buffer[i*8+0] + (buffer[i*8+1] << 8);
	  data[1] = buffer[i*8+2] + (buffer[i*8+3] << 8);
	  data[2] = buffer[i*8+4] + (buffer[i*8+5] << 8);
	  data[3] = buffer[i*8+6] + (buffer[i*8+7] << 8);

	  magic *=3;
	  vector[0] ^= data[1] ^ magic;
	  vector[1] ^= data[3] ^ magic;
	  vector[2] ^= data[0] ^ magic;
	  vector[3] ^= data[2] ^ magic;

	  Idea_Crypt(data, data, iKey);

	  for (k=0; k<4; k++)
	    {
	      a = data[k];
	      b = (word8)a;
	      buffer[i*8+0+k*2] = b;
	      a >>= 8;
	      b = (word8)a;
	      buffer[i*8+1+k*2] = b;
	    }
	}
    }

  i = len / 8; /* there may be some left over bits if the buffer is not divisible by 8 */
  len %= 8;    /* we need to take them into account as well */

  if (len > 0)
    {
      data[0]=data[1]=data[2]=data[3]=len*131071;

      data[0] ^= buffer[i*8+0];
      if (len > 1)  data[0] ^= (buffer[i*8+1]<<8);  /* make sure we don't buffer overrun */
      if (len > 2)  data[1] ^= buffer[i*8+2];
      if (len > 3)  data[1] ^= (buffer[i*8+3]<<8);
      if (len > 4)  data[2] ^= buffer[i*8+4];
      if (len > 5)  data[2] ^= (buffer[i*8+5]<<8);
      if (len > 6)  data[3] ^= buffer[i*8+6];
      if (len > 7)  data[3] ^= (buffer[i*8+7]<<8);

      //encrypt(data,key);
      data[0] = (vector[0] ^ data[0]);
      data[1] = (vector[1] ^ data[1]);
      data[2] = (vector[2] ^ data[2]);
      data[3] = (vector[3] ^ data[3]);

      for (k=0; k<4; k++)
	{
	  if ((k*2) < len)  /* make sure we don't buffer overrun */
	    {
	      a = data[k];
	      b = (word8)a;
	      buffer[i*8+0+k*2] = b;
	    }
	  if ((k*2+1) < len) 
	    {
	      a >>= 8;
	      b = (word8)a;
	      buffer[i*8+1+k*2] = b;
	    }
	}
    }
  return 0;
}

/******************************************************************************/
/* Decrypts a set of data of length len                                       */
/* pre: buffer                                                                */
/* post: buffer (modified)                                                    */
/* (C) Carl van Schaik (carl@leg.uct.ac.za)                                   */

int decrypt_idea(Idea_UserKey key, word8* buffer, word32 len)
{
  word32 i,k, magic=0x12345678;
  Idea_Data data,vector;
  Idea_Key iKey;
  word16 a;
  word8 b;

  Idea_ExpandUserKey(key, iKey);
  Idea_InvertKey(iKey,iKey);

   vector[0]=vector[1]=vector[2]=vector[3]=0;
   if (len >= 8)
    {
      word32 j = len/8;  /* 8 bytes = Idea_dataSize */
      for (i=0; i<j; i++)
	{
	  data[0] = buffer[i*8+0] + (buffer[i*8+1] << 8);
	  data[1] = buffer[i*8+2] + (buffer[i*8+3] << 8);
	  data[2] = buffer[i*8+4] + (buffer[i*8+5] << 8);
	  data[3] = buffer[i*8+6] + (buffer[i*8+7] << 8);

	  magic *=3;

	  Idea_Crypt(data, data, iKey);

	  vector[0] ^= data[1] ^ magic;
	  vector[1] ^= data[3] ^ magic;
	  vector[2] ^= data[0] ^ magic;
	  vector[3] ^= data[2] ^ magic;

	  for (k=0; k<4; k++)
	    {
	      a = data[k];
	      b = (word8)a;
	      buffer[i*8+0+k*2] = b;
	      a >>= 8;
	      b = (word8)a;
	      buffer[i*8+1+k*2] = b;
	    }
	}
    }

  i = len / 8; /* there may be some left over bits if the buffer is not divisible by 8 */
  len %= 8;    /* we need to take them into account as well */

  if (len > 0)
    {
      data[0]=data[1]=data[2]=data[3]=len*131071;

      data[0] ^= buffer[i*8+0];
      if (len > 1)  data[0] ^= (buffer[i*8+1]<<8);  /* make sure we don't buffer overrun */
      if (len > 2)  data[1] ^= buffer[i*8+2];
      if (len > 3)  data[1] ^= (buffer[i*8+3]<<8);
      if (len > 4)  data[2] ^= buffer[i*8+4];
      if (len > 5)  data[2] ^= (buffer[i*8+5]<<8);
      if (len > 6)  data[3] ^= buffer[i*8+6];
      if (len > 7)  data[3] ^= (buffer[i*8+7]<<8);

      //encrypt(data,key);
      data[0] = (vector[0] ^ data[0]);
      data[1] = (vector[1] ^ data[1]);
      data[2] = (vector[2] ^ data[2]);
      data[3] = (vector[3] ^ data[3]);

      for (k=0; k<4; k++)
	{
	  if ((k*2) < len)  /* make sure we don't buffer overrun */
	    {
	      a = data[k];
	      b = (word8)a;
	      buffer[i*8+0+k*2] = b;
	    }
	  if ((k*2+1) < len) 
	    {
	      a >>= 8;
	      b = (word8)a;
	      buffer[i*8+1+k*2] = b;
	    }
	}
    }
 return 0;
}


void make_idea_key(char* text_key, Idea_UserKey idea_user_key)
{
	unsigned int i = 0;

	idea_user_key[0] = 401;
	idea_user_key[1] = 302;
	idea_user_key[2] = 203;
	idea_user_key[3] = 104;

	while ((text_key[i] != '\0') && (i < 128))
	  {
	    idea_user_key[0] += (word16)((word32)text_key[(text_key[i]^i)%(i+1)]*i) << ((i*7) % 10);
	    idea_user_key[1] ^= (word16)((word32)text_key[(text_key[i]+i)%(i+1)]*i) << ((i*7) % 11);
	    idea_user_key[2] += (word16)((word32)text_key[(text_key[i]^i)%(i+1)]*i) << ((i*7) % 12);
	    idea_user_key[3] ^= (word16)((word32)text_key[(text_key[i]+i)%(i+1)]*i) << ((i*7) % 7);
	    i++;
	  }
}

