/* fsckvs3.c - Check a vs3fs (steganographic) filesystem. 
 * Copyright (C) 1998 Carl van Schaik (carl@leg.uct.ac.za)
 *                    Paul Smeddle (psmeddle@cs.uct.ac.za)
 * VS3FS - [v]an
 *         [S]chaik
 *         [S]meddle
 *         [S]teganographic [F]ile [S]ystem
 *
 *
 * %Begin-Header%
 * This file may be redistributed under the terms of the GNU Public
 * License.
 * %End-Header%
 */

/* Usage: fsckvs3 [options] <device>
 * 
 * fsckvs3 by Carl van Schaik
 * ALL code except where noted, by Carl van Schaik
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <termios.h>
#include <time.h>
#include <errno.h>
#include <mntent.h>
#include <malloc.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <linux/fs.h>
#include <linux/vs3_fs.h>

#include <crypt_funcs.h>
#include <rand.h>  /* Strong Random Number Generator */
#include <vs3lib.h>

char 	*program_name = "fsckvs3";
unsigned int size;

/* Initial filesystem Check */
static void inital_check()
{
	struct stat statbuf;

	if (fstat(device,&statbuf) < 0) {
	    printf ("Unable to get stats for %s\n",device_name);
	    die("Can't continue");
	  }

	if (!S_ISBLK(statbuf.st_mode))
	  {
	    printf("%s: only block devices are supported\n",device_name);
	    die("Can't continue");
	  }

	if (statbuf.st_rdev == 0x0300 || statbuf.st_rdev == 0x0340)
	  {
	    printf("Cannot make a filesystem on %s\n",device_name);
	    die("Can't continue");
	  }

	size = get_size();
	if (size == 0)
	{
	  printf("%s: device has zero lenght\n",device_name);
	  die("Can't continue");
	}
	else if (size < 64)
	{
	  printf("%s: device is too small\n",device_name);
	  die("Can't continue");
	}
	else
	  printf("Device %s is %iMB\n",device_name, size/1024 );

	check_mount();
}


static int ask_user(char * question)
{
   char k;

   printf("%s, [y/n]? ",question);
   fflush(stdout);

   k = getchar();

   if ((k == 'y') || (k == 'Y'))
      return (1);

   return(0);
}

/* Check superblocks */
static void check_supers()
{
	int off = VS3_BOOT_SECT_SIZE/VS3_BLK;
	unsigned int pos, good=1;
        int update = 0;

	read_block(off,(char*)&sb);
	if (vs3_verify(sb))
	  {
	     pos = off;
             if (!ask_user("Error in Superblock... try to fix"));
               die ("Cannot continue");
	     printf("Scanning for good superblock\n");
	     good = 0;
	     while ((pos < size) && (!good))
		{
		  read_block(pos,(char*)&sb);
		  if (vs3_verify(sb))
			good = 0;
		  else
			good = 1;
		  pos += VS3_GROUP;
		}
		if (good)
		  {
		    printf("Found good superblock at %i \nFixing...",pos);
		    update_super_block(sb);
		    printf("Fixed superblocks          \n");
		  }
		else
		  die("The disk is badly damaged, or you have the wrong password\n");
	  }

	if (sb.s_block_total != size)
	  {
	    printf("Disk size error... fixing\n");
	    sb.s_block_total = size;
            update = 1;
	  }
	if (sb.s_group_no != (size / VS3_GROUP + ((size % VS3_GROUP) > 2)))
	  {
	    printf("Superblock group number error... Fixing\n");
	    sb.s_group_no = (size / VS3_GROUP + ((size % VS3_GROUP) > 2));
            update = 1;
	  }
        if (update)
          update_super_block(sb);

}


/* This checks and tries to fix the root inode */
static void check_root()
{
	unsigned int pos, dpos,i,k,l;
	struct vs3_inode ino;
        char* block;
        struct vs3_dir_entry *de;
        char dir[256];
        int update = 0;

	printf("Checking root inode ...\n");
	pos = sb.s_root_ptr;
        if (!is_block_used(pos))
           if (ask_user("Root inode is not marked used... fix"))
              set_block_used(pos);

	read_data_block(pos,(char*)&ino);
	if (vs3_verify_i(ino))
	  {
	    printf("Root inode verification failed\n");
	    die("Fix will come later");
	    // pos = find_root();
	  }
 	
	if (ino.i_type != VS3_ROOT_INO)
	  {
	    printf("Root Inode is not marked as root\n");
            if (ino.i_type == VS3_DIRECTORY_INO)
               {
                  ino.i_type = VS3_ROOT_INO;
                  printf("Root was marked as directory... fixed\n");
               }
            else
               die ("Cannot continue");
	    // pos = find_root();
	  }
	if (ino.i_dir_entries < 2)
          {
             if (ask_user("Root directory too small... fix"))
                {
                   block = (char*)malloc(VS3_BLK);
                   if (block == NULL) die("Memory allocation error");
                   de = (struct vs3_dir_entry *)(block);

                   /* Create the '.' directory */
                   de->inode_p = pos;
                   de->rec_len = 4+2+2+2; /* 32bits + 16 + 16 + `.\0' */
                   de->name_len = 2;
                   strcpy(de->name,".");

                   /* Make ".." */
                   de = (struct vs3_dir_entry *)(block + de->rec_len);
                   de->inode_p = pos;
                   de->rec_len = 4+2+2+3; /* 32bits + 16 + 16 + `..\0' */
                   de->name_len = 3;
                   strcpy(de->name,"..");

                   /* Update the root inode and write the directory */
                   dpos = genrand() % sb.s_block_total;
                   while (is_block_used(dpos))
                      dpos = (++dpos) % sb.s_block_total;
                   set_block_used(dpos);

                   ino.i_direct[0] = dpos;
                   ino.i_dir_entries += 2;
                   write_data_block(pos, (char *)&ino);
                   write_data_block(dpos, (char*)&block);

                   free(block);
                }
          }

        block = read_dir(ino);
        if (block != NULL)
          {
             de = (struct vs3_dir_entry *)(block);
             for (i=0;i<2;i++)
               {
                  k = de->name_len;
                  for (l=0;l<k;l++)
                     dir[l] = de->name[l];
                  dir[l] = '\0';
                  if (i == 0)
                  {
                     if (strcmp(".",dir))
                       {
                          printf("First entry in root inode is not a \".\"\n");
                       }
                   }
                   else
                   {
                     if (strcmp("..",dir))
                       {
                          printf("Second entry in root inode is not a \"..\"\n");
                       }
                   }
                     
                  if (de->inode_p != pos)
                     {
                        printf("%s does not point to it's self\n",dir);
                        if (ask_user("Fix this"))
                           {
                              update = 1;
                              de->inode_p = pos;
                           }
                     }
                  de = (struct vs3_dir_entry *)(block + de->rec_len);
               }
          }

        if (update)
           update_dir(block,ino);
        free(block);
}

/* This checks and tries to fix the bad-block inode */
static void check_bad_block_inode()
{
   unsigned int pos;
   struct vs3_inode bb;
}


/* Recursive find inode */
static unsigned int find_inode(struct vs3_inode dir,
   struct vs3_inode target,int depth,unsigned int prev)
{
   unsigned int pos,i;
   struct vs3_dir_entry *de;
   struct vs3_inode ino;

   char *block;

   if (depth > 255)
      die("Directory too deep");

   block = read_dir(dir);
   if (block == NULL)
      die ("Not enough memory");
   if (block != NULL)
   {
      de = (struct vs3_dir_entry *)(block);
      de = (struct vs3_dir_entry *)((char*)de + de->rec_len);
      de = (struct vs3_dir_entry *)((char*)de + de->rec_len);

      for (i=2;i<dir.i_dir_entries;i++)
      {
         read_data_block(de->inode_p,(char*)&ino);
         if (vs3_verify_i(ino))
            die("Error reading inode");

         if (ino.i_direct[0]==target.i_direct[0])
            {
               free(block);
               return prev;
            }
         else
            {
               pos = find_inode(ino,target,depth+1,de->inode_p);
               if (pos)
                  {
                     free(block);
                     return pos;
                  }
            }
         de = (struct vs3_dir_entry *)(de + de->rec_len);
      }
   }
        
   free(block);
   return 0;
}

/* Finds the previous inode to the given one */
static unsigned int previous_dir(struct vs3_inode dir)
{
   unsigned int pos, found=0;
   struct vs3_inode ino;

   pos = sb.s_root_ptr;
   read_data_block(pos,(char*)&ino);
   if (vs3_verify_i(ino))
      die("Error reading root inode");

   found = find_inode(ino,dir,1,pos);
   if (found == 0)
      die ("Could not find parent");

   return found;
}

/* The recursive directory part of check_dir_struct*/
static void scan_dir(struct vs3_inode dir,char *path,int depth)
{
   unsigned int pos,dpos,i,k,l;
   char *block;
   char name[256],p[strlen(path)+256];
   struct vs3_dir_entry *de;
   struct vs3_inode ino;
   int update = 0;

   if (depth > 255)
      {
         printf("Directory too deep\n");
         return;
      }

   pos = sb.s_root_ptr;
   if (ino.i_dir_entries < 2)
      {
         /* This does not work yet */
         if (ask_user("Directory too small... fix"))
            {
               block = (char*)malloc(VS3_BLK);
               if (block == NULL) die("Memory allocation error");
               de = (struct vs3_dir_entry *)(block);

               /* Create the '.' directory */
               de->inode_p = 1; //dir;
               de->rec_len = 4+2+2+2; /* 32bits + 16 + 16 + `.\0' */
               de->name_len = 2;
               strcpy(de->name,".");

               /* Make ".." */
               de = (struct vs3_dir_entry *)(block + de->rec_len);
               de->inode_p = previous_dir(dir);
               de->rec_len = 4+2+2+3; /* 32bits + 16 + 16 + `..\0' */
               de->name_len = 3;
               strcpy(de->name,"..");

               /* Update the root inode and write the directory */
               dpos = genrand() % sb.s_block_total;
               while (is_block_used(dpos))
                  dpos = (++dpos) % sb.s_block_total;
               set_block_used(dpos);

               ino.i_direct[0] = dpos;
               ino.i_dir_entries += 2;
               write_data_block(pos, (char *)&ino);
               write_data_block(dpos, (char*)&block);

               free(block);
            }
      }

   block = read_dir(dir);
   if (block == NULL)
      die ("Not enough memory");
   if (block != NULL)
   {
      de = (struct vs3_dir_entry *)(block);
      de = (struct vs3_dir_entry *)((char*)de + de->rec_len);
      de = (struct vs3_dir_entry *)((char*)de + de->rec_len);

      for (i=2;i<dir.i_dir_entries;i++)
      {
         k = de->name_len;
         for (l=0;l<k;l++)
            name[l] = de->name[l];
         name[l] = '\0';
         
         printf("%s%s\r",path,name);
         fflush(stdout);

         if (!is_block_used(de->inode_p))
            {
               printf("%i : ",de->inode_p);
               if (ask_user("Inode is not marked used... fix"))
                  set_block_used(de->inode_p);
            }
         read_data_block(de->inode_p,(char*)&ino);
         if (vs3_verify_i(ino))
            die("Error reading inode");

         if ((ino.i_file_size/VS3_BLK-ino.i_blocks) != 0)
            {
               printf("Inode %i, size %i : ",de->inode_p,ino.i_file_size);
               if (ask_user("Invalid file size... try to fix"))
                  {
                     update = 1;
                     ino.i_file_size = ino.i_blocks*VS3_BLK;
                  }
            }
         if (ino.i_link_count == 0)
            {
               printf("Fixing link count for inode %i\n",de->inode_p);
               ino.i_link_count = 1;
               update = 1;
            }

         if (ino.i_type == VS3_DIRECTORY_INO)
            {
               strcpy(p,path);
               strcat(p,name);
               strcat(p,"\\");
               scan_dir(ino,p,depth+1);
            }
         
         de = (struct vs3_dir_entry *)(de + de->rec_len);
      }
   }
        
   if (update)
      update_dir(block,ino);
   free(block);
}

/* This checks and tries to fix the directory structure */
static void check_dir_struct()
{
   unsigned int pos;
   struct vs3_inode ino;

   printf("Checking directory structure ...\n");
   pos = sb.s_root_ptr;
   read_data_block(pos,(char*)&ino);
   if (vs3_verify_i(ino))
       die("Error reading root inode");

   scan_dir(ino,"",1);
}

/* Displays the programs options
 */
static void show_options()
{
	printf("USAGE:  fsckvs3 [options] <device>\n");
	printf("	Options : -i      : show fs information on completion\n");
	printf("		  -k      : display encryption keys\n");
}


/* This checks if all the options were gathered properly */
static int check_options()
{
	if (got_device != 1)
	  return(0);

	return(1);
}

/* Process command line arguments */
static void PRS(int argc, char *argv[])
{
	char k,l;

	printf("[0;1;37;33m");
	printf("fsckvs3 v0.0, Steganographic File System\n");
	printf("[0;1;37;00m");
	printf("(C) 1998 Paul Smeddle (psmeddle@cs.uct.ac.za)\n");
	printf("         Carl van Schaik (carl@leg.uct.ac.zan\n");

	argc--;
	argv++;

	if (argc == 0) {
		show_options();
		exit(1);
		}

	while (argc--)
	{
	  if (argv[0][0] == '-')
	    {
top:
	      k = (*(++argv[0])); 
	      k = argv[0][0];
	      l = 1;
	      switch(k)
		{
		  case 'i' :
			{
			  show_stats = 1;
			  break;
			}
		  case 'k' :
			{
			  wants_key  = 1;
			  break;
			}
		  default :
			{
			  printf("Invalid Option : %c\n",k);
			  show_options();
			  exit(1);
			}
		}
		if ((l) && (argv[0][1] != '\0'))
		  { goto top; }
	    }
	  else
	    {
	          device_name = *argv;
		  got_device = 1;
	     }
	  argv++;
	}

  if (!check_options())
    {
      show_options();
      exit(1);
    }
}


/* main procedure */
int main (int argc, char *argv[])
{
	PRS(argc, argv);  /* Process command line options */

	ask_for_key('n');    /* Ask the user for his Key */
	init_3way();      /* Initialise the 3-Way encryption cipher */

	device = open(device_name,O_RDWR|O_SYNC); /* Open read/write, with sync */
	if(device < 0) {
	  clean_up();
	  perror(device_name);
	  exit(errno);
	}

        /* Check if decive mounted etc */
	inital_check();

	check_supers();           /* Check the super blocks */
	check_root();             /* Check the root inode */
	check_bad_block_inode();  /* Check the bad-block inode */
        check_dir_struct();       /* Check the directory structure */

	if (show_stats)
	  get_fs_info();

	clean_up();
	close(device);
	printf("\nCompleted\n");
	return 0;
}
