/* vs3mount.c - Mount a vs3fs (steganographic) filesystem. 
 * Copyright (C) 1998 Carl van Schaik (carl@leg.uct.ac.za)
 *                    Paul Smeddle (psmeddle@cs.uct.ac.za)
 * VS3FS - [v]an
 *         [S]chaik
 *         [S]meddle
 *         [S]teganographic [F]ile [S]ystem
 *
 *
 * %Begin-Header%
 * This file may be redistributed under the terms of the GNU Public
 * License.
 * %End-Header%
 */
 

/* Usage: vs3mount [options] <device> <dir>
 * 
 * vs3mount by Carl van Schaik
 * ALL code except where noted, by Carl van Schaik
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <termios.h>
#include <time.h>
#include <errno.h>
#include <mntent.h>
#include <malloc.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mount.h>

#include <linux/vs3_fs.h>

#include <crypt_funcs.h>
#include <rand.h>  /* Strong Random Number Generator */
#include <vs3lib.h>

char 	*program_name = "vs3add";
char    *dir_name = NULL;

/* Command line options */
int got_dir = 0;


/* Displays the programs options
 */
static void show_options()
{
	printf("USAGE:  mkvs3fs [options] <device> <dir>\n");
	printf("	Options : -i      : show fs information on completion\n");
	printf("		  -k      : display encryption keys\n");
}


/* This checks if all the options were gathered properly */
static int check_options()
{
	if (got_device != 1)
	  return(0);
	if (got_dir != 1)
	  return(0);

	return(1);
}


static void do_mount()
{
    	FILE *mtab;
	struct mntent mp;

	read_block(VS3_BOOT_SECT_SIZE/VS3_BLK, (char *)&sb);
	if (vs3_verify(sb))
	  die("superblock encryption test failed");
        clean_up();
        close(device);

        if(!(mtab = setmntent("/etc/mtab", "a"))) {
            fprintf(stderr, "mount: open(%s) failed.\n", "/etc/mtab");
            return;
        }

        if (!mount (device_name,dir_name,"vs3fs",MS_MGC_VAL,text_key))
	   printf("Mount succeeded\n");
	else
	   die("Mount failed\n");

        if (dir_name[strlen(dir_name)-1]=='/')
           dir_name[strlen(dir_name)-1]='\0';
        mp.mnt_type="vs3fs";
        mp.mnt_fsname=device_name;
        mp.mnt_dir=dir_name;
        mp.mnt_opts="rw";
	mp.mnt_freq=0;
	mp.mnt_passno=0;
        addmntent(mtab, &mp);
        endmntent(mtab);
}


/* Process command line arguments */
static void PRS(int argc, char *argv[])
{
	char k,l;

	printf("[0;1;37;33m");
	printf("vs3mount v0.0, Steganographic File System\n");
	printf("[0;1;37;00m");
	printf("(C) 1998 Paul Smeddle (psmeddle@cs.uct.ac.za)\n");
	printf("         Carl van Schaik (carl@leg.uct.ac.za)\n");

	argc--;
	argv++;

	if (argc == 0) {
		show_options();
		exit(1);
		}

	while (argc--)
	{
	  if (argv[0][0] == '-')
	    {
top:
	      k = (*(++argv[0])); 
	      k = argv[0][0];
	      l = 1;
	      switch(k)
		{
		  case 'i' :
			{
			  show_stats = 1;
			  break;
			}
		  case 'k' :
			{
			  wants_key  = 1;
			  break;
			}
		  default :
			{
			  printf("Invalid Option : %c\n",k);
			  show_options();
			  exit(1);
			}
		}
		if ((l) && (argv[0][1] != '\0'))
		  { goto top; }
	    }
	  else
	    {
	      if (!got_device)
		{
	          device_name = *argv;
		  got_device = 1;
		}
	      else
		{
		  dir_name = *argv;
		  got_dir++;
		}
	     }
	  argv++;
	}

  if (!check_options())
    {
      show_options();
      exit(1);
    }
}


/* main procedure */
int main (int argc, char *argv[])
{
	struct stat statbuf;

	PRS(argc, argv);  /* Process command line options */

        if (stat("/proc/vs3",&statbuf) < 0)
          die ("vs3fs not found in kernel");
	if (!S_ISREG(statbuf.st_mode))
	  die ("vs3fs not found in kernel");

	ask_for_key('n');    /* Ask the user for his Key */
	init_3way();      /* Initialise the 3-Way encryption cipher */

	device = open(device_name,O_RDONLY|O_SYNC); /* Open read/write, with sync */
	if(device < 0) {
	  clean_up();
	  perror(device_name);
	  exit(errno);
	}

	/* Check to see if device is valid etc */
	if (check_device())
	{
	  clean_up();
	  close(device);
	  exit(0);
	}

 
	if (stat(dir_name,&statbuf) < 0) {
	  printf ("Unable to get stats for %s\n",dir_name);
	  clean_up();
	  close(device);
	  exit(0);
	}
	if (!S_ISDIR(statbuf.st_mode))
	  {
	    printf("%s is not a directory\n",dir_name);
	    clean_up();
	    close(device);
	    exit(0);
	  }

        if (show_stats)
          get_fs_info();

        do_mount();

	printf("\nCompleted\n");
	return 0;
}
