
// I retain copyright in this code but I encourage its free use provided
// that I don't carry any responsibility for the results. I am especially 
// happy to see it used in free and open source software. If you do use 
// it I would appreciate an acknowledgement of its origin in the code or
// the product that results and I would also appreciate knowing a liitle
// about the use to which it is being put.
//
// Dr B. R. Gladman <brg@gladman.uk.net> 20th December 2000.

// This is an implementation of the AES encryption algorithm (Rijndael)
// designed by Joan Daemen and Vincent Rijmen. This version is designed
// for a fixed block length of 128 bits (Ncol = 4) and can run with either
// big or little endian internal byte order (see aes.h). It implements
// the three key lengths of 128, 192 and 256 bits, the length being set
// by Nk in the key set interface where Nk = 4, 6 or 8 resepectively.
// Note that the validity of input parameters is NOT checked in this
// implementation.
//
// To compile AES (Rijndael) in C
//   a. Exclude (remove or comment out) the AES_IN_CPP define in aes.h
//   b. Exclude the AES_DLL define in aes.h
//   c. Rename aes.cpp to aes.c (if necessary) and compile as a C file
//
// To compile AES (Rijndael) in C++
//   a. Include the AES_IN_CPP define in aes.h
//   b. Exclude the AES_DLL define in aes.h
//   c. Rename aes.c to aes.cpp (if necessary) and compile as a C++ file
//
// To compile AES (Rijndael) in C as a Dynamic Link Library
//   a. Exclude the AES_IN_CPP define in aes.h
//   b. Include the AES_DLL define in aes.h
//   c. Rename aes.cpp to aes.c (if necessary) and compile as a C file

#define AES_BIG_ENDIAN      1   // do not change
#define AES_LITTLE_ENDIAN   2   // do not change

// set AES_DLL if AES (Rijndael) is to be compiled to a DLL

//#define AES_DLL
  
// set AES_IN_CPP if AES (Rijndael) is to be compiled in C++ rather than C
  
//#define   AES_IN_CPP

// set INTERNAL_BYTE_ORDER to one of the above constants to set the
// internal byte order (the order used within the algorithm code)

#include <asm/byteorder.h>

#ifdef __LITTLE_ENDIAN
#define INTERNAL_BYTE_ORDER AES_LITTLE_ENDIAN
#else
#define INTERNAL_BYTE_ORDER AES_BIG_ENDIAN
#endif

// set EXTERNAL_BYTE_ORDER to one of the above constants to set the byte
// order used at the external interfaces for the input, output and key
// byte arrays.

#define EXTERNAL_BYTE_ORDER AES_LITTLE_ENDIAN

typedef unsigned char   byte;   // must be an 8-bit storage unit
typedef unsigned long   word;   // must be a 32-bit storage unit



// rot1 changes [b0,b1,b2,b3] to [b3,b0,b1,b2]
// rot2 changes [b0,b1,b2,b3] to [b2,b3,b0,b1]
// rot3 changes [b0,b1,b2,b3] to [b1,b2,b3,b0]

#if(INTERNAL_BYTE_ORDER == AES_LITTLE_ENDIAN)

// Circular byte rotates of 32-bit words

#define rot1(x) (((x) <<  8) | ((x) >> 24))
#define rot2(x) (((x) << 16) | ((x) >> 16))
#define rot3(x) (((x) << 24) | ((x) >>  8))

// Extract bytes from a 32-bit words

#define byte0(x)    ((byte) (x))
#define byte1(x)    ((byte)((x) >>  8))
#define byte2(x)    ((byte)((x) >> 16))
#define byte3(x)    ((byte)((x) >> 24))

#define bytes2word(b0, b1, b2, b3)  \
    ((word)(b3) << 24 | (word)(b2) << 16 | (word)(b1) << 8 | (b0))

// Invert byte order in a 32 bit variable

#define byte_swap(x)    (rot1(x) & 0x00ff00ff | rot3(x) & 0xff00ff00)

#else

// Circular byte rotates of 32-bit words

#define rot3(x) (((x) <<  8) | ((x) >> 24))
#define rot2(x) (((x) << 16) | ((x) >> 16))
#define rot1(x) (((x) << 24) | ((x) >>  8))

// Extract bytes from a 32-bit words

#define byte3(x)    ((byte) (x))
#define byte2(x)    ((byte)((x) >>  8))
#define byte1(x)    ((byte)((x) >> 16))
#define byte0(x)    ((byte)((x) >> 24))

#define bytes2word(b0, b1, b2, b3)  \
    ((word)(b0) << 24 | (word)(b1) << 16 | (word)(b2) << 8 | (b3))

// Invert byte order in a 32 bit variable

#define byte_swap(x)    (rot3(x) & 0x00ff00ff | rot1(x) & 0xff00ff00)

#endif

#if(INTERNAL_BYTE_ORDER == EXTERNAL_BYTE_ORDER)

#define word_in(x)      *(word*)(x)
#define word_out(x,v)   *(word*)(x) = (v)

#else

#define word_in(x)      byte_swap(*(word*)(x))
#define word_out(x,v)   *(word*)(x) = byte_swap(v)

#endif

enum aes_const  {   Nrow =  4,  // the number of rows in the cipher state
                    Ncol =  4,  // the number of columns in the cipher state
                    Mcol =  8,  // maximum number of columns in the state
                    Mkbl = 15   // maximum number of key schedule blocks
                };

enum aes_key    {   enc  =  1,  // set if encryption is needed
                    dec  =  2,  // set if decryption is needed
                    both =  3,  // set if both are needed
                };


typedef struct          // note that Nkey = key_len_in_bits / 32
{
    word            Nkey;       // the number of words in the key input block
    word            Nrnd;       // the number of cipher rounds
    word            Sr[Nrow];   // the row shift counts (note that Sr[0] is not used)
    word            e_key[64];  // the encryption key schedule (128 bit block only)
    word            d_key[64];  // the decryption key schedule (128 bit block only)
    enum aes_key    mode;       // encrypt, decrypt or both
} aes;

// definitions needed to allow C, C++ and DLL (dynamic) calls to be put in 
// the same form

#define aes_ref             aes*
#define f_dat(a,b)          (a->b)


