/*
 *  linux/fs/stegfs/level.c
 *
 * Copyright (C) 1999-2001
 * Andrew McDonald (andrew@mcdonald.org.uk)
 */


#include "fs.h"
#include "stegfs_fs.h"

#include <asm/uaccess.h>
#include <asm/system.h>

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/locks.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/random.h>

#if defined(CONFIG_CIPHERS) || defined(CONFIG_CIPHERS_MODULE) || \
    defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
#include <linux/crypto.h>
#endif
#if !defined(CONFIG_DIGEST) && !defined(CONFIG_DIGEST_MODULE)
#include "sha1.h"
#endif

/* we can't use the default_llseek in fs/read_write.c since it
   isn't exported */
static loff_t do_llseek(struct file *file, loff_t offset, int origin)
{
        long long retval;

	if (file->f_op && file->f_op->llseek)
		return (file->f_op->llseek(file, offset, origin));

        switch (origin) {
                case 2:
                        offset += file->f_dentry->d_inode->i_size;
                        break;
                case 1:
                        offset += file->f_pos;
        }
        retval = -EINVAL;
        if (offset >= 0) {
                if (offset != file->f_pos) {
                        file->f_pos = offset;
                        file->f_reada = 0;
                        file->f_version = ++global_event;
                }
                retval = offset;
        }
        return retval;
}

int stegfs_open_level(struct super_block * sb, int nlevel, int tlevel,
		      char * pass)
{

	int i,j;
	struct file *file;
	struct inode * inode;
	char slkey[STEGFS_KEY_BYTES];
	int attempts, block;
	int gotroot = 0;
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
	struct digest_implementation *di;
	struct digest_context dx;
#else
	SHA1_CTX sha1ctx;
#endif
	
	unsigned int digest[5];
	unsigned int keystr[6];
	unsigned int maxblock;
	unsigned int scale;
	struct stegfs_btable ibtab;
	char sbuf[15];
	struct dentry * rdentry;
	mm_segment_t old_fs;
	int result;
	struct stegfs_btable_head btabh;

#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
	di = sb->u.stegfs_sb.s_x->s_di;
#endif

	stegfs_debug("nlevel: %d pass: %s\n", nlevel, pass);
	if (/*nlevel-1 != STEGFS_SB(sb)->s_seclevel ||*/
	    !(STEGFS_SB(sb)->s_x->s_btabfname))
		return 0;

	file = &(STEGFS_SB(sb)->s_x->s_btabfile);


	if (!sb->u.stegfs_sb.s_x->s_isstegfs) {
		/* open btabfile before opening first hidden level */
	        STEGFS_SB(sb)->s_x->s_btabfiled = open_namei(STEGFS_SB(sb)->
							s_x->s_btabfname,
							0, 0);
		stegfs_debug("btabfname: %s\n", STEGFS_SB(sb)->s_x->s_btabfname);
		i = PTR_ERR(sb->u.stegfs_sb.s_x->s_btabfiled);
		if (IS_ERR(sb->u.stegfs_sb.s_x->s_btabfiled)) {
			sb->u.stegfs_sb.s_x->s_btabfiled = NULL;
			stegfs_debug("failed to open btable file: error %d\n", i);
			return 0;
		}

                inode = STEGFS_SB(sb)->s_x->s_btabfiled->d_inode;

                if (!inode->i_op || !inode->i_op->default_file_ops) {
                        stegfs_debug("i_op or default_file_ops missing\n");
                        goto btabfail2;
                }

                if (init_private_file(file, sb->u.stegfs_sb.s_x->s_btabfiled, 1)) {
                        stegfs_debug("failed in init_private_file\n");
                        goto btabfail2;
                }
                if (!file->f_op->read) {
                        stegfs_debug("read function missing\n");
                        goto btabfail;
                }
		if (do_llseek(file,0,0) != 0) {
			stegfs_debug("llseek failed\n");
			goto btabfail;
		}

        }

	if((STEGFS_SB(sb)->s_x->s_btabfiled) == NULL)
		return 0;

	/* check btab file is ok */
	result = do_llseek(file, 0, 0);
	if (result != 0)
		return 0;
	old_fs = get_fs();
	set_fs(get_ds());

	result = file->f_op->read(file,
				  (char *)&btabh,
				  sizeof(struct stegfs_btable_head),
				  &(file->f_pos));
	set_fs(old_fs);

	if (btabh.magic == STEGFS_BTAB_MAGIC) {
		if(btabh.version > 1 && btabh.version < 3) {
			/* we can handle it */
			sb->u.stegfs_sb.s_x->s_btabver = btabh.version;
		}
		else {
			stegfs_debug("Bad btab version\n");
			if (file->f_op->release)
				file->f_op->release(STEGFS_SB(sb)->s_x->s_btabfiled->d_inode,file);

			dput(STEGFS_SB(sb)->s_x->s_btabfiled);
			STEGFS_SB(sb)->s_x->s_btabfiled = NULL;
			return 0;
		}
	}
	else {
		/* see if it is old style */
		if (STEGFS_SB(sb)->s_x->s_btabfiled->d_inode->i_size ==
		    sizeof(struct stegfs_btable)*STEGFS_SB(sb)->s_es->s_blocks_count + 
		    STEGFS_KEY_BYTES*120) {
			sb->u.stegfs_sb.s_x->s_btabver = 1;
		}
		else {
			stegfs_debug("Bad btab version\n");
			if (file->f_op->release)
				file->f_op->release(STEGFS_SB(sb)->s_x->s_btabfiled->d_inode,file);

			dput(STEGFS_SB(sb)->s_x->s_btabfiled);
			STEGFS_SB(sb)->s_x->s_btabfiled = NULL;
			return 0;
		}
	}

	if (sb->u.stegfs_sb.s_x->s_btabver == 1 &&
	    nlevel > tlevel)
		return 0;
	stegfs_debug("nlevel: %d tlevel: %d\n", nlevel, tlevel);
	stegfs_decrypt_slkey(sb, nlevel, tlevel, pass, slkey);
	sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1] = kmalloc(STEGFS_KEY_BYTES, GFP_KERNEL);
	memcpy(sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1], slkey, STEGFS_KEY_BYTES);

#ifdef STEGFS_DEBUG
	{
		unsigned int *slkx;

		stegfs_debug("btabver: %d\n", STEGFS_SB(sb)->s_x->s_btabver);
		slkx = (unsigned int *)slkey;
		stegfs_debug("nlevel: %u key: %08x %08x %08x %08x\n",
			     nlevel, slkx[0], slkx[1], slkx[2], slkx[3]);
	}
#endif

	/*
	 * open the block table file and check that we
	 * can find a copy of the root inode for this level
	 */

	j = EXT2_ROOT_INO|STEGFS_INO_MASK|(nlevel<<STEGFS_LVL_SHIFT);
	maxblock = STEGFS_SB(sb)->s_es->s_blocks_count - 1;
	scale = ((unsigned int)-1) / maxblock;

	stegfs_debug("looking for root inode\n");

	memcpy((unsigned char *)keystr,
	       sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1], 16);
	keystr[4] = j;
	keystr[5] = 0;
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
	di = sb->u.stegfs_sb.s_x->s_di;
	dx.di = di;
	dx.digest_info = (u32 *)kmalloc(di->working_size, GFP_KERNEL);
	di->open(&dx);
	di->update(&dx, (unsigned char *)keystr, 24);
	di->close(&dx, (unsigned char *)digest);
	memset(dx.digest_info, 0, di->working_size);
	kfree(dx.digest_info);
#else
	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
	SHA1Final((unsigned char *)digest, &sha1ctx);
#endif
	attempts=0;
	block = -1;
	while (1) {

		for (i=0; i<5; i++) {
			block = digest[i] / scale;
#if 0
/*#ifdef STEGFS_DEBUG*/
			if (attempts < 50) {
				stegfs_debug("trying: %u\n", block);
			}
			else if(attempts == 50) {
				stegfs_debug("trying: more (not logging)\n");
			}
#endif
			if (!(block > maxblock || block < 1)) {
				/* only need to check if block
				   is used by _this_ inode number */

				stegfs_get_btab(sb, block, &ibtab);
				stegfs_decrypt_btab(sb, nlevel,
						    block, &ibtab);
				if(ibtab.magic1 == 0 && ibtab.magic2 == 1 &&
				   ibtab.ino == j) {
					gotroot = 1;
					break;
				}
			}

			attempts++;
		}

		if(gotroot || attempts > STEGFS_INO_BATT)
			break;

#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
		dx.digest_info = (u32 *)kmalloc(di->working_size, GFP_KERNEL);
		di->open(&dx);
		keystr[5]++;
		di->update(&dx, (unsigned char *)keystr, 24);
		di->close(&dx, (unsigned char *)digest);
		memset(dx.digest_info, 0, di->working_size);
		kfree(dx.digest_info);
#else
		SHA1Init(&sha1ctx);
		keystr[5]++;
		SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
		SHA1Final((unsigned char *)digest, &sha1ctx);
#endif
	}

/*
	sb->u.stegfs_sb.s_seclevel = nlevel;
*/
	if (!gotroot) {
		stegfs_debug("root not found\n");

		memset(sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1], 0, STEGFS_KEY_BYTES);
		kfree(sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1]);
		sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1] = NULL;

		if (!sb->u.stegfs_sb.s_x->s_isstegfs && STEGFS_SB(sb)->s_x->s_btabfiled) {
			file = &sb->u.stegfs_sb.s_x->s_btabfile;
			if (file->f_op->release)
				file->f_op->release(STEGFS_SB(sb)->s_x->s_btabfiled->d_inode,file);

			dput(STEGFS_SB(sb)->s_x->s_btabfiled);
			STEGFS_SB(sb)->s_x->s_btabfiled = NULL;
			stegfs_debug("closed btabfile\n");
		}

		return 0;
	}

	/* add some dentries for the 'directories' we've just added */
	if (!sb->s_root) {
		stegfs_debug("Umm, not got s_root dentry\n");
		return 1;
	}

	i = sb->u.stegfs_sb.s_x->s_isstegfs;
	sb->u.stegfs_sb.s_x->s_isstegfs = 1;
	if(!i) {
		rdentry = lookup_dentry("stegfs", dget(sb->s_root), 0);
		if (IS_ERR(rdentry)) {
			stegfs_debug("Not got rdentry\n");
			/* never mind don't bother :( */
			return 1;
		}
#if 0
		if(!rdentry->d_inode) {
			/* make negative dentry happy */
			inode = iget(sb, STEGFS_HID_ROOT_INO);
			if(inode) {
				stegfs_debug("adding stegfs dentry\n");
				d_add(rdentry, inode);
			}
		}
#endif
		d_invalidate(rdentry);
		dput(rdentry);
	}

	sprintf(sbuf, "stegfs/%u", nlevel);
	rdentry = lookup_dentry(sbuf, dget(sb->s_root), 0);
	if (IS_ERR(rdentry)) {
		stegfs_debug("Not got rdentry\n");
		/* never mind don't bother :( */
		return 1;
	}
#if 0
	if (!rdentry->d_inode) {
		inode = iget(sb, EXT2_ROOT_INO | STEGFS_INO_MASK |
			          (nlevel)<<(STEGFS_LVL_SHIFT));
		if(inode) {
			stegfs_debug("adding level %d dentry\n", nlevel);
			d_add(rdentry, inode);
		}
	}
#endif
	d_invalidate(rdentry);
	dput(rdentry);

	return 1;

btabfail:
	if (file->f_op->release)
		file->f_op->release(inode,file);
btabfail2:
	if (sb->u.stegfs_sb.s_x->s_btabfiled)
		dput(sb->u.stegfs_sb.s_x->s_btabfiled);
	sb->u.stegfs_sb.s_x->s_btabfiled = NULL;
	return 0;
}


int stegfs_close_level(struct super_block * sb, int nlevel)
{
	struct file * file;
	struct dentry *rdentry;
	char sbuf[15];
	int i,j;
/*
	if (nlevel != STEGFS_SB(sb)->s_seclevel)
		return -EPERM;
*/
	stegfs_debug("level: %d\n", nlevel);

	if (!STEGFS_SB(sb)->s_x->s_slkeys[nlevel-1])
		return 0;

	/* FIXME: should be be holding the superblock lock? probably */
	fsync_dev(sb->s_dev);

	j = 0;
	for (i=0; i<(STEGFS_MAX_LEVELS-1); i++) {
		if (i+1 != nlevel)
			j |= (STEGFS_SB(sb)->s_x->s_slkeys[i] != NULL);
	}

	/* if not umounting, we need to invalidate the dentries here */
	if (sb->s_root) {
		if(!j)
			sprintf(sbuf, "stegfs");
		else
			sprintf(sbuf, "stegfs/%u", nlevel);

		rdentry = lookup_dentry(sbuf, dget(sb->s_root), 0);
		if (IS_ERR(rdentry)) {
			stegfs_debug("Not got rdentry\n");
			goto fail_killdent;
		}
		else if (!rdentry->d_inode) {
			stegfs_debug("No inode with rdentry\n");
			dput(rdentry);
			goto fail_killdent;
		}

		stegfs_debug("is_root_busy: %d\n", is_root_busy(rdentry));
		if (is_root_busy(rdentry)) {
			dput(rdentry);
			return -EBUSY;
		}

		shrink_dcache_parent(rdentry);
		d_delete(rdentry);
		dput(rdentry);
		/* d_drop(rdentry); */
	}

fail_killdent:

/*
	sb->u.stegfs_sb.s_seclevel = nlevel - 1;
*/
	memset(sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1], 0, STEGFS_KEY_BYTES);
	kfree(sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1]);
	sb->u.stegfs_sb.s_x->s_slkeys[nlevel-1] = NULL;
	sb->u.stegfs_sb.s_x->s_isstegfs = 0;
	for (i=0; i<(STEGFS_MAX_LEVELS-1); i++) {
		if (i+1 != nlevel)
			sb->u.stegfs_sb.s_x->s_isstegfs |=
				(STEGFS_SB(sb)->s_x->s_slkeys[i] != NULL);
	}

	if (!sb->u.stegfs_sb.s_x->s_isstegfs && STEGFS_SB(sb)->s_x->s_btabfiled) {
		file = &sb->u.stegfs_sb.s_x->s_btabfile;
		if (file->f_op->release)
			file->f_op->release(STEGFS_SB(sb)->s_x->s_btabfiled->d_inode,file);

		dput(STEGFS_SB(sb)->s_x->s_btabfiled);
		STEGFS_SB(sb)->s_x->s_btabfiled = NULL;
		stegfs_debug("closed btabfile\n");
	}

	return 1;
}


int stegfs_get_btab(struct super_block * sb, u32 block,
		    struct stegfs_btable * btab)
{
	mm_segment_t old_fs;
	struct file * bfile;
	int result;
	loff_t pos;

	bfile = &(STEGFS_SB(sb)->s_x->s_btabfile);
	pos = block * sizeof(struct stegfs_btable);
	if (sb->u.stegfs_sb.s_x->s_btabver == 2)
		pos += sizeof(struct stegfs_btable_head);
	result = do_llseek(bfile, pos, 0);
	if (result != pos)
		return 0;
	old_fs = get_fs();
	set_fs(get_ds());
	
	result = bfile->f_op->read(bfile,
				   (char *)btab,
				   sizeof(struct stegfs_btable),
				   &(bfile->f_pos));
	set_fs(old_fs);

	return 1;

}

int stegfs_put_btab(struct super_block * sb, u32 block,
		    struct stegfs_btable * btab)
{
	struct file * bfile;
	int result;
	mm_segment_t old_fs;
	loff_t pos = block * sizeof(struct stegfs_btable);

	if (sb->u.stegfs_sb.s_x->s_btabver == 2)
		pos += sizeof(struct stegfs_btable_head);

	bfile = &(STEGFS_SB(sb)->s_x->s_btabfile);
	result = do_llseek(bfile, pos, 0);
	if (result != pos)
		return 0;

	old_fs = get_fs();
	set_fs(get_ds());
                                        
	result = bfile->f_op->write(bfile, (char *)btab,
				    sizeof(struct stegfs_btable),
				    &(bfile->f_pos));
	set_fs(old_fs);

	return 1;
}


unsigned int stegfs_chksum(unsigned char * data, int size)
{
	unsigned char *datax;

	datax = data + size - sizeof(unsigned int);

	return *datax;
}

void stegfs_decrypt_slkey(struct super_block *sb, int slevel, int topslevel,
			  char *passwd, char *slkey)
{

	mm_segment_t old_fs;
	struct file * bfile;
	int result;
	loff_t pos;
	char slkey2[STEGFS_KEY_BYTES];
	unsigned int digest[5];
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
	struct digest_implementation *di;
	struct digest_context dx;
#else
	SHA1_CTX sha1ctx;
#endif
#if defined(CONFIG_CIPHERS) || defined(CONFIG_CIPHERS_MODULE)
	struct cipher_implementation *ci;
	struct cipher_context *cx;
#else
	char *keys;
#endif

	bfile = &(STEGFS_SB(sb)->s_x->s_btabfile);
	if (sb->u.stegfs_sb.s_x->s_btabver == 1) {
		pos = STEGFS_SB(sb)->s_es->s_blocks_count
			* sizeof(struct stegfs_btable);
		pos += 0.5*topslevel*(topslevel-1)*STEGFS_KEY_BYTES;
		pos += (slevel-1)*STEGFS_KEY_BYTES;
	}
	else if (sb->u.stegfs_sb.s_x->s_btabver == 2) {
		pos = sizeof(struct stegfs_btable_head) +
			STEGFS_SB(sb)->s_es->s_blocks_count
			* sizeof(struct stegfs_btable);
		pos += (topslevel-1)*(STEGFS_MAX_LEVELS-1)*STEGFS_KEY_BYTES;
		pos += (slevel-1)*STEGFS_KEY_BYTES;
	}
	else {
		stegfs_debug("Unknown btab version %u\n", 
			     sb->u.stegfs_sb.s_x->s_btabver);
		pos = 0;
	}
	result = do_llseek(bfile, pos, 0);
	if (result != pos)
		return;

	old_fs = get_fs();
	set_fs(get_ds());
	
	result = bfile->f_op->read(bfile, slkey2, STEGFS_KEY_BYTES,
				   &(bfile->f_pos));
	set_fs(old_fs);

#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
	di = STEGFS_SB(sb)->s_x->s_di;
	dx.di = di;
	dx.digest_info = (u32 *)kmalloc(di->working_size, GFP_KERNEL);
	di->open(&dx);
	di->update(&dx, passwd, strlen(passwd));
	di->close(&dx, (unsigned char *)digest);
	memset(dx.digest_info, 0, di->working_size);
	kfree(dx.digest_info);
#else
	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, passwd, strlen(passwd));
	SHA1Final((unsigned char *)digest, &sha1ctx);
#endif

#if defined(CONFIG_CIPHERS) || defined(CONFIG_CIPHERS_MODULE)
	cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
					       GFP_KERNEL);
	memset(cx, 0, sizeof(struct cipher_context));
	ci = STEGFS_SB(sb)->s_x->s_ci;
	cx->ci = ci;
	cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
	ci->set_key(cx, (unsigned char*)digest, STEGFS_KEY_BYTES);
	ci->decrypt(cx, slkey2, slkey, STEGFS_KEY_BYTES);
	memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
	kfree(cx->keyinfo);
	memset(cx, 0, sizeof(*cx));
	kfree(cx);
        memset(digest, 0, 5 * sizeof(unsigned int));
#else
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key((unsigned char *)digest, 128);
	STEGFS_SB(sb)->s_x->s_cipher->decrypt(keys, slkey2, slkey);
	memset(digest, 0, 5 * sizeof(unsigned int));
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);
#endif

	return;
}

char *stegfs_make_key(struct super_block *sb, int slevel,
		      unsigned int blocknum)
{
	char *key;
	unsigned int *uikey;
	key = kmalloc(STEGFS_KEY_BYTES, GFP_KERNEL);
	memcpy(key, sb->u.stegfs_sb.s_x->s_slkeys[slevel-1], STEGFS_KEY_BYTES);
	uikey = (unsigned int *)key;
	uikey[3] = uikey[3] ^ blocknum;
	return key;
}

#if defined(CONFIG_CIPHERS) || defined(CONFIG_CIPHERS_MODULE)
void stegfs_encrypt_btab(struct super_block *sb, int slevel, unsigned int bnum,
                         struct stegfs_btable *btab)
{
        struct cipher_implementation *ci;
        struct cipher_context *cx;
        struct stegfs_btable btab2;
        char *key;

        key = stegfs_make_key(sb, slevel, bnum);
        memcpy(&btab2, btab, sizeof(struct stegfs_btable));

        cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
                                               GFP_KERNEL);
        memset(cx, 0, sizeof(struct cipher_context));
        ci = STEGFS_SB(sb)->s_x->s_ci;
        cx->ci = ci;
        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
        ci->set_key(cx, key, STEGFS_KEY_BYTES);
        ci->encrypt(cx, (u8 *)&btab2, (u8 *)btab,
                    sizeof(struct stegfs_btable));
        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
        kfree(cx->keyinfo);
        memset(cx, 0, sizeof(*cx));
        kfree(cx);
        memset(key, 0, STEGFS_KEY_BYTES);
        kfree(key);

        return;
}

void stegfs_encrypt_btab2(struct super_block *sb, int slevel, unsigned int bnum,
                         struct stegfs_btable *btab, struct stegfs_btable *btab2)
{
        struct cipher_implementation *ci;
        struct cipher_context *cx;
        char *key;

        key = stegfs_make_key(sb, slevel, bnum);

        cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
                                               GFP_KERNEL);
        memset(cx, 0, sizeof(struct cipher_context));
        ci = STEGFS_SB(sb)->s_x->s_ci;
        cx->ci = ci;
        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
        ci->set_key(cx, key, STEGFS_KEY_BYTES);
        ci->encrypt(cx, (u8 *)btab, (u8 *)btab2,
                    sizeof(struct stegfs_btable));
        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
        kfree(cx->keyinfo);
        memset(cx, 0, sizeof(*cx));
        kfree(cx);
        memset(key, 0, STEGFS_KEY_BYTES);
        kfree(key);

        return;
}

void stegfs_decrypt_btab(struct super_block *sb, int slevel, unsigned int bnum,
                         struct stegfs_btable *btab)
{
        struct cipher_implementation *ci;
        struct cipher_context *cx;
        struct stegfs_btable btab2;
        char *key;

        memcpy(&btab2, btab, sizeof(struct stegfs_btable));
        key = stegfs_make_key(sb, slevel, bnum);

        cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
                                               GFP_KERNEL);
        memset(cx, 0, sizeof(struct cipher_context));
        ci = STEGFS_SB(sb)->s_x->s_ci;
        cx->ci = ci;
        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
        ci->set_key(cx, key, STEGFS_KEY_BYTES);
        ci->decrypt(cx, (u8 *)&btab2, (u8 *)btab,
                    sizeof(struct stegfs_btable));
        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
        kfree(cx->keyinfo);
        memset(cx, 0, sizeof(*cx));
        kfree(cx);
        memset(key, 0, STEGFS_KEY_BYTES);
        kfree(key);

        return;
}


void stegfs_decrypt_btab2(struct super_block *sb, int slevel,
                          unsigned int bnum,
                          struct stegfs_btable *btab,
                          struct stegfs_btable *btab2)
{
        struct cipher_implementation *ci;
        struct cipher_context *cx;
        char *key;

        key = stegfs_make_key(sb, slevel, bnum);

        cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
                                               GFP_KERNEL);
        memset(cx, 0, sizeof(struct cipher_context));
        ci = STEGFS_SB(sb)->s_x->s_ci;
        cx->ci = ci;
        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
        ci->set_key(cx, key, STEGFS_KEY_BYTES);
        ci->decrypt(cx, (u8 *)btab, (u8 *)btab2,
                    sizeof(struct stegfs_btable));
        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
        kfree(cx->keyinfo);
        memset(cx, 0, sizeof(*cx));
        kfree(cx);
        memset(key, 0, STEGFS_KEY_BYTES);
        kfree(key);

        return;
}


void stegfs_encrypt_cbc(struct super_block *sb, int slevel, unsigned int bnum,
                        char *data, int size, unsigned short *iv)
{
        struct cipher_implementation *ci;
        struct cipher_context *cx;
        char *tmpcopy;
        char *key;

        key = stegfs_make_key(sb, slevel, bnum);
        tmpcopy = (char *)kmalloc(size, GFP_KERNEL);
        memcpy(tmpcopy, data, size);

        cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
                                               GFP_KERNEL);
        memset(cx, 0, sizeof(struct cipher_context));
        ci = STEGFS_SB(sb)->s_x->s_ci;
        cx->ci = ci;
        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
        ci->set_key(cx, key, STEGFS_KEY_BYTES);
#ifdef STEGFS_NOIV
        *iv = 0;
#else
        /* get_random_bytes(iv,2); */
        *iv = *iv + 1;
#endif
        cx->iv[0] = *iv;
        ci->encrypt(cx, tmpcopy, data, size);
        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
        kfree(cx->keyinfo);
        memset(cx, 0, sizeof(*cx));
        kfree(cx);
        memset(key, 0, STEGFS_KEY_BYTES);
        kfree(key);
        kfree(tmpcopy);
        return;
}

void stegfs_encrypt_cbc2(struct super_block *sb, int slevel, unsigned int bnum,
                         char *data, char *data2, int size, unsigned short *iv)
{
        struct cipher_implementation *ci;
        struct cipher_context *cx;
        char *key;

        key = stegfs_make_key(sb, slevel, bnum);

        cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
                                               GFP_KERNEL);
        memset(cx, 0, sizeof(struct cipher_context));
        ci = STEGFS_SB(sb)->s_x->s_ci;
        cx->ci = ci;
        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
        ci->set_key(cx, key, STEGFS_KEY_BYTES);
#ifdef STEGFS_NOIV
        *iv = 0;
#else
        /* get_random_bytes(iv,2); */
        *iv = *iv + 1;
#endif
        cx->iv[0] = *iv;
        ci->encrypt(cx, data, data2, size);
        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
        kfree(cx->keyinfo);
        kfree(cx);
        memset(key, 0, STEGFS_KEY_BYTES);
        kfree(key);
        return;
}

void stegfs_decrypt_cbc(struct super_block *sb, int slevel, unsigned int bnum,
                        char *data, int size, unsigned short iv)
{
        struct cipher_implementation *ci;
        struct cipher_context *cx;
        char *tmpcopy;
        char *key;

        key = stegfs_make_key(sb, slevel, bnum);
        tmpcopy = (char *)kmalloc(size, GFP_KERNEL);
        memcpy(tmpcopy, data, size);

        cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
                                               GFP_KERNEL);
        memset(cx, 0, sizeof(struct cipher_context));
        ci = STEGFS_SB(sb)->s_x->s_ci;
        cx->ci = ci;
        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
        ci->set_key(cx, key, STEGFS_KEY_BYTES);
        cx->iv[0] = iv;
        ci->decrypt(cx, tmpcopy, data, size);
        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
        kfree(cx->keyinfo);
        memset(cx, 0, sizeof(*cx));
        kfree(cx);
        memset(key, 0, STEGFS_KEY_BYTES);
        kfree(key);
        return;
}

void stegfs_decrypt_cbc2(struct super_block *sb, int slevel, unsigned int bnum,
                         char *data, char *data2, int size, unsigned short iv)
{
        struct cipher_implementation *ci;
        struct cipher_context *cx;
        char *key;

        key = stegfs_make_key(sb, slevel, bnum);

        cx = (struct cipher_context *) kmalloc(sizeof(struct cipher_context),
                                               GFP_KERNEL);
        memset(cx, 0, sizeof(struct cipher_context));
        ci = STEGFS_SB(sb)->s_x->s_ci;
        cx->ci = ci;
        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size, GFP_KERNEL);
        ci->set_key(cx, key, STEGFS_KEY_BYTES);
        cx->iv[0] = iv;
        ci->decrypt(cx, data, data2, size);
        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
        kfree(cx->keyinfo);
        memset(cx, 0, sizeof(*cx));
        kfree(cx);
        memset(key, 0, STEGFS_KEY_BYTES);
        kfree(key);
        return;
}


#else
void stegfs_encrypt_btab(struct super_block *sb, int slevel, unsigned int bnum,
			 struct stegfs_btable *btab)
{
	char *tmpdata;
	char *key, *keys;

	tmpdata = kmalloc( sizeof(struct stegfs_btable), GFP_KERNEL );

	memcpy(tmpdata, btab, sizeof(struct stegfs_btable));

	key = stegfs_make_key(sb, slevel, bnum);
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
	memset(key, 0, STEGFS_KEY_BYTES);
	kfree(key);
	STEGFS_SB(sb)->s_x->s_cipher->encrypt(keys, tmpdata, (char *)btab);
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);
	kfree(tmpdata);

	return;
}

void stegfs_encrypt_btab2(struct super_block *sb, int slevel, unsigned int bnum,
			 struct stegfs_btable *btab, struct stegfs_btable *btab2)
{
	char *key, *keys;

	key = stegfs_make_key(sb, slevel, bnum);
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
	memset(key, 0, STEGFS_KEY_BYTES);
	kfree(key);
	STEGFS_SB(sb)->s_x->s_cipher->encrypt(keys, (char *)btab, (char *)btab2);
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);

	return;
}

void stegfs_decrypt_btab(struct super_block *sb, int slevel, unsigned int bnum,
			 struct stegfs_btable *btab)
{
	char *tmpdata;
	char *key, *keys;

	tmpdata = kmalloc( sizeof(struct stegfs_btable), GFP_KERNEL );

	memcpy(tmpdata, btab, sizeof(struct stegfs_btable));

	key = stegfs_make_key(sb, slevel, bnum);
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
	memset(key, 0, STEGFS_KEY_BYTES);
	kfree(key);
	STEGFS_SB(sb)->s_x->s_cipher->decrypt(keys, tmpdata, (char *)btab);
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);
	kfree(tmpdata);

	return;
}


void stegfs_decrypt_btab2(struct super_block *sb, int slevel,
			  unsigned int bnum,
			  struct stegfs_btable *btab,
			  struct stegfs_btable *btab2)
{
	char *key, *keys;

	key = stegfs_make_key(sb, slevel, bnum);
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
	memset(key, 0, STEGFS_KEY_BYTES);
	kfree(key);
	STEGFS_SB(sb)->s_x->s_cipher->decrypt(keys, (char *)btab, (char *)btab2);
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);

	return;
}


void stegfs_encrypt_cbc(struct super_block *sb, int slevel, unsigned int bnum,
			char *data, int size, unsigned short *iv)
{
	char *key, *keys;
	char *p;
	char inb[16], outb[16];
	int i;

	if(size % 16) {
		stegfs_debug("error: not multiple of 128 bits\n");
		return;
	}

	key = stegfs_make_key(sb, slevel, bnum);
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
	for(p=data; p<(data+size); p+=16) {
		if (p>data) {
			for(i=0; i<16; i++)
				inb[i] = p[i]^outb[i];
		}
		else {
			memcpy(inb, p, 16);
#ifdef STEGFS_NOIV
			*iv = 0;
#else
			/* get_random_bytes(iv,2); */
			*iv = *iv + 1;
#endif
			((unsigned short *)inb)[0] ^= *iv;
		}
		STEGFS_SB(sb)->s_x->s_cipher->encrypt(keys, inb, outb);
		memcpy(p, outb, 16);
	}
	memset(key, 0, STEGFS_KEY_BYTES);
	kfree(key);
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);
	return;

}

void stegfs_encrypt_cbc2(struct super_block *sb, int slevel, unsigned int bnum,
			 char *data, char *data2, int size, unsigned short *iv)
{
	char *key, *keys;
	char *p, *p2;
	char inb[16];
	int i;

	if(size % 16) {
		stegfs_debug("error: not multiple of 128 bits\n");
		return;
	}

	key = stegfs_make_key(sb, slevel, bnum);
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
	for(p=data, p2=data2; p<(data+size); p+=16, p2+=16) {
		if (p>data) {
			for(i=0; i<16; i++)
				inb[i] = p[i]^(p2-16)[i];
		}
		else {
			memcpy(inb, p, 16);
#ifdef STEGFS_NOIV
			*iv = 0;
#else
			/* get_random_bytes(iv,2); */
			*iv = *iv + 1;
#endif
			((unsigned short *)inb)[0] ^= *iv;
		}
		STEGFS_SB(sb)->s_x->s_cipher->encrypt(keys, inb, p2);
	}
	memset(key, 0, STEGFS_KEY_BYTES);
	kfree(key);
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);
	return;

}

void stegfs_decrypt_cbc(struct super_block *sb, int slevel, unsigned int bnum,
			char *data, int size, unsigned short iv)
{
	char *key, *keys;
	char *p;
	char outb[16];
	char inb[16], inb2[16];
	int i;

	if(size % 16) {
		stegfs_debug("error: not multiple of 128 bits\n");
		return;
	}

	key = stegfs_make_key(sb, slevel, bnum);
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
	for(p=data; p<(data+size); p+=16) {
		if (p>data) {
			memcpy(inb2, inb, 16);
			memcpy(inb, p, 16);
		}
		STEGFS_SB(sb)->s_x->s_cipher->decrypt(keys, p, outb);
		if (p>data) {
			for(i=0; i<16; i++)
				p[i] = outb[i]^inb2[i];
		}
		else {
			((unsigned short *)outb)[0] ^= iv;
			memcpy(inb, p, 16);
			memcpy(p, outb, 16);
		}
	}
	memset(key, 0, STEGFS_KEY_BYTES);
	kfree(key);
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);
	return;

}

void stegfs_decrypt_cbc2(struct super_block *sb, int slevel, unsigned int bnum,
			 char *data, char *data2, int size, unsigned short iv)
{
	char *key, *keys;
	char *p, *p2;
	int i;

	if(size % 16) {
		stegfs_debug("error: not multiple of 128 bits\n");
		return;
	}

	key = stegfs_make_key(sb, slevel, bnum);
	keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
	for(p=data, p2=data2; p<(data+size); p+=16, p2+=16) {
		STEGFS_SB(sb)->s_x->s_cipher->decrypt(keys, p, p2);
		if (p>data) {
			for(i=0; i<16; i++)
				p2[i] = p2[i]^(p-16)[i];
		}
		else
			((unsigned short *)p2)[0] ^= iv;
	}
	memset(key, 0, STEGFS_KEY_BYTES);
	kfree(key);
	memset(keys, 0, STEGFS_SB(sb)->s_x->s_cipher->keys_size);
	kfree(keys);
	return;

}

#endif
