/*
 *  linux/fs/stegfs/namei.c
 *
 * Copyright (C) 1999-2001
 * Andrew McDonald (andrew@mcdonald.org.uk)
 *
 *  from
 *
 *  linux/fs/ext2/namei.c
 *
 *  Copyright (C) 1992, 1993, 1994, 1995
 *  Remy Card (card@masi.ibp.fr)
 *  Laboratoire MASI - Institut Blaise Pascal
 *  Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/fs/minix/namei.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  Big-endian to little-endian byte-swapping/bitmaps by
 *        David S. Miller (davem@caip.rutgers.edu), 1995
 *  Directory entry file type support and forward compatibility hooks
 *  	for B-tree directories by Theodore Ts'o (tytso@mit.edu), 1998
 */

#include "fs.h"
#include "stegfs_fs.h"

#include <asm/uaccess.h>

#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/locks.h>
#include <linux/quotaops.h>
#include <linux/malloc.h>
#include <linux/ctype.h>

/*
 * define how far ahead to read directories while searching them.
 */
#define NAMEI_RA_CHUNKS  2
#define NAMEI_RA_BLOCKS  4
#define NAMEI_RA_SIZE        (NAMEI_RA_CHUNKS * NAMEI_RA_BLOCKS)
#define NAMEI_RA_INDEX(c,b)  (((c) * NAMEI_RA_BLOCKS) + (b))

/*
 * NOTE! unlike strncmp, stegfs_match returns 1 for success, 0 for failure.
 *
 * `len <= EXT2_NAME_LEN' is guaranteed by caller.
 * `de != NULL' is guaranteed by caller.
 */
static inline int stegfs_match (int len, const char * const name,
		       struct ext2_dir_entry_2 * de)
{
	if (len != de->name_len)
		return 0;
	if (!de->inode)
		return 0;
	return !memcmp(name, de->name, len);
}

/*
 *	stegfs_find_entry()
 *
 * finds an entry in the specified directory with the wanted name. It
 * returns the cache buffer in which the entry was found, and the entry
 * itself (as a parameter - res_dir). It does NOT read the inode of the
 * entry - you'll have to do that yourself if you want to.
 */
static struct buffer_head * stegfs_find_entryn (struct inode * dir,
				   const char * const name, int namelen,
				   char ** bufplain, unsigned short * biv,
				   struct ext2_dir_entry_2 ** res_dir,
				   int * blocknum, int ncopy)
{
	struct super_block * sb;
	struct buffer_head * bh_use[NAMEI_RA_SIZE];
	struct buffer_head * bh_read[NAMEI_RA_SIZE];
	unsigned long offset;
	int block, toread, i, err, cerr;
	unsigned short iv;
	unsigned short iv_use[NAMEI_RA_SIZE];

	*bufplain = NULL;
	*res_dir = NULL;
	*blocknum = 0;
	sb = dir->i_sb;

	if (namelen > EXT2_NAME_LEN)
		return NULL;

	memset (bh_use, 0, sizeof (bh_use));
	toread = 0;
	for (block = 0; block < NAMEI_RA_SIZE; ++block) {
		struct buffer_head * bh;

		if ((block << EXT2_BLOCK_SIZE_BITS (sb)) >= dir->i_size)
			break;
		bh = stegfs_getblkn (dir, block, &iv, 0,
				     &err, ncopy, 0, &cerr);
		bh_use[block] = bh;
		iv_use[block] = iv;
		if (bh && !buffer_uptodate(bh))
			bh_read[toread++] = bh;
	}

	for (block = 0, offset = 0; offset < dir->i_size; block++) {
		struct buffer_head * bh;
		struct ext2_dir_entry_2 * de;
		char * dlimit;

		if ((block % NAMEI_RA_BLOCKS) == 0 && toread) {
			ll_rw_block (READ, toread, bh_read);
			toread = 0;
		}
		bh = bh_use[block % NAMEI_RA_SIZE];
		iv = iv_use[block % NAMEI_RA_SIZE];
		if (!bh) {
#if 0
			stegfs_error (sb, "stegfs_find_entry",
				    "directory #%lu contains a hole at offset %lu",
				    dir->i_ino, offset);
#endif
			offset += sb->s_blocksize;
			continue;
		}
		wait_on_buffer (bh);
		if (!buffer_uptodate(bh)) {
			/*
			 * read error: all bets are off
			 */
			break;
		}
		if (STEGFS_IS_HID_INO(dir->i_ino)) {
			*bufplain = kmalloc(bh->b_size, GFP_KERNEL);
			stegfs_decrypt_cbc2(dir->i_sb,
					    STEGFS_INO_LVL(dir->i_ino),
					    bh->b_blocknr,
					    bh->b_data, *bufplain,
					    bh->b_size, iv);
		}
		else
			*bufplain = bh->b_data;

		*blocknum = block;
		de = (struct ext2_dir_entry_2 *) *bufplain;
		dlimit = *bufplain + sb->s_blocksize;
		while ((char *) de < dlimit) {
			/* this code is executed quadratically often */
			/* do minimal checking `by hand' */
			int de_len;

			if ((char *) de + namelen <= dlimit &&
			    stegfs_match (namelen, name, de)) {
				/* found a match -
				   just to be sure, do a full check */
				if (!stegfs_check_dir_entry("stegfs_find_entry",
							    dir, de, *bufplain,
							    offset))
					goto failure;
				for (i = 0; i < NAMEI_RA_SIZE; ++i) {
					if (bh_use[i] != bh)
						brelse (bh_use[i]);
				}
				*res_dir = de;
				*biv = iv;
				return bh;
			}
			/* prevent looping on a bad block */
			de_len = le16_to_cpu(de->rec_len);
			if (de_len <= 0)
				goto failure;
			offset += de_len;
			de = (struct ext2_dir_entry_2 *)
				((char *) de + de_len);
		}

		brelse (bh);
		if(STEGFS_IS_HID_INO(dir->i_ino))
			kfree(*bufplain);
		*bufplain = NULL;

		if (((block + NAMEI_RA_SIZE) << EXT2_BLOCK_SIZE_BITS (sb)) >=
		    dir->i_size)
			bh = NULL;
		else
			bh = stegfs_getblkn (dir, block + NAMEI_RA_SIZE, &iv, 0,
					     &err, ncopy, 0, &cerr);
		bh_use[block % NAMEI_RA_SIZE] = bh;
		iv_use[block % NAMEI_RA_SIZE] = iv;
		if (bh && !buffer_uptodate(bh))
			bh_read[toread++] = bh;
	}

failure:
	for (i = 0; i < NAMEI_RA_SIZE; ++i)
		brelse (bh_use[i]);
	if(STEGFS_IS_HID_INO(dir->i_ino))
		kfree(*bufplain);
	*bufplain = NULL;

	return NULL;
}

static struct buffer_head * stegfs_find_entry (struct inode * dir,
					     const char * const name, int namelen,
					       char ** bufplain, unsigned short * iv,
					     struct ext2_dir_entry_2 ** res_dir,
					       int * blocknum)
{
	return stegfs_find_entryn (dir, name, namelen, bufplain, iv, res_dir,
				   blocknum, STEGFS_GB_ANYCOPY);
}
				  
struct dentry *stegfs_lookup(struct inode * dir, struct dentry *dentry)
{
	struct inode * inode;
	struct ext2_dir_entry_2 * de;
	struct buffer_head * bh;
	char * bufplain;
	int blocknum;
	unsigned short iv;

	if (dentry->d_name.len > EXT2_NAME_LEN)
		return ERR_PTR(-ENAMETOOLONG);

	bh = NULL;
        stegfs_debug("ino: %lu l: %u name: %s\n",
                    dir->i_ino,
                    dentry->d_name.len, dentry->d_name.name);

	if (STEGFS_IS_HID_INO(dir->i_ino)) {
		if (!dir->i_sb->u.stegfs_sb.s_x->s_slkeys[STEGFS_INO_LVL(dir->i_ino)-1])
			return ERR_PTR(-EACCES);
	}

        if (dir->i_ino == EXT2_ROOT_INO && dir->i_sb->u.stegfs_sb.s_x->s_isstegfs  &&
            dentry->d_name.len == 6 &&
            !memcmp(dentry->d_name.name, "stegfs", 6)) {

                inode = iget(dir->i_sb, STEGFS_HID_ROOT_INO);
                
                if (!inode)
                        return ERR_PTR(-EACCES);
		
                d_add(dentry, inode);
                return NULL;
        }
	else if(dir->i_ino == STEGFS_HID_ROOT_INO) {
		unsigned long slevel;

		if (dentry->d_name.len < 1 || dentry->d_name.len > 2 ||
		    dentry->d_name.name[0] == '0' ||
		      !isdigit(dentry->d_name.name[0]) ||
		      (dentry->d_name.len == 2 &&
		       !isdigit(dentry->d_name.name[1])))
			return ERR_PTR(-ENOENT);

		slevel = simple_strtoul(dentry->d_name.name, NULL, 10);

		if (slevel < 1 || slevel > STEGFS_MAX_LEVELS-1)
			return ERR_PTR(-ENOENT);
		/*
		  stegfs_debug("%lu\n",
		  simple_strtoul(dentry->d_name.name,NULL,0));
		*/
		if (!dir->i_sb->u.stegfs_sb.s_x->s_slkeys[slevel-1])
			return ERR_PTR(-ENOENT);

		inode = iget(dir->i_sb, EXT2_ROOT_INO|STEGFS_INO_MASK|
			     (slevel<<STEGFS_LVL_SHIFT));

		if (!inode)
			return ERR_PTR(-EACCES);

		stegfs_debug("%s -> %lu\n", dentry->d_name.name,
			     inode->i_ino);

		d_add(dentry, inode);
		return NULL;



	}
	else {
		bh = stegfs_find_entry (dir, dentry->d_name.name,
					dentry->d_name.len, &bufplain,
					&iv, &de, &blocknum);
	}

	inode = NULL;
	if (bh) {
		unsigned long ino = le32_to_cpu(de->inode);
		brelse (bh);
		if(STEGFS_IS_HID_INO(dir->i_ino))
			kfree(bufplain);
		inode = iget(dir->i_sb, ino);

		if (!inode)
			return ERR_PTR(-EACCES);
	}
	d_add(dentry, inode);
	return NULL;
}

/*
 *	stegfs_add_entry()
 *
 * adds a file entry to the specified directory, using the same
 * semantics as stegfs_find_entry(). It returns NULL if it failed.
 *
 * NOTE!! The inode part of 'de' is left at 0 - which means you
 * may not sleep between calling this and putting something into
 * the entry, as someone else might have used it while you slept.
 */
static struct buffer_head * stegfs_add_entry (struct inode * dir,
					    const char * name, int namelen,
					    char ** bufplain,
					    unsigned short * iv,
					    struct ext2_dir_entry_2 ** res_dir,
					    int *blocknum, int *err)
{
	unsigned long offset;
	unsigned short rec_len;
	struct buffer_head * bh;
	struct ext2_dir_entry_2 * de, * de1;
	struct super_block * sb;

	*err = -EINVAL;
	*res_dir = NULL;
	if (!dir || !dir->i_nlink)
		return NULL;
	sb = dir->i_sb;

	if (!namelen)
		return NULL;
	/*
	 * Is this a busy deleted directory?  Can't create new files if so
	 */
	if (dir->i_size == 0)
	{
		*err = -ENOENT;
		return NULL;
	}
	*blocknum = 0;
	bh = stegfs_bread (dir, 0, iv, 0, err);
	if (!bh)
		return NULL;
	rec_len = EXT2_DIR_REC_LEN(namelen);
	offset = 0;
	if (STEGFS_IS_HID_INO(dir->i_ino)) {
		*bufplain = kmalloc(bh->b_size, GFP_KERNEL);
		stegfs_decrypt_cbc2(dir->i_sb,
				    STEGFS_INO_LVL(dir->i_ino),
				    bh->b_blocknr,
				    bh->b_data, *bufplain, bh->b_size, *iv);
	}
	else
		*bufplain = bh->b_data;
	de = (struct ext2_dir_entry_2 *) *bufplain;
	*err = -ENOSPC;
	while (1) {
		if ((char *)de >= sb->s_blocksize + *bufplain) {
			brelse (bh);
			if(STEGFS_IS_HID_INO(dir->i_ino))
				kfree(*bufplain);
			*bufplain = NULL;
			bh = NULL;
			*bufplain = NULL;
			*blocknum = offset >> EXT2_BLOCK_SIZE_BITS(sb);
			bh = stegfs_bread (dir, offset >> EXT2_BLOCK_SIZE_BITS(sb), iv, 1, err);
			if (!bh)
				return NULL;
			if (dir->i_size <= offset) {
				if (dir->i_size == 0) {
					*err = -ENOENT;
					return NULL;
				}

				stegfs_debug ("creating next block\n");
				if (STEGFS_IS_HID_INO(dir->i_ino)) {
					*bufplain = kmalloc(bh->b_size, GFP_KERNEL);
					stegfs_decrypt_cbc2(dir->i_sb,
							    STEGFS_INO_LVL(dir->i_ino),
							    bh->b_blocknr,
							    bh->b_data, *bufplain,
							    bh->b_size, *iv);
				}
				else
					*bufplain = bh->b_data;

				de = (struct ext2_dir_entry_2 *) *bufplain;
				de->inode = 0;
				de->rec_len = le16_to_cpu(sb->s_blocksize);
				dir->i_size = offset + sb->s_blocksize;
				dir->u.stegfs_i.i_flags &= ~EXT2_BTREE_FL;
				mark_inode_dirty(dir);
			} else {

				stegfs_debug ("skipping to next block\n");
				if (STEGFS_IS_HID_INO(dir->i_ino)) {
					*bufplain = kmalloc(bh->b_size, GFP_KERNEL);
					stegfs_decrypt_cbc2(dir->i_sb,
							    STEGFS_INO_LVL(dir->i_ino),
							    bh->b_blocknr,
							    bh->b_data, *bufplain,
							    bh->b_size, *iv);
				}
				else
					*bufplain = bh->b_data;

				de = (struct ext2_dir_entry_2 *) *bufplain;
			}
		}
		if (!stegfs_check_dir_entry ("stegfs_add_entry", dir, de,
					     *bufplain, offset)) {
			*err = -ENOENT;
			brelse (bh);
			if(STEGFS_IS_HID_INO(dir->i_ino))
				kfree(*bufplain);
			*bufplain = NULL;
			return NULL;
		}
		if (stegfs_match (namelen, name, de)) {
			*err = -EEXIST;
			brelse (bh);
			if(STEGFS_IS_HID_INO(dir->i_ino))
				kfree(*bufplain);
			*bufplain = NULL;
			return NULL;
		}
		if ((le32_to_cpu(de->inode) == 0 && le16_to_cpu(de->rec_len) >= rec_len) ||
		    (le16_to_cpu(de->rec_len) >= EXT2_DIR_REC_LEN(de->name_len) + rec_len)) {
			offset += le16_to_cpu(de->rec_len);
			if (le32_to_cpu(de->inode)) {
				de1 = (struct ext2_dir_entry_2 *) ((char *) de +
					EXT2_DIR_REC_LEN(de->name_len));
				de1->rec_len = cpu_to_le16(le16_to_cpu(de->rec_len) -
					EXT2_DIR_REC_LEN(de->name_len));
				de->rec_len = cpu_to_le16(EXT2_DIR_REC_LEN(de->name_len));
				de = de1;
			}
			de->inode = 0;
			de->name_len = namelen;
			de->file_type = 0;
			memcpy (de->name, name, namelen);
			/*
			 * XXX shouldn't update any times until successful
			 * completion of syscall, but too many callers depend
			 * on this.
			 *
			 * XXX similarly, too many callers depend on
			 * stegfs_new_inode() setting the times, but error
			 * recovery deletes the inode, so the worst that can
			 * happen is that the times are slightly out of date
			 * and/or different from the directory change time.
			 */
			dir->i_mtime = dir->i_ctime = CURRENT_TIME;
			dir->u.stegfs_i.i_flags &= ~EXT2_BTREE_FL;
			mark_inode_dirty(dir);
			dir->i_version = ++global_event;
/*
			if (STEGFS_IS_HID_INO(dir->i_ino)) {
				stegfs_encrypt_cbc2(dir->i_sb,
						    STEGFS_INO_LVL(dir->i_ino),
						    bh->b_blocknr,
						    *bufplain, bh->b_data,
						    bh->b_size, &iv);
			}
			mark_buffer_dirty(bh, 1);
*/
			*res_dir = de;
			*err = 0;
			return bh;
		}
		offset += le16_to_cpu(de->rec_len);
		de = (struct ext2_dir_entry_2 *) ((char *) de + le16_to_cpu(de->rec_len));
	}
	brelse (bh);
	if(STEGFS_IS_HID_INO(dir->i_ino))
		kfree(*bufplain);
	*bufplain = NULL;
	return NULL;
}

/*
 * stegfs_delete_entry deletes a directory entry by merging it with the
 * previous entry
 */
static int stegfs_delete_entry (struct ext2_dir_entry_2 * dir,
				struct buffer_head * bh,
				char * bufplain)
{
	struct ext2_dir_entry_2 * de, * pde;
	int i;

	i = 0;
	pde = NULL;
	de = (struct ext2_dir_entry_2 *) bufplain;
	while (i < bh->b_size) {
		if (!stegfs_check_dir_entry ("stegfs_delete_entry", NULL, 
					   de, bufplain, i))
			return -EIO;
		if (de == dir)  {
			if (pde)
				pde->rec_len =
					cpu_to_le16(le16_to_cpu(pde->rec_len) +
						    le16_to_cpu(dir->rec_len));
			else
				dir->inode = 0;
			return 0;
		}
		i += le16_to_cpu(de->rec_len);
		pde = de;
		de = (struct ext2_dir_entry_2 *) ((char *) de + le16_to_cpu(de->rec_len));
	}
	return -ENOENT;
}

/*
 * By the time this is called, we already have created
 * the directory cache entry for the new file, but it
 * is so far negative - it has no inode.
 *
 * If the create succeeds, we fill in the inode information
 * with d_instantiate(). 
 */
int stegfs_create (struct inode * dir, struct dentry * dentry, int mode)
{
	struct inode * inode;
	struct buffer_head * bh;
	struct ext2_dir_entry_2 * de;
	int err = -EIO;
	struct stegfs_btable btab, btab2;
	int blocknum = -1;
	unsigned int bnum;
	char * bufplain;
	int i;
	int cerr;

	/*
	 * N.B. Several error exits in stegfs_new_inode don't set err.
	 */
	inode = stegfs_new_inode (dir, mode, &err);
	if (!inode)
		return err;

	inode->i_op = &stegfs_file_inode_operations;
	inode->i_mode = mode;
	mark_inode_dirty(inode);
	bh = stegfs_add_entry (dir, dentry->d_name.name, dentry->d_name.len,
			       &bufplain, &btab.iv, &de, &blocknum, &err);
	if (!bh) {
		inode->i_nlink--;
		mark_inode_dirty(inode);
		iput (inode);
		return err;
	}
	de->inode = cpu_to_le32(inode->i_ino);
	if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
				      EXT2_FEATURE_INCOMPAT_FILETYPE))
		de->file_type = EXT2_FT_REG_FILE;
	dir->i_version = ++global_event;
	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		mark_buffer_dirty(bh, 1);
		if (IS_SYNC(dir)) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		brelse (bh);
		d_instantiate(dentry, inode);
		return 0;
	}
	/* else STEGFS_IS_HID_INO */
	stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
			    bh->b_blocknr, bufplain, bh->b_data,
			    bh->b_size, &btab.iv);
	mark_buffer_dirty(bh, 1);
	btab.magic1 = 0;
	btab.magic2 = 0;
	btab.ino = dir->i_ino;
	btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
	stegfs_encrypt_btab2(dir->i_sb,
			     STEGFS_INO_LVL(dir->i_ino),
			     bh->b_blocknr, &btab, &btab2);
	stegfs_put_btab(dir->i_sb, bh->b_blocknr, &btab2);
	if (IS_SYNC(dir)) {
		ll_rw_block (WRITE, 1, &bh);
		wait_on_buffer (bh);
	}
	bnum = bh->b_blocknr;
	brelse (bh);
	for (i=0; i<dir->u.stegfs_i.i_bcopies; i++) {
		bh = stegfs_getblkn(dir, blocknum, &btab.iv, 1, &err,
				    i, 1, &cerr);
		if (bnum == bh->b_blocknr) {
			brelse(bh);
			continue;
		}

		stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
				    bh->b_blocknr, bufplain, bh->b_data,
				    bh->b_size, &btab.iv);
		mark_buffer_uptodate(bh, 1);
		mark_buffer_dirty(bh, 1);
		btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
		if (IS_SYNC(dir)) {
			ll_rw_block(WRITE, 1, &bh);
			wait_on_buffer(bh);
			if (!buffer_uptodate(bh)) {
				brelse (bh);
				kfree(bufplain);
				return -EIO;
			}
		}
		stegfs_encrypt_btab2(dir->i_sb,
				     STEGFS_INO_LVL(dir->i_ino),
				     bh->b_blocknr, &btab, &btab2);
		stegfs_put_btab(dir->i_sb,
				bh->b_blocknr, &btab2);
		brelse(bh);

	}
	kfree(bufplain);

	d_instantiate(dentry, inode);
	return 0;
}

int stegfs_mknod (struct inode * dir, struct dentry *dentry, int mode, int rdev)
{
	struct inode * inode;
	struct buffer_head * bh;
	struct ext2_dir_entry_2 * de;
	int err = -EIO;
        struct stegfs_btable btab, btab2;
 	int blocknum = -1;
	unsigned int bnum;
	char * bufplain;
	int i;
	int cerr;

	inode = stegfs_new_inode (dir, mode, &err);
	if (!inode)
		goto out;

	inode->i_uid = current->fsuid;
	inode->i_mode = mode;
	inode->i_op = NULL;
	bh = stegfs_add_entry (dir, dentry->d_name.name, dentry->d_name.len,
			       &bufplain, &btab.iv, &de, &blocknum, &err);
	if (!bh)
		goto out_no_entry;
	de->inode = cpu_to_le32(inode->i_ino);
	dir->i_version = ++global_event;
	if (S_ISREG(inode->i_mode)) {
		inode->i_op = &stegfs_file_inode_operations;
		if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
					      EXT2_FEATURE_INCOMPAT_FILETYPE))
			de->file_type = EXT2_FT_REG_FILE;
	} else if (S_ISSOCK(inode->i_mode)) {
		if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
					      EXT2_FEATURE_INCOMPAT_FILETYPE))
			de->file_type = EXT2_FT_SOCK;
	} else if (S_ISCHR(inode->i_mode)) {
		inode->i_op = &chrdev_inode_operations;
		if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
					      EXT2_FEATURE_INCOMPAT_FILETYPE))
			de->file_type = EXT2_FT_CHRDEV;
	} else if (S_ISBLK(inode->i_mode)) {
		inode->i_op = &blkdev_inode_operations;
		if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
					      EXT2_FEATURE_INCOMPAT_FILETYPE))
			de->file_type = EXT2_FT_BLKDEV;
	} else if (S_ISFIFO(inode->i_mode))  {
		init_fifo(inode);
		if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
					      EXT2_FEATURE_INCOMPAT_FILETYPE))
			de->file_type = EXT2_FT_FIFO;
	}
	if (S_ISBLK(mode) || S_ISCHR(mode))
		inode->i_rdev = to_kdev_t(rdev);
	mark_inode_dirty(inode);
	if (!STEGFS_IS_HID_INO(dir->i_ino)) {
		mark_buffer_dirty(bh, 1);
		if (IS_SYNC(dir)) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		d_instantiate(dentry, inode);
		brelse(bh);
		return 0;
	}
        /* else STEGFS_IS_HID_INO */
	stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
			    bh->b_blocknr, bufplain, bh->b_data,
			    bh->b_size, &btab.iv);
	mark_buffer_dirty(bh, 1);
        btab.magic1 = 0;
        btab.magic2 = 0;
        btab.ino = dir->i_ino;
        btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);

	stegfs_encrypt_btab2(dir->i_sb,
			     STEGFS_INO_LVL(dir->i_ino),
			     bh->b_blocknr, &btab, &btab2);
        stegfs_put_btab(dir->i_sb, bh->b_blocknr, &btab2);
        if (IS_SYNC(dir)) {
                ll_rw_block (WRITE, 1, &bh);
                wait_on_buffer (bh);
        }
	bnum = bh->b_blocknr;
        brelse (bh);
        for (i=0; i<dir->u.stegfs_i.i_bcopies; i++) {
                bh = stegfs_getblkn(dir, blocknum, &btab.iv, 1, &err,
				    i, 1, &cerr);
		if (bnum == bh->b_blocknr) {
			brelse(bh);
			continue;
		}
		stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
				    bh->b_blocknr, bufplain, bh->b_data,
				    bh->b_size, &btab.iv);
                mark_buffer_uptodate(bh, 1);
                mark_buffer_dirty(bh, 1);
		btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
                if (IS_SYNC(dir)) {
                        ll_rw_block(WRITE, 1, &bh);
                        wait_on_buffer(bh);
                        if (!buffer_uptodate(bh)) {
                                brelse (bh);
				kfree(bufplain);
                                return -EIO;
                        }
                }
 
		stegfs_encrypt_btab2(dir->i_sb,
				     STEGFS_INO_LVL(dir->i_ino),
				     bh->b_blocknr, &btab, &btab2);
                stegfs_put_btab(dir->i_sb,
                                bh->b_blocknr, &btab2);
                brelse(bh);

        }
        kfree(bufplain);

        d_instantiate(dentry, inode);

	err = 0;
out:
	return err;

out_no_entry:
	inode->i_nlink--;
	mark_inode_dirty(inode);
	iput(inode);
	goto out;
}

int stegfs_mkdir(struct inode * dir, struct dentry * dentry, int mode)
{
	struct inode * inode;
	struct buffer_head * bh, * dir_block;
	struct ext2_dir_entry_2 * de;
	int err;
	struct stegfs_btable btab, btab2;
	int blocknum = -1;
	char * bufplain;
	unsigned int bnum;
	int i;
	int cerr;
	unsigned short iv;

	err = -EMLINK;
	if (dir->i_nlink >= EXT2_LINK_MAX)
		goto out;

	err = -EIO;
	inode = stegfs_new_inode (dir, S_IFDIR, &err);
	if (!inode)
		goto out;

	inode->i_op = &stegfs_dir_inode_operations;
	inode->i_size = 0;
	inode->i_blocks = 0;	
	dir_block = stegfs_bread (inode, 0, &btab.iv, 1, &err);
       	inode->i_size = inode->i_sb->s_blocksize;
	if (!dir_block) {
		inode->i_nlink--; /* is this nlink == 0? */
		mark_inode_dirty(inode);
		iput (inode);
		return err;
	}
	if(STEGFS_IS_HID_INO(inode->i_ino)) {
		bufplain = kmalloc(dir_block->b_size, GFP_KERNEL);
		memset(bufplain, 0, dir_block->b_size);
	}
	else
		bufplain = dir_block->b_data;
	de = (struct ext2_dir_entry_2 *) bufplain;
	de->inode = cpu_to_le32(inode->i_ino);
	de->name_len = 1;
	de->rec_len = cpu_to_le16(EXT2_DIR_REC_LEN(de->name_len));
	strcpy (de->name, ".");
	if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
				      EXT2_FEATURE_INCOMPAT_FILETYPE))
		de->file_type = EXT2_FT_DIR;
	de = (struct ext2_dir_entry_2 *) ((char *) de + le16_to_cpu(de->rec_len));
	de->inode = cpu_to_le32(dir->i_ino);
	de->rec_len = cpu_to_le16(inode->i_sb->s_blocksize - EXT2_DIR_REC_LEN(1));
	de->name_len = 2;
	strcpy (de->name, "..");
	if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
				      EXT2_FEATURE_INCOMPAT_FILETYPE))
		de->file_type = EXT2_FT_DIR;
	inode->i_nlink = 2;

	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		mark_buffer_dirty(dir_block, 1);
		brelse (dir_block);
	}
	else {
		stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
				    dir_block->b_blocknr, bufplain,
				    dir_block->b_data, dir_block->b_size,
				    &btab.iv);
		btab.magic1 = 0;
		btab.magic2 = 0;
		btab.ino = inode->i_ino;
		btab.bchecksum = stegfs_chksum(dir_block->b_data,
					       dir_block->b_size);
		stegfs_encrypt_btab2(inode->i_sb,
				     STEGFS_INO_LVL(inode->i_ino),
				     dir_block->b_blocknr, &btab, &btab2);
		stegfs_put_btab(inode->i_sb, dir_block->b_blocknr, &btab2);
		mark_buffer_dirty(dir_block, 1);
		bnum = dir_block->b_blocknr;
		brelse (dir_block);

		for (i=0; i<inode->u.stegfs_i.i_bcopies; i++) {
			dir_block = stegfs_getblkn (inode, 0, &btab.iv,
						    1, &err, i, 1, &cerr);
			if (bnum == dir_block->b_blocknr) {
				brelse(dir_block);
				continue;
			}
			stegfs_encrypt_cbc2(dir->i_sb,
					    STEGFS_INO_LVL(dir->i_ino),
					    dir_block->b_blocknr, bufplain,
					    dir_block->b_data,
					    dir_block->b_size, &btab.iv);
			mark_buffer_uptodate(dir_block, 1);
			mark_buffer_dirty(dir_block, 1);
			btab.bchecksum = stegfs_chksum(dir_block->b_data,
						       dir_block->b_size);
			stegfs_encrypt_btab2(inode->i_sb,
					     STEGFS_INO_LVL(inode->i_ino),
					     dir_block->b_blocknr, &btab, &btab2);
			stegfs_put_btab(inode->i_sb,
					dir_block->b_blocknr, &btab2);
			brelse(dir_block);

		}
		kfree(bufplain);
	}

	inode->i_mode = S_IFDIR | (mode & (S_IRWXUGO|S_ISVTX) & ~current->fs->umask);
	if (dir->i_mode & S_ISGID)
		inode->i_mode |= S_ISGID;
	mark_inode_dirty(inode);
	bh = stegfs_add_entry (dir, dentry->d_name.name, dentry->d_name.len,
			       &bufplain, &btab.iv, &de, &blocknum, &err);
	if (!bh)
		goto out_no_entry;
	de->inode = cpu_to_le32(inode->i_ino);
	if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
				      EXT2_FEATURE_INCOMPAT_FILETYPE))
		de->file_type = EXT2_FT_DIR;
	dir->i_version = ++global_event;
        if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		mark_buffer_dirty(bh, 1);
		if (IS_SYNC(dir)) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		dir->i_nlink++;
		dir->u.stegfs_i.i_flags &= ~EXT2_BTREE_FL;
		mark_inode_dirty(dir);
		d_instantiate(dentry, inode);
		brelse (bh);
		return 0;
	}
	/* else STEGFS_IS_HID_INO */
	stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
			    bh->b_blocknr, bufplain, bh->b_data,
			    bh->b_size, &btab.iv);
	mark_buffer_dirty(bh, 1);
        btab.magic1 = 0;
        btab.magic2 = 0;
        btab.ino = dir->i_ino;
        btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
	stegfs_encrypt_btab2(dir->i_sb,
			     STEGFS_INO_LVL(dir->i_ino),
			     bh->b_blocknr, &btab, &btab2);
        stegfs_put_btab(dir->i_sb, bh->b_blocknr, &btab2);
        if (IS_SYNC(dir)) {
                ll_rw_block (WRITE, 1, &bh);
                wait_on_buffer (bh);
        }
	bnum = bh->b_blocknr;
        brelse (bh);

	for (i=0; i<dir->u.stegfs_i.i_bcopies; i++) {
                bh = stegfs_getblkn(dir, blocknum, &iv, 1, &err, i, 1, &cerr);
		if (bnum == bh->b_blocknr) {
			brelse(bh);
			continue;
		}

        	stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
			    bh->b_blocknr, bufplain, bh->b_data,
			    bh->b_size, &btab.iv);
                mark_buffer_uptodate(bh, 1);
                mark_buffer_dirty(bh, 1);
		btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
                if (IS_SYNC(dir)) {
                        ll_rw_block(WRITE, 1, &bh);
                        wait_on_buffer(bh);
                        if (!buffer_uptodate(bh)) {
                                brelse (bh);
				kfree(bufplain);
                                return -EIO;
                        }
                }
		stegfs_encrypt_btab2(dir->i_sb,
				     STEGFS_INO_LVL(dir->i_ino),
				     bh->b_blocknr, &btab, &btab2);
		stegfs_put_btab(dir->i_sb,
                                bh->b_blocknr, &btab2);
                brelse(bh);

        }
        kfree(bufplain);

	dir->i_nlink++;
	dir->u.stegfs_i.i_flags &= ~EXT2_BTREE_FL;
	mark_inode_dirty(dir);
        d_instantiate(dentry, inode);

	err = 0;
out:
	return err;

out_no_entry:
	inode->i_nlink = 0;
	mark_inode_dirty(inode);
	iput (inode);
	goto out;
}

/*
 * routine to check that the specified directory is empty (for rmdir)
 */
static int empty_dir (struct inode * inode)
{
	unsigned long offset;
	struct buffer_head * bh;
	struct ext2_dir_entry_2 * de, * de1;
	char * bufplain;
	struct super_block * sb;
	int err;
	unsigned short iv;

	sb = inode->i_sb;
	if (inode->i_size < EXT2_DIR_REC_LEN(1) + EXT2_DIR_REC_LEN(2) ||
	    !(bh = stegfs_bread (inode, 0, &iv, 0, &err))) {
	    	stegfs_warning (inode->i_sb, "empty_dir",
			      "bad directory (dir #%lu) - no data block",
			      inode->i_ino);
		return 1;
	}
	if (STEGFS_IS_HID_INO(inode->i_ino)) {
		bufplain = kmalloc(bh->b_size, GFP_KERNEL);
		stegfs_decrypt_cbc2(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    bh->b_blocknr,
				    bh->b_data, bufplain, bh->b_size, iv);
	}
	else
		bufplain = bh->b_data;
	de = (struct ext2_dir_entry_2 *) bufplain;
	de1 = (struct ext2_dir_entry_2 *) ((char *) de + le16_to_cpu(de->rec_len));
	if (le32_to_cpu(de->inode) != inode->i_ino || !le32_to_cpu(de1->inode) || 
	    strcmp (".", de->name) || strcmp ("..", de1->name)) {
	    	stegfs_warning (inode->i_sb, "empty_dir",
			      "bad directory (dir #%lu) - no `.' or `..'",
			      inode->i_ino);
		brelse (bh);
		if(STEGFS_IS_HID_INO(inode->i_ino))
			kfree(bufplain);
		return 1;
	}
	offset = le16_to_cpu(de->rec_len) + le16_to_cpu(de1->rec_len);
	de = (struct ext2_dir_entry_2 *) ((char *) de1 + le16_to_cpu(de1->rec_len));
	while (offset < inode->i_size ) {
		if (!bh || (void *) de >= (void *) (bufplain + sb->s_blocksize)) {
			brelse (bh);
			if(STEGFS_IS_HID_INO(inode->i_ino))
				kfree(bufplain);
			bh = stegfs_bread (inode, offset >> EXT2_BLOCK_SIZE_BITS(sb), &iv, 1, &err);
			if (!bh) {
#if 0
				stegfs_error (sb, "empty_dir",
					    "directory #%lu contains a hole at offset %lu",
					    inode->i_ino, offset);
#endif
				offset += sb->s_blocksize;
				continue;
			}
			if (STEGFS_IS_HID_INO(inode->i_ino)) {
				bufplain = kmalloc(bh->b_size, GFP_KERNEL);
				stegfs_decrypt_cbc2(inode->i_sb,
						    STEGFS_INO_LVL(inode->i_ino),
						    bh->b_blocknr,
						    bh->b_data, bufplain,
						    bh->b_size, iv);
			}
			else
				bufplain = bh->b_data;
			de = (struct ext2_dir_entry_2 *) bufplain;
		}
		if (!stegfs_check_dir_entry ("empty_dir", inode, de, bufplain,
					   offset)) {
			brelse (bh);
			if (STEGFS_IS_HID_INO(inode->i_ino))
				kfree(bufplain);
			return 1;
		}
		if (le32_to_cpu(de->inode)) {
			brelse (bh);
			if (STEGFS_IS_HID_INO(inode->i_ino))
				kfree(bufplain);
			return 0;
		}
		offset += le16_to_cpu(de->rec_len);
		de = (struct ext2_dir_entry_2 *) ((char *) de + le16_to_cpu(de->rec_len));
	}
	brelse (bh);
	if (STEGFS_IS_HID_INO(inode->i_ino))
		kfree(bufplain);
	return 1;
}

int stegfs_rmdir (struct inode * dir, struct dentry *dentry)
{
	int retval;
	struct inode * inode;
	struct buffer_head * bh;
	char * bufplain;
	struct stegfs_btable btab, btab2;
	struct ext2_dir_entry_2 * de;
        int i, blocknum;
	unsigned int bnum;
	int err, cerr;

	retval = -ENOENT;
	bh = stegfs_find_entry (dir, dentry->d_name.name, dentry->d_name.len,
				&bufplain, &btab.iv, &de, &blocknum);
	if (!bh)
		goto end_rmdir;

	inode = dentry->d_inode;
	if (!STEGFS_IS_HID_INO(inode->i_ino))
		DQUOT_INIT(inode);

	retval = -EIO;
	if (le32_to_cpu(de->inode) != inode->i_ino)
		goto end_rmdir;

	retval = -ENOTEMPTY;
	if (!empty_dir (inode))
		goto end_rmdir;

	retval = stegfs_delete_entry(de, bh, bufplain);
	dir->i_version = ++global_event;
	if (retval)
		goto end_rmdir;
	if (!STEGFS_IS_HID_INO(dir->i_ino)) {
		mark_buffer_dirty(bh, 1);
		if (IS_SYNC(dir)) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		brelse(bh);
	}
	else {
		stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
				    bh->b_blocknr, bufplain, bh->b_data,
				    bh->b_size, &btab.iv);

		mark_buffer_dirty(bh, 1);
		btab.magic1 = 0;
		btab.magic2 = 0;
		btab.ino = dir->i_ino;
		btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
		stegfs_encrypt_btab2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
				     bh->b_blocknr, &btab, &btab2);
		stegfs_put_btab(dir->i_sb, bh->b_blocknr, &btab2);
		bnum = bh->b_blocknr;
		brelse(bh);

		for (i=0; i<dir->u.stegfs_i.i_bcopies; i++) {
			bh = stegfs_getblkn(dir, blocknum, &btab.iv, 1, 
					    &err, i, 1, &cerr);
			if (bnum == bh->b_blocknr) {
				brelse(bh);
				continue;
			}

			stegfs_encrypt_cbc2(dir->i_sb,
					    STEGFS_INO_LVL(dir->i_ino),
					    bh->b_blocknr, bufplain,
					    bh->b_data, bh->b_size, &btab.iv);
			mark_buffer_uptodate(bh, 1);
			mark_buffer_dirty(bh, 1);
			btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
			if (IS_SYNC(dir)) {
				ll_rw_block(WRITE, 1, &bh);
				wait_on_buffer(bh);
				if (!buffer_uptodate(bh)) {
					brelse (bh);
					kfree(bufplain);
					return -EIO;
				}
			}
			stegfs_encrypt_btab2(dir->i_sb,
					     STEGFS_INO_LVL(dir->i_ino),
					     bh->b_blocknr, &btab, &btab2);
			stegfs_put_btab(dir->i_sb,
					bh->b_blocknr, &btab2);
			brelse(bh);

		}
		kfree(bufplain);

	}

	if (inode->i_nlink != 2)
		stegfs_warning (inode->i_sb, "stegfs_rmdir",
			      "empty directory has nlink!=2 (%d)",
			      inode->i_nlink);
	inode->i_version = ++global_event;
	inode->i_nlink = 0;
	inode->i_size = 0;
	mark_inode_dirty(inode);
	dir->i_nlink--;
	inode->i_ctime = dir->i_ctime = dir->i_mtime = CURRENT_TIME;
	dir->u.stegfs_i.i_flags &= ~EXT2_BTREE_FL;
	mark_inode_dirty(dir);
	d_delete(dentry);
	return 0;

end_rmdir:
	brelse (bh);
	if (STEGFS_IS_HID_INO(dir->i_ino))
		kfree(bufplain);
	return retval;
}

int stegfs_unlink(struct inode * dir, struct dentry *dentry)
{
	int retval;
	struct inode * inode;
	struct buffer_head * bh = NULL;
	char * bufplain;
	struct stegfs_btable btab, btab2;
	struct ext2_dir_entry_2 * de;
	int i, blocknum;
	unsigned int bnum;
	int err, cerr;

	retval = -ENOENT;
	bh = stegfs_find_entry (dir, dentry->d_name.name, dentry->d_name.len,
				&bufplain, &btab.iv, &de, &blocknum);
	if (!bh)
		goto end_unlink;

	inode = dentry->d_inode;

	if(!STEGFS_IS_HID_INO(inode->i_ino))
		DQUOT_INIT(inode);

	retval = -EIO;
	if (le32_to_cpu(de->inode) != inode->i_ino)
		goto end_unlink;

	if (!inode->i_nlink) {
		stegfs_warning (inode->i_sb, "stegfs_unlink",
				"Deleting nonexistent file (%lu), %d",
				inode->i_ino, inode->i_nlink);
		inode->i_nlink = 1;
	}
	retval = stegfs_delete_entry (de, bh, bufplain);

	if (retval)
		goto end_unlink;
	dir->i_version = ++global_event;
	if (!STEGFS_IS_HID_INO(dir->i_ino)) {
		mark_buffer_dirty(bh, 1);
		if (IS_SYNC(dir)) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		brelse(bh);
	}
	else {
		stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
				    bh->b_blocknr, bufplain, bh->b_data,
				    bh->b_size, &btab.iv);
		mark_buffer_dirty(bh, 1);
		btab.magic1 = 0;
		btab.magic2 = 0;
		btab.ino = dir->i_ino;
		btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
		stegfs_encrypt_btab2(dir->i_sb,
				     STEGFS_INO_LVL(dir->i_ino),
				     bh->b_blocknr, &btab, &btab2);
		stegfs_put_btab(dir->i_sb, bh->b_blocknr, &btab2);
		bnum = bh->b_blocknr;
		brelse(bh);

		for (i=0; i<dir->u.stegfs_i.i_bcopies; i++) {
			bh = stegfs_getblkn(dir, blocknum, &btab.iv, 1,
					    &err, i, 1, &cerr);
			if (bnum == bh->b_blocknr) {
				brelse(bh);
				continue;
			}

			stegfs_encrypt_cbc2(dir->i_sb,
					    STEGFS_INO_LVL(dir->i_ino),
					    bh->b_blocknr, bufplain,
					    bh->b_data, bh->b_size, &btab.iv);
			mark_buffer_uptodate(bh, 1);
			mark_buffer_dirty(bh, 1);
			btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
			if (IS_SYNC(dir)) {
				ll_rw_block(WRITE, 1, &bh);
				wait_on_buffer(bh);
				if (!buffer_uptodate(bh)) {
					brelse (bh);
					kfree(bufplain);
					return -EIO;
				}
			}
			stegfs_encrypt_btab2(dir->i_sb,
					     STEGFS_INO_LVL(dir->i_ino),
					     bh->b_blocknr, &btab, &btab2);
			stegfs_put_btab(dir->i_sb,
					bh->b_blocknr, &btab2);
			brelse(bh);

		}
		kfree(bufplain);
	}

	dir->i_ctime = dir->i_mtime = CURRENT_TIME;
	dir->u.stegfs_i.i_flags &= ~EXT2_BTREE_FL;
	mark_inode_dirty(dir);
	inode->i_nlink--;
	mark_inode_dirty(inode);
	inode->i_ctime = dir->i_ctime;
	retval = 0;

	d_delete(dentry);	/* This also frees the inode */
	return retval;

end_unlink:
	if (bh) {
		brelse (bh);
		if (STEGFS_IS_HID_INO(dir->i_ino))
			kfree(bufplain);
	}
	return retval;
}

int stegfs_symlink (struct inode * dir, struct dentry *dentry, const char * symname)
{
	struct ext2_dir_entry_2 * de;
	struct inode * inode;
	struct buffer_head * bh = NULL, * name_block = NULL;
	char * link;
	int i, l, err = -EIO;
	char c;
        struct stegfs_btable btab, btab2;
	int blocknum = -1;
	unsigned int bnum;
        char * bufplain;
        int cerr;

	if (!(inode = stegfs_new_inode (dir, S_IFLNK, &err))) {
		return err;
	}
	inode->i_mode = S_IFLNK | S_IRWXUGO;
	inode->i_op = &stegfs_symlink_inode_operations;
	for (l = 0; l < inode->i_sb->s_blocksize - 1 &&
	     symname [l]; l++)
		;
	if ((!STEGFS_IS_HID_INO(inode->i_ino) &&
	     l >= (sizeof (u32) * EXT2_N_BLOCKS))
	    || (STEGFS_IS_HID_INO(inode->i_ino) &&
		l >= sizeof (inode->u.stegfs_i.i_x->i_data))) {

		stegfs_debug ("l=%d, normal symlink\n", l);

		name_block = stegfs_bread (inode, 0, &btab.iv, 1, &err);
		if (!name_block) {
			inode->i_nlink--;
			mark_inode_dirty(inode);
			iput (inode);
			return err;
		}
		if (!STEGFS_IS_HID_INO(inode->i_ino))
			link = name_block->b_data;
		else {
			link = kmalloc(name_block->b_size, GFP_KERNEL);
			memset(link, 0, name_block->b_size);
		}
	} else {
		link = (char *) inode->u.stegfs_i.i_x->i_data;

		stegfs_debug ("l=%d, fast symlink\n", l);

	}
	i = 0;
	while (i < inode->i_sb->s_blocksize - 1 && (c = *(symname++)))
		link[i++] = c;
	link[i] = 0;
	if (name_block) {
		if (!STEGFS_IS_HID_INO(inode->i_ino)) {
			mark_buffer_dirty(name_block, 1);
			brelse (name_block);
		}
		else {
			stegfs_encrypt_cbc2(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
					    name_block->b_blocknr, link,
					    name_block->b_data,
					    name_block->b_size, &btab.iv);

			mark_buffer_dirty(bh, 1);
			btab.magic1 = 0;
			btab.magic2 = 0;
			btab.ino = inode->i_ino;
			btab.bchecksum = stegfs_chksum(name_block->b_data,
						       name_block->b_size);
			stegfs_encrypt_btab2(inode->i_sb,
					     STEGFS_INO_LVL(inode->i_ino),
					     name_block->b_blocknr,
					     &btab, &btab2);
			stegfs_put_btab(inode->i_sb,
					name_block->b_blocknr, &btab2);
			bnum = name_block->b_blocknr;
			brelse(name_block);
			for (i=0; i<inode->u.stegfs_i.i_bcopies; i++) {
				bh = stegfs_getblkn(inode, 0, &btab.iv,
						    1, &err, i, 1, &cerr);
				if (bnum == bh->b_blocknr) {
					brelse(bh);
					continue;
				}
				stegfs_encrypt_cbc2(inode->i_sb,
						    STEGFS_INO_LVL(inode->i_ino),
						    bh->b_blocknr, link,
						    bh->b_data,
						    bh->b_size, &btab.iv);

				mark_buffer_dirty(bh, 1);
				btab.bchecksum = stegfs_chksum(bh->b_data,
							       bh->b_size);
				stegfs_encrypt_btab2(inode->i_sb,
						     STEGFS_INO_LVL(inode->i_ino),
						     bh->b_blocknr,
						     &btab, &btab2);
				stegfs_put_btab(inode->i_sb,
						bh->b_blocknr, &btab2);
				brelse(bh);

			}
			kfree(link);
		}
	}
	inode->i_size = i;
	mark_inode_dirty(inode);

	bh = stegfs_add_entry (dir, dentry->d_name.name, dentry->d_name.len,
			       &bufplain, &btab.iv, &de, &blocknum, &err);
	if (!bh)
		goto out_no_entry;
	de->inode = cpu_to_le32(inode->i_ino);
	if (STEGFS_HAS_INCOMPAT_FEATURE(dir->i_sb,
				      EXT2_FEATURE_INCOMPAT_FILETYPE))
		de->file_type = EXT2_FT_SYMLINK;
	dir->i_version = ++global_event;
	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		mark_buffer_dirty(bh, 1);
		if (IS_SYNC(dir)) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		brelse (bh);
		d_instantiate(dentry, inode);
		return 0;
	}
        /* else STEGFS_IS_HID_INO */
	stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
			    bh->b_blocknr, bufplain, bh->b_data,
			    bh->b_size, &btab.iv);
	mark_buffer_dirty(bh, 1);
        btab.magic1 = 0;
        btab.magic2 = 0;
        btab.ino = dir->i_ino;
        btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
	stegfs_encrypt_btab2(dir->i_sb,
			     STEGFS_INO_LVL(dir->i_ino),
			     bh->b_blocknr, &btab, &btab2);
        stegfs_put_btab(dir->i_sb, bh->b_blocknr, &btab2);
        if (IS_SYNC(dir)) {
                ll_rw_block (WRITE, 1, &bh);
                wait_on_buffer (bh);
        }
	bnum = bh->b_blocknr;
        brelse (bh);
        for (i=0; i<dir->u.stegfs_i.i_bcopies; i++) {
                bh = stegfs_getblkn(dir, blocknum, &btab.iv,
				    1, &err, i, 1, &cerr);
		if (bnum == bh->b_blocknr) {
			brelse(bh);
			continue;
		}
		stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
				    bh->b_blocknr, bufplain, bh->b_data,
				    bh->b_size, &btab.iv);

                mark_buffer_uptodate(bh, 1);
                mark_buffer_dirty(bh, 1);
		btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
                if (IS_SYNC(dir)) {
                        ll_rw_block(WRITE, 1, &bh);
                        wait_on_buffer(bh);
                        if (!buffer_uptodate(bh)) {
                                brelse (bh);
                                return -EIO;
                        }
                }
		stegfs_encrypt_btab2(dir->i_sb,
				     STEGFS_INO_LVL(dir->i_ino),
				     bh->b_blocknr, &btab, &btab2);
                stegfs_put_btab(dir->i_sb,
                                bh->b_blocknr, &btab2);
                brelse(bh);

        }
        kfree(bufplain);

        d_instantiate(dentry, inode);
 
	err = 0;
out:
	return err;

out_no_entry:
	inode->i_nlink--;
	mark_inode_dirty(inode);
	iput (inode);
	goto out;
}

int stegfs_link (struct dentry * old_dentry,
		struct inode * dir, struct dentry *dentry)
{
	struct inode *inode = old_dentry->d_inode;
	struct ext2_dir_entry_2 * de;
	struct buffer_head * bh;
	int err;
        struct stegfs_btable btab, btab2;
 	int blocknum = -1;
	unsigned int bnum;
        char * bufplain;
        int i;
        int cerr;

	if (S_ISDIR(inode->i_mode))
		return -EPERM;

	if (inode->i_nlink >= EXT2_LINK_MAX)
		return -EMLINK;

	/* can't do links across levels */
	if ((STEGFS_IS_HID_INO(inode->i_ino) != STEGFS_IS_HID_INO(dir->i_ino)) ||
	    (STEGFS_IS_HID_INO(inode->i_ino) && STEGFS_IS_HID_INO(dir->i_ino) &&
	     STEGFS_INO_LVL(inode->i_ino) != STEGFS_INO_LVL(dir->i_ino)))
	{
		stegfs_debug("%lu %lu %lu %lu\n",
			     STEGFS_IS_HID_INO(inode->i_ino),
			     STEGFS_IS_HID_INO(dir->i_ino),
			     STEGFS_INO_LVL(inode->i_ino),
			     STEGFS_INO_LVL(dir->i_ino));
		return -EXDEV;
	}

	bh = stegfs_add_entry (dir, dentry->d_name.name, dentry->d_name.len,
			       &bufplain, &btab.iv, &de, &blocknum, &err);
	if (!bh)
		return err;

	de->inode = cpu_to_le32(inode->i_ino);
	if (STEGFS_HAS_INCOMPAT_FEATURE(inode->i_sb,
				      EXT2_FEATURE_INCOMPAT_FILETYPE)) {
		if (S_ISREG(inode->i_mode))
			de->file_type = EXT2_FT_REG_FILE;
		else if (S_ISDIR(inode->i_mode))
			de->file_type = EXT2_FT_DIR;
		else if (S_ISLNK(inode->i_mode))
			de->file_type = EXT2_FT_SYMLINK;
		else if (S_ISSOCK(inode->i_mode))  
			de->file_type = EXT2_FT_SOCK;
		else if (S_ISCHR(inode->i_mode))
			de->file_type = EXT2_FT_CHRDEV;
		else if (S_ISBLK(inode->i_mode))
			de->file_type = EXT2_FT_BLKDEV;
		else if (S_ISFIFO(inode->i_mode))  
			de->file_type = EXT2_FT_FIFO;
	}
	dir->i_version = ++global_event;
        if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		mark_buffer_dirty(bh, 1);
		if (IS_SYNC(dir)) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		brelse (bh);
		inode->i_nlink++;
		inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(inode);
		inode->i_count++;
		d_instantiate(dentry, inode);
		return 0;
	}
	/* else STEGFS_IS_HID_INO */
	stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
			    bh->b_blocknr, bufplain, bh->b_data,
			    bh->b_size, &btab.iv);
	mark_buffer_dirty(bh, 1);
        btab.magic1 = 0;
        btab.magic2 = 0;
        btab.ino = dir->i_ino;
        btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
	stegfs_encrypt_btab2(dir->i_sb,
			     STEGFS_INO_LVL(dir->i_ino),
			     bh->b_blocknr, &btab, &btab2);
        stegfs_put_btab(dir->i_sb, bh->b_blocknr, &btab2);
        if (IS_SYNC(dir)) {
                ll_rw_block (WRITE, 1, &bh);
                wait_on_buffer (bh);
        }
	bnum = bh->b_blocknr;
        brelse (bh);
	for (i=0; i<dir->u.stegfs_i.i_bcopies; i++) {
                bh = stegfs_getblkn(dir, blocknum, &btab.iv,
				    1, &err, i, 1, &cerr);
		if (bnum == bh->b_blocknr) {
			brelse(bh);
			continue;
		}
         	stegfs_encrypt_cbc2(dir->i_sb, STEGFS_INO_LVL(dir->i_ino),
				    bh->b_blocknr, bufplain, bh->b_data,
				    bh->b_size, &btab.iv);
                mark_buffer_uptodate(bh, 1);
                mark_buffer_dirty(bh, 1);
		btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
                if (IS_SYNC(dir)) {
                        ll_rw_block(WRITE, 1, &bh);
                        wait_on_buffer(bh);
                        if (!buffer_uptodate(bh)) {
                                brelse (bh);
                                return -EIO;
                        }
                }
		stegfs_encrypt_btab2(dir->i_sb,
				     STEGFS_INO_LVL(dir->i_ino),
				     bh->b_blocknr, &btab, &btab2);
                stegfs_put_btab(dir->i_sb,
                                bh->b_blocknr, &btab2);
                brelse(bh);

        }
        kfree(bufplain);
	inode->i_nlink++;
	inode->i_ctime = CURRENT_TIME;
	mark_inode_dirty(inode);
	inode->i_count++;
	d_instantiate(dentry, inode);
	return 0;
}

#define PARENT_INO(buffer) \
	((struct ext2_dir_entry_2 *) ((char *) buffer + \
	le16_to_cpu(((struct ext2_dir_entry_2 *) buffer)->rec_len)))->inode

/*
 * Anybody can rename anything with this: the permission checks are left to the
 * higher-level routines.
 */
int stegfs_rename (struct inode * old_dir, struct dentry *old_dentry,
		   struct inode * new_dir,struct dentry *new_dentry)
{
	struct inode * old_inode, * new_inode;
	struct buffer_head * old_bh, * new_bh, * dir_bh;
	char * old_bufp, * new_bufp, * dir_bufp = NULL;
	struct ext2_dir_entry_2 * old_de, * new_de;
	int retval;
	int blocknum_old, blocknum_new;
	int i, cerr, err;
	struct stegfs_btable btab, btab2;
	unsigned short dir_iv = 0;

	old_bh = new_bh = dir_bh = NULL;
	old_bufp = new_bufp = NULL;

	/* can't do rename's across levels */
	if ((STEGFS_IS_HID_INO(old_dir->i_ino) != 
	     STEGFS_IS_HID_INO(new_dir->i_ino)) ||
	    (STEGFS_IS_HID_INO(old_dir->i_ino) &&
	     STEGFS_IS_HID_INO(new_dir->i_ino) &&
	     STEGFS_INO_LVL(old_dir->i_ino) != STEGFS_INO_LVL(new_dir->i_ino)))
	{
		return -EXDEV;
	}

	old_bh = stegfs_find_entry (old_dir, old_dentry->d_name.name,
				    old_dentry->d_name.len, &old_bufp,
				    &btab.iv, &old_de, &blocknum_old);
	/*
	 *  Check for inode number is _not_ due to possible IO errors.
	 *  We might rmdir the source, keep it as pwd of some process
	 *  and merrily kill the link to whatever was created under the
	 *  same name. Goodbye sticky bit ;-<
	 */
	old_inode = old_dentry->d_inode;
	retval = -ENOENT;
	if (!old_bh || le32_to_cpu(old_de->inode) != old_inode->i_ino)
		goto end_rename;

	new_inode = new_dentry->d_inode;
	new_bh = stegfs_find_entry (new_dir, new_dentry->d_name.name,
				    new_dentry->d_name.len, &new_bufp,
				    &btab.iv, &new_de, &blocknum_new);
	if (new_bh) {
		if (!new_inode) {
			brelse (new_bh);
			new_bh = NULL;
			if (STEGFS_IS_HID_INO(new_dir->i_ino))
				kfree (new_bufp);
		} else {
			if (!STEGFS_IS_HID_INO(new_dir->i_ino))
				DQUOT_INIT(new_inode);
		}
	}
	if (S_ISDIR(old_inode->i_mode)) {
		if (new_inode) {
			retval = -ENOTEMPTY;
			if (!empty_dir (new_inode))
				goto end_rename;
		}
		retval = -EIO;
		dir_bh = stegfs_bread (old_inode, 0, &btab.iv, 0, &retval);
		if (!dir_bh)
			goto end_rename;

		if (STEGFS_IS_HID_INO(old_inode->i_ino)) {
			dir_bufp = kmalloc(dir_bh->b_size, GFP_KERNEL);
			stegfs_decrypt_cbc2(old_inode->i_sb,
					    STEGFS_INO_LVL(old_inode->i_ino),
					    dir_bh->b_blocknr,
					    dir_bh->b_data, dir_bufp,
					    dir_bh->b_size, dir_iv);
		}
		else
			dir_bufp = dir_bh->b_data;

		if (le32_to_cpu(PARENT_INO(dir_bufp)) != old_dir->i_ino)
			goto end_rename;
		retval = -EMLINK;
		if (!new_inode && new_dir!=old_dir &&
		    new_dir->i_nlink >= EXT2_LINK_MAX)
			goto end_rename;
	}
	if (!new_bh) {
		new_bh = stegfs_add_entry (new_dir, new_dentry->d_name.name,
					   new_dentry->d_name.len, &new_bufp,
					   &btab.iv,
					   &new_de, &blocknum_new, &retval);
		if (!new_bh)
			goto end_rename;
	}
	new_dir->i_version = ++global_event;

	if (STEGFS_IS_HID_INO(new_dir->i_ino)) {
		if (old_bh == new_bh) {
			stegfs_debug("old_bh == new_bh\n");
			/* Urgh */
			old_de = (struct ext2_dir_entry_2 *)( (char *)old_de -
							      old_bufp + new_bufp);
			kfree(old_bufp);
			old_bufp = new_bufp;
		}
	}

	/*
	 * Like most other Unix systems, set the ctime for inodes on a
	 * rename.
	 */
	old_inode->i_ctime = CURRENT_TIME;
	mark_inode_dirty(old_inode);

	/*
	 * ok, that's it
	 */
	new_de->inode = le32_to_cpu(old_inode->i_ino);
	if (STEGFS_HAS_INCOMPAT_FEATURE(new_dir->i_sb,
				      EXT2_FEATURE_INCOMPAT_FILETYPE))
		new_de->file_type = old_de->file_type;
	
	stegfs_delete_entry (old_de, old_bh, old_bufp);

	old_dir->i_version = ++global_event;
	if (new_inode) {
		new_inode->i_nlink--;
		new_inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(new_inode);
	}
	old_dir->i_ctime = old_dir->i_mtime = CURRENT_TIME;
	old_dir->u.stegfs_i.i_flags &= ~EXT2_BTREE_FL;
	mark_inode_dirty(old_dir);
	if (dir_bh) {
		PARENT_INO(dir_bufp) = le32_to_cpu(new_dir->i_ino);
		if (STEGFS_IS_HID_INO(old_inode->i_ino)) {
			btab.magic1 = 0;
			btab.magic2 = 0;
			btab.iv = dir_iv;
			btab.ino = old_inode->i_ino;

			stegfs_encrypt_cbc2(old_inode->i_sb,
					    STEGFS_INO_LVL(old_inode->i_ino),
					    dir_bh->b_blocknr, dir_bufp,
					    dir_bh->b_data, dir_bh->b_size,
					    &btab.iv);
			btab.bchecksum = stegfs_chksum(dir_bh->b_data,
						       dir_bh->b_size);
			stegfs_encrypt_btab2(old_inode->i_sb,
					     STEGFS_INO_LVL(old_inode->i_ino),
					     dir_bh->b_blocknr, &btab, &btab2);
			stegfs_put_btab(old_inode->i_sb,
					dir_bh->b_blocknr, &btab2);
		}
		mark_buffer_dirty(dir_bh, 1);
		old_dir->i_nlink--;
		mark_inode_dirty(old_dir);
		if (new_inode) {
			new_inode->i_nlink--;
			mark_inode_dirty(new_inode);
		} else {
			new_dir->i_nlink++;
			new_dir->u.stegfs_i.i_flags &= ~EXT2_BTREE_FL;
			mark_inode_dirty(new_dir);
		}
	}
	mark_buffer_dirty(old_bh,  1);
	if (IS_SYNC(old_dir)) {
		ll_rw_block (WRITE, 1, &old_bh);
		wait_on_buffer (old_bh);
	}
	mark_buffer_dirty(new_bh, 1);
	if (IS_SYNC(new_dir)) {
		ll_rw_block (WRITE, 1, &new_bh);
		wait_on_buffer (new_bh);
	}
	/* FIXME: ought to be writing out btab entries if hid? */
	brelse (dir_bh);
	brelse (old_bh);
	brelse (new_bh);
	if (!STEGFS_IS_HID_INO(old_dir->i_ino))
		return 0;

	kfree(dir_bufp);
	btab.magic1 = 0;
	btab.magic2 = 0;
	btab.ino = old_dir->i_ino;
	for (i=0; i<old_dir->u.stegfs_i.i_bcopies; i++) {
		old_bh = stegfs_getblkn(old_dir, blocknum_old, &btab.iv,
					1, &err, i, 1, &cerr);

		stegfs_encrypt_cbc2(old_dir->i_sb,
				    STEGFS_INO_LVL(old_dir->i_ino),
				    old_bh->b_blocknr, old_bufp,
				    old_bh->b_data, old_bh->b_size, &btab.iv);
		mark_buffer_uptodate(old_bh, 1);
		mark_buffer_dirty(old_bh, 1);
		btab.bchecksum = stegfs_chksum(old_bh->b_data, old_bh->b_size);
		if (IS_SYNC(old_dir)) {
			ll_rw_block(WRITE, 1, &old_bh);
			wait_on_buffer(old_bh);
			if (!buffer_uptodate(old_bh)) {
				brelse (old_bh);
				kfree(old_bufp);
				return -EIO;
			}
		}
		stegfs_encrypt_btab2(old_dir->i_sb,
				     STEGFS_INO_LVL(old_dir->i_ino),
				     old_bh->b_blocknr, &btab, &btab2);
		stegfs_put_btab(old_dir->i_sb,
				old_bh->b_blocknr, &btab2);
		brelse(old_bh);

	}
	if (old_bufp != new_bufp)
		kfree(old_bufp);
	old_bufp = NULL;

	for (i=0; i<new_dir->u.stegfs_i.i_bcopies; i++) {
		new_bh = stegfs_getblkn(new_dir, blocknum_new, &btab.iv,
					1, &err, i, 1, &cerr);

		stegfs_encrypt_cbc2(new_dir->i_sb,
				    STEGFS_INO_LVL(new_dir->i_ino),
				    new_bh->b_blocknr, new_bufp,
				    new_bh->b_data, new_bh->b_size, &btab.iv);
		mark_buffer_uptodate(new_bh, 1);
		mark_buffer_dirty(new_bh, 1);
		btab.bchecksum = stegfs_chksum(new_bh->b_data, new_bh->b_size);
		if (IS_SYNC(new_dir)) {
			ll_rw_block(WRITE, 1, &new_bh);
			wait_on_buffer(new_bh);
			if (!buffer_uptodate(new_bh)) {
				brelse (new_bh);
				kfree(new_bufp);
				return -EIO;
			}
		}
		stegfs_encrypt_btab2(new_dir->i_sb,
				     STEGFS_INO_LVL(new_dir->i_ino),
				     new_bh->b_blocknr, &btab, &btab2);
		stegfs_put_btab(new_dir->i_sb,
				new_bh->b_blocknr, &btab2);
		brelse(new_bh);

	}
	kfree(new_bufp);

	return 0;

end_rename:
	brelse (dir_bh);
	brelse (old_bh);
	brelse (new_bh);
	return retval;
}
