/*
 *  linux/include/linux/stegfs_fs.h
 *
 * Copyright (C) 1999-2001
 * Andrew McDonald (andrew@mcdonald.org.uk)
 *
 *  from
 *
 *  linux/include/linux/ext2_fs.h
 *
 *  Copyright (C) 1992, 1993, 1994, 1995
 *  Remy Card (card@masi.ibp.fr)
 *  Laboratoire MASI - Institut Blaise Pascal
 *  Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/include/linux/minix_fs.h
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#ifndef _LINUX_STEGFS_FS_H
#define _LINUX_STEGFS_FS_H

/*
 * Use most stuff from standard ext2 filesystem
 */
#include <linux/ext2_fs.h>


/*
 * The stegfs filesystem constants/structures
 */

/*
 * Define STEGFS_DEBUG to produce debug messages
 */
/*#define STEGFS_DEBUG*/

/*
 * The stegfs file system version
 */
#include "../version.h"

/*
 * Debug code
 */
#ifdef STEGFS_DEBUG
#	define stegfs_debug(f, a...)	{ \
					printk ("stegfs DEBUG (%s, %d): %s:", \
						__FILE__, __LINE__, __FUNCTION__); \
				  	printk (f, ## a); \
					}
#else
#	define stegfs_debug(f, a...)	/**/
#endif


#define STEGFS_MAX_LEVELS        16
#define STEGFS_MAX_CONTEXTS      15
#define STEGFS_INO_MASK          0x40000000
#define STEGFS_IS_HID_INO(n)     ((n) & STEGFS_INO_MASK) 
#define STEGFS_MAX_INO_COPIES    28
#define STEGFS_MAX_BLOCK_COPIES  14
#define STEGFS_HID_ROOT_INO      0x40000000
#define STEGFS_LVL_SHIFT         26
#define STEGFS_INO_BMASK         0x01FFFFFF
#define STEGFS_INO_LVL(n)        (((n)>>STEGFS_LVL_SHIFT)&0xF)
#define STEGFS_INO_BATT          100
#define STEGFS_KEY_BYTES         16
#define STEGFS_PASS_LEN          256

/* #define EXT2_MOUNT_STEGFS        0x?? */


#ifdef __KERNEL__
#define STEGFS_ADDR_PER_BLOCK_BITS(s)     ((s)->u.stegfs_sb.s_addr_per_block_bits)
#define STEGFS_INODE_SIZE(s)              ((s)->u.stegfs_sb.s_inode_size)
#define STEGFS_FIRST_INO(s)               ((s)->u.stegfs_sb.s_first_ino)
#else
#define STEGFS_INODE_SIZE(s)      (((s)->s_rev_level == EXT2_GOOD_OLD_REV) ? \
                                 EXT2_GOOD_OLD_INODE_SIZE : \
                                 (s)->s_inode_size)
#define STEGFS_FIRST_INO(s)       (((s)->s_rev_level == EXT2_GOOD_OLD_REV) ? \
                                 EXT2_GOOD_OLD_FIRST_INO : \
                                 (s)->s_first_ino)
#endif

#ifdef __KERNEL__
# define STEGFS_FRAG_SIZE(s)              ((s)->u.stegfs_sb.s_frag_size)
# define STEGFS_FRAGS_PER_BLOCK(s)        ((s)->u.stegfs_sb.s_frags_per_block)
#else
# define STEGFS_FRAG_SIZE(s)              (EXT2_MIN_FRAG_SIZE << (s)->s_log_frag_size)
# define STEGFS_FRAGS_PER_BLOCK(s)        (EXT2_BLOCK_SIZE(s) / EXT2_FRAG_SIZE(s))
#endif

#ifdef __KERNEL__
# define STEGFS_BLOCKS_PER_GROUP(s)       ((s)->u.stegfs_sb.s_blocks_per_group)
# define STEGFS_DESC_PER_BLOCK(s)         ((s)->u.stegfs_sb.s_desc_per_block)
# define STEGFS_INODES_PER_GROUP(s)       ((s)->u.stegfs_sb.s_inodes_per_group)
# define STEGFS_DESC_PER_BLOCK_BITS(s)    ((s)->u.stegfs_sb.s_desc_per_block_bits)
#else
# define STEGFS_BLOCKS_PER_GROUP(s)       ((s)->s_blocks_per_group)
# define STEGFS_DESC_PER_BLOCK(s)         (EXT2_BLOCK_SIZE(s) / sizeof (struct ext2_group_desc))
# define STEGFS_INODES_PER_GROUP(s)       ((s)->s_inodes_per_group)
#endif


#ifdef __KERNEL__
#define STEGFS_SB(sb)     (&((sb)->u.stegfs_sb))
#else
/* Assume that user mode programs are passing in an stegfs superblock, not
 * a kernel struct super_block.  This will allow us to call the feature-test
 * macros from user land. */
#define STEGFS_SB(sb)     (sb)
#endif

#define STEGFS_HAS_COMPAT_FEATURE(sb,mask)                        \
        ( STEGFS_SB(sb)->s_feature_compat & (mask) )
#define STEGFS_HAS_RO_COMPAT_FEATURE(sb,mask)                     \
        ( STEGFS_SB(sb)->s_feature_ro_compat & (mask) )
#define STEGFS_HAS_INCOMPAT_FEATURE(sb,mask)                      \
        ( STEGFS_SB(sb)->s_feature_incompat & (mask) )


/*
 * Structure of an entry in the block table
 */
struct stegfs_btable {
	__u32 magic1;
	__u16 magic2;
	__u16 iv;
	__u32 bchecksum;
	__u32 ino;
};

struct stegfs_btable_head {
	__u32 magic;
	__u16 version;
	__u16 pad1;
	__u32 pad2;
	__u32 pad3;
};

#define STEGFS_BTAB_MAGIC 0x62615462

/*
 * Structure of a stegfs inode on the disk
 */
struct stegfs_inode {
        __u16   i_mode;         /* File mode */
        __u16   i_uid;          /* Owner Uid */
        __u32   i_size;         /* Size in bytes */
        __u32   i_atime;        /* Access time */
        __u32   i_ctime;        /* Creation time */
        __u32   i_mtime;        /* Modification time */
        __u32   i_dtime;        /* Deletion Time */
        __u16   i_gid;          /* Group Id */
        __u16   i_links_count;  /* Links count */
        __u32   i_blocks;       /* Blocks count */
        __u32   i_flags;        /* File flags */
        union {
                struct {
                        __u32  l_i_reserved1;
                } linux1;
                struct {
                        __u32  h_i_translator;
                } hurd1;
                struct {
                        __u32  m_i_reserved1;
                } masix1;
        } osd1;                         /* OS dependent 1 */
        /*__u32 i_block[EXT2_N_BLOCKS];*//* Pointers to blocks */
        __u32   i_generation;      /* File version (for NFS) */
        __u32   i_file_acl;     /* File ACL */
        __u32   i_dir_acl;      /* Directory ACL */
        __u32   i_faddr;        /* Fragment address */
        union {
                struct {
                        __u8    l_i_frag;       /* Fragment number */
                        __u8    l_i_fsize;      /* Fragment size */
                        __u16   i_pad1;
                        __u32   l_i_reserved2[2];
                } linux2;
                struct {
                        __u8    h_i_frag;       /* Fragment number */
                        __u8    h_i_fsize;      /* Fragment size */
                        __u16   h_i_mode_high;
                        __u16   h_i_uid_high;
                        __u16   h_i_gid_high;
                        __u32   h_i_author;
                } hurd2;
                struct {
                        __u8    m_i_frag;       /* Fragment number */
                        __u8    m_i_fsize;      /* Fragment size */
                        __u16   m_pad1;
                        __u32   m_i_reserved2[2];
                } masix2;
        } osd2;                         /* OS dependent 2 */
        __u8    i_icopies;        /* Number of copies of inode */
        __u8    i_bcopies;        /* Number of copies of blocks */
	__u16   i_pad0;
	__u32   i_inode[STEGFS_MAX_INO_COPIES];
        __u32   i_block[EXT2_N_BLOCKS *
		       STEGFS_MAX_BLOCK_COPIES];   /* Pointers to blocks */
};


#define STEGFS_GB_OK         0
#define STEGFS_GB_NOCOPYN    1
#define STEGFS_GB_NOVALID    2
#define STEGFS_GB_CREATED    3
#define STEGFS_GB_RECREATED  4
#define STEGFS_GB_ANYCOPY    99

/* a crypto algorithm instance must fill out one of these */
/* only 128-bit block ciphers are available */
struct stegfs_cipher {
	char *id;
	char * (*set_key) (const char *, const unsigned int);
	void (*encrypt) (const char *, const char *, char *);
	void (*decrypt) (const char *l_key, const char *, char *out_blk);
	int keys_size;
};

/*
 * ioctl commands
 */

struct stegfs_levelset {
	int level;
	char passw[256];
};

struct stegfs_setlevel {
	int context;
	int level;
	int flags;
	char passw[STEGFS_PASS_LEN];
};

#define STEGFS_IOC_GETICOPIES		_IOR('i', 1, char)
#define STEGFS_IOC_SETICOPIES		_IOW('i', 2, char)
#define STEGFS_IOC_GETBCOPIES		_IOR('b', 1, char)
#define STEGFS_IOC_SETBCOPIES		_IOW('b', 2, char)
#define STEGFS_IOC_GETLEVEL		_IOR('l', 1, int)
#define STEGFS_IOC_SETLEVEL		_IOW('l', 2, struct stegfs_levelset)
#define STEGFS_IOC_GETLEVELS		_IOR('s', 1, int)
#define STEGFS_IOC_SETLEVELS		_IOW('s', 2, struct stegfs_setlevel)
#define EBADPASS	150
#define ENOBTAB		151

#ifdef __KERNEL__

/* Filesize hard limits for 64-bit file offsets */
extern long long stegfs_max_sizes[];

/*
 * Function prototypes
 */

/* acl.c */
extern int stegfs_permission (struct inode *, int);

/* balloc.c */
extern int stegfs_group_sparse(int group);
extern int stegfs_new_block (const struct inode *, unsigned long,
			    __u32 *, __u32 *, int *);
extern void stegfs_free_blocks (const struct inode *, unsigned long,
			       unsigned long);
extern unsigned long stegfs_count_free_blocks (struct super_block *);
extern void stegfs_check_blocks_bitmap (struct super_block *);
extern struct ext2_group_desc * stegfs_get_group_desc(struct super_block * sb,
						     unsigned int block_group,
						     struct buffer_head ** bh);
extern int stegfs_rand_new_block(struct super_block *);

/* bitmap.c */
extern unsigned long stegfs_count_free (struct buffer_head *, unsigned);

/* dir.c */
extern int stegfs_check_dir_entry (const char *, struct inode *,
				  struct ext2_dir_entry_2 *,
				  char *,
				  unsigned long);

/* file.c */
extern ssize_t stegfs_file_read(struct file * filp, char * buf,
				size_t count, loff_t *ppos);
extern void stegfs_remove_suid (struct inode *);
/* fsync.c */
extern int stegfs_sync_file (struct file *, struct dentry *);

/* ialloc.c */
extern struct inode * stegfs_new_inode (const struct inode *, int, int *);
extern void stegfs_free_inode (struct inode *);
extern unsigned long stegfs_count_free_inodes (struct super_block *);
extern void stegfs_check_inodes_bitmap (struct super_block *);

/* inode.c */
extern int stegfs_bmap (struct inode *, int);

extern struct buffer_head * stegfs_getblk (struct inode *, long,
					   unsigned short *, int, int *);
extern struct buffer_head * stegfs_getblkn (struct inode *, long,
					    unsigned short *, int,
					    int *, int, int, int *);
extern struct buffer_head * stegfs_bread (struct inode *, int,
					  unsigned short *, int, int *);

extern int stegfs_getcluster (struct inode * inode, long block);
extern void stegfs_read_inode (struct inode *);
extern void stegfs_write_inode (struct inode *);
extern void stegfs_put_inode (struct inode *);
extern void stegfs_clear_inode (struct inode * inode);
extern void stegfs_delete_inode (struct inode *);
extern int stegfs_sync_inode (struct inode *);
extern int stegfs_notify_change(struct dentry *, struct iattr *);
extern void stegfs_discard_prealloc (struct inode *);

/* ioctl.c */
extern int stegfs_ioctl (struct inode *, struct file *, unsigned int,
			unsigned long);

/* mmap.c */
extern int stegfs_readpage(struct file *, struct page *);

/* namei.c */
extern void stegfs_release (struct inode *, struct file *);
extern struct dentry *stegfs_lookup (struct inode *, struct dentry *);
extern int stegfs_create (struct inode *,struct dentry *,int);
extern int stegfs_mkdir (struct inode *,struct dentry *,int);
extern int stegfs_rmdir (struct inode *,struct dentry *);
extern int stegfs_unlink (struct inode *,struct dentry *);
extern int stegfs_symlink (struct inode *,struct dentry *,const char *);
extern int stegfs_link (struct dentry *, struct inode *, struct dentry *);
extern int stegfs_mknod (struct inode *, struct dentry *, int, int);
extern int stegfs_rename (struct inode *, struct dentry *,
			 struct inode *, struct dentry *);

/* super.c */
extern void stegfs_error (struct super_block *, const char *, const char *, ...)
	__attribute__ ((format (printf, 3, 4)));
extern NORET_TYPE void stegfs_panic (struct super_block *, const char *,
				    const char *, ...)
	__attribute__ ((NORET_AND format (printf, 3, 4)));
extern void stegfs_warning (struct super_block *, const char *, const char *, ...)
	__attribute__ ((format (printf, 3, 4)));
extern void stegfs_put_super (struct super_block *);
extern void stegfs_write_super (struct super_block *);
extern int stegfs_remount (struct super_block *, int *, char *);
extern struct super_block * stegfs_read_super (struct super_block *,void *,int);
extern int init_stegfs_fs(void);
extern int stegfs_statfs (struct super_block *, struct statfs *, int);

/* truncate.c */
extern void stegfs_truncate (struct inode *);

/* level.c */
extern int stegfs_open_level(struct super_block *, int,int, char *);
extern int stegfs_close_level(struct super_block *, int);
extern int stegfs_get_btab(struct super_block *, u32, struct stegfs_btable *);
extern int stegfs_put_btab(struct super_block *, u32, struct stegfs_btable *);
extern unsigned int stegfs_chksum(unsigned char *, int);
extern char *stegfs_make_key(struct super_block *, int, unsigned int);
extern void stegfs_encrypt_btab(struct super_block *, int, unsigned int,
				struct stegfs_btable *);
extern void stegfs_encrypt_btab2(struct super_block *, int, unsigned int,
				 struct stegfs_btable *, struct stegfs_btable *);
extern void stegfs_decrypt_btab(struct super_block *, int, unsigned int,
				struct stegfs_btable *);
extern void stegfs_decrypt_btab2(struct super_block *, int, unsigned int,
				 struct stegfs_btable *, struct stegfs_btable *);
extern void stegfs_encrypt_cbc(struct super_block *, int, unsigned int,
			       char *, int, unsigned short *);
extern void stegfs_encrypt_cbc2(struct super_block *, int, unsigned int,
				char *, char *, int, unsigned short *);
extern void stegfs_decrypt_cbc(struct super_block *, int, unsigned int,
			       char *, int, unsigned short);
extern void stegfs_decrypt_cbc2(struct super_block *, int, unsigned int,
				char *, char *, int, unsigned short);
extern void stegfs_decrypt_slkey(struct super_block *, int, int,
				 char *, char *);

/* aes.c */
extern char *stegfs_aes_set_key(const char *cin_key,
				const unsigned int key_len);
extern void stegfs_aes_encrypt(const char *cl_key,
			       const char *cin_blk, char *cout_blk);
extern void stegfs_aes_decrypt(const char *cl_key,
			       const char *cin_blk, char *cout_blk);

/* serpent.c */
extern char *stegfs_serpent_set_key(const char *cin_key,
				    const unsigned int key_len);
extern void stegfs_serpent_encrypt(const char *cl_key,
				   const char *cin_blk, char *cout_blk);
extern void stegfs_serpent_decrypt(const char *cl_key,
				   const char *cin_blk, char *cout_blk);
/* nullenc.c */
extern char *stegfs_nullenc_set_key(const char *cin_key,
				    const unsigned int key_len);
extern void stegfs_nullenc_encrypt(const char *cl_key,
				   const char *cin_blk, char *cout_blk);
extern void stegfs_nullenc_decrypt(const char *cl_key,
				   const char *cin_blk, char *cout_blk);
/* rc6.c */
extern char *stegfs_rc6_set_key(const char *cin_key,
				const unsigned int key_len);
extern void stegfs_rc6_encrypt(const char *cl_key,
			       const char *cin_blk, char *cout_blk);
extern void stegfs_rc6_decrypt(const char *cl_key,
			       const char *cin_blk, char *cout_blk);

/* twofish.c */
extern char *stegfs_twofish_set_key(const char *cin_key,
				    const unsigned int key_len);
extern void stegfs_twofish_encrypt(const char *cl_key,
				   const char *cin_blk, char *cout_blk);
extern void stegfs_twofish_decrypt(const char *cl_key,
				   const char *cin_blk, char *cout_blk);

/* mars.c */
extern char *stegfs_mars_set_key(const char *cin_key,
				 const unsigned int key_len);
extern void stegfs_mars_encrypt(const char *cl_key,
				const char *cin_blk, char *cout_blk);
extern void stegfs_mars_decrypt(const char *cl_key,
				const char *cin_blk, char *cout_blk);

/*
 * Inodes and files operations
 */

/* dir.c */
extern struct inode_operations stegfs_dir_inode_operations;

/* file.c */
extern struct inode_operations stegfs_file_inode_operations;

/* symlink.c */
extern struct inode_operations stegfs_symlink_inode_operations;

#endif	/* __KERNEL__ */

#endif	/* _LINUX_STEGFS_FS_H */
