/*
 *  linux/fs/stegfs/truncate.c
 *
 * Copyright (C) 1999-2001
 * Andrew McDonald (andrew@mcdonald.org.uk)
 *
 *  from
 *
 *  linux/fs/ext2/truncate.c
 *
 *  Copyright (C) 1992, 1993, 1994, 1995
 *  Remy Card (card@masi.ibp.fr)
 *  Laboratoire MASI - Institut Blaise Pascal
 *  Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/fs/minix/truncate.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  Big-endian to little-endian byte-swapping/bitmaps by
 *        David S. Miller (davem@caip.rutgers.edu), 1995
 *
 *  General cleanup and race fixes, wsh, 1998
 */

/*
 * Real random numbers for secure rm added 94/02/18
 * Idea from Pierre del Perugia <delperug@gla.ecoledoc.ibp.fr>
 */

#include "fs.h"
#include "stegfs_fs.h"

#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/locks.h>
#include <linux/string.h>
#include <linux/slab.h>

/*
 * Macros to return the block number for the inode size and offset.
 * Currently we always hold the inode semaphore during truncate, so
 * there's no need to test for changes during the operation.
 */
#define DIRECT_BLOCK(inode) \
	((inode->i_size + inode->i_sb->s_blocksize - 1) / \
			  inode->i_sb->s_blocksize)
#define INDIRECT_BLOCK(inode,offset) ((int)DIRECT_BLOCK(inode) - offset)
#define DINDIRECT_BLOCK(inode,offset) \
	(INDIRECT_BLOCK(inode,offset) / addr_per_block)
#define TINDIRECT_BLOCK(inode,offset) \
	(INDIRECT_BLOCK(inode,offset) / (addr_per_block*addr_per_block))

/*
 * Truncate has the most races in the whole filesystem: coding it is
 * a pain in the a**. Especially as I don't do any locking...
 *
 * The code may look a bit weird, but that's just because I've tried to
 * handle things like file-size changes in a somewhat graceful manner.
 * Anyway, truncating a file at the same time somebody else writes to it
 * is likely to result in pretty weird behaviour...
 *
 * The new code handles normal truncates (size = 0) as well as the more
 * general case (size = XXX). I hope.
 *
 *
 * Truncate operations have been rewritten to avoid various races. The
 * previous code was allowing blocking operations to precede a call to
 * bforget(), possible allowing the buffer to be used again.
 *
 * We now ensure that b_count == 1 before calling bforget() and that the
 * parent buffer (if any) is unlocked before clearing the block pointer.
 * The operations are always performed in this order:
 *	(1) Make sure that the parent buffer is unlocked.
 *	(2) Use find_buffer() to find the block buffer without blocking,
 *	    and set 'retry' if the buffer is locked or b_count > 1.
 *	(3) Clear the block pointer in the parent (buffer or inode).
 *	(4) Update the inode block count and mark the inode dirty.
 *	(5) Forget the block buffer, if any. This call won't block, as
 *	    we know the buffer is unlocked from (2).
 *	(6) If the block pointer is in a (parent) buffer, mark the buffer
 *	    dirty. (Note that this can block on a loop device.)
 *	(7) Accumulate the blocks to free and/or update the block bitmap.
 *	    (This operation will frequently block.)
 *
 * The requirement that parent buffers be unlocked follows from the general
 * principle of not modifying a buffer that may be undergoing I/O. With the
 * the present kernels there's no problem with modifying a locked inode, as
 * the I_DIRTY bit is cleared before setting I_LOCK.
 *		-- WSH, 1998
 */
/* *** FIXME: need to check btab entries before doing stuff *** */
/*
 * Check whether any of the slots in an indirect block are
 * still in use, and if not free the block.
 */
static int check_block_empty(struct inode *inode, struct buffer_head *bh,
			     char *bufplain, unsigned short bhiv,
			     u32 *p, struct buffer_head *ind_bh, char *ind_bp)
{
	int addr_per_block = EXT2_ADDR_PER_BLOCK(inode->i_sb);
	u32 * ind;
	int i, retry;

	if (STEGFS_IS_HID_INO(inode->i_ino)) {
		ind = (u32 *) bufplain;
	}
	else
		ind = (u32 *) bh->b_data;

	/* Make sure both buffers are unlocked */
	do {
		retry = 0;
		if (buffer_locked(bh)) {
			__wait_on_buffer(bh);
			retry = 1;
		}
		if (ind_bh && buffer_locked(ind_bh)) {
			__wait_on_buffer(ind_bh);
			retry = 1;
		}
	} while (retry);

	for (i = 0; i < addr_per_block; i++)
		if (*(ind++))
			goto in_use;

	if (bh->b_count == 1) {
		int tmp = le32_to_cpu(*p);
		*p = 0;
		inode->i_blocks -= (inode->i_sb->s_blocksize / 512);
		mark_inode_dirty(inode);
		/*
		 * Forget the buffer, then mark the parent buffer dirty.
		 */
		bforget(bh);
		if (ind_bh) {
			/* FIXME: this here? */
/*
			if (STEGFS_IS_HID_INO(inode->i_ino))
				stegfs_encrypt_cbc2(inode->i_sb,
						    STEGFS_INO_LVL(inode->i_ino),
						    ind_bh->b_blocknr,
						    ind_bp,
						    ind_bh->b_data,
						    ind_bh->b_size, &iv??);
*/
			mark_buffer_dirty(ind_bh, 1);
		}
		stegfs_free_blocks (inode, tmp, 1);
		goto out;
	}
	retry = 1;

in_use:
	if (STEGFS_IS_HID_INO(inode->i_ino)) {
		struct stegfs_btable btab;
		btab.iv = bhiv;
		stegfs_encrypt_cbc2(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    bh->b_blocknr,
				    bufplain,
				    bh->b_data, bh->b_size, &btab.iv);
		btab.magic1 = 0;
		btab.magic2 = 0;
		btab.ino = inode->i_ino;
		btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
		stegfs_encrypt_btab(inode->i_sb, STEGFS_INO_LVL(inode->i_ino),
				    bh->b_blocknr, &btab);
		stegfs_put_btab(inode->i_sb, bh->b_blocknr, &btab);
		kfree(bufplain);
		mark_buffer_dirty(bh, 1);
	}

	if (IS_SYNC(inode) && buffer_dirty(bh)) {
		ll_rw_block (WRITE, 1, &bh);
		wait_on_buffer (bh);
	}
	brelse (bh);

out:
	return retry;
}

static int trunc_direct (struct inode * inode)
{
	struct buffer_head * bh;
	int i, j, retry = 0;
	unsigned long block_to_free = 0, free_count = 0;
	int blocks = inode->i_sb->s_blocksize / 512;
	int direct_block = DIRECT_BLOCK(inode);

	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		for (i = direct_block ; i < EXT2_NDIR_BLOCKS ; i++) {
			u32 * p = inode->u.stegfs_i.i_x->i_data + i;
			int tmp = le32_to_cpu(*p);

			if (!tmp)
				continue;

			bh = find_buffer(inode->i_dev, tmp,
					 inode->i_sb->s_blocksize);
			if (bh) {
				bh->b_count++;
				if(bh->b_count != 1 || buffer_locked(bh)) {
					brelse(bh);
					retry = 1;
					continue;
				}
			}

			*p = 0;
			inode->i_blocks -= blocks;
			mark_inode_dirty(inode);
			bforget(bh);

			/* accumulate blocks to free if they're contiguous */
			if (free_count == 0)
				goto free_this;
			else if (block_to_free == tmp - free_count)
				free_count++;
			else {
				stegfs_free_blocks (inode, block_to_free, free_count);
			free_this:
				block_to_free = tmp;
				free_count = 1;
			}
		}
		if (free_count > 0)
			stegfs_free_blocks (inode, block_to_free, free_count);
		return retry;
	}
	/* else STEGFS_IS_HID_INO */
	for (i = direct_block ; i < EXT2_NDIR_BLOCKS ; i++) {

		for (j = 0; j < inode->u.stegfs_i.i_bcopies; j++) {
			u32 * p = inode->u.stegfs_i.i_x->i_data + i
				+ j * EXT2_N_BLOCKS;
			int tmp = le32_to_cpu(*p);

			if (!tmp)
				goto next_block;

			bh = find_buffer(inode->i_dev, tmp,
					 inode->i_sb->s_blocksize);
			if (bh) {
				bh->b_count++;
				if(bh->b_count != 1 || buffer_locked(bh)) {
					brelse(bh);
					retry = 1;
					goto next_block;
				}
			}
			bforget(bh);
		}
		inode->i_blocks -= blocks;
		mark_inode_dirty(inode);

		for (j = 0; j < inode->u.stegfs_i.i_bcopies; j++) {
			u32 * p = inode->u.stegfs_i.i_x->i_data + i
				+ j * EXT2_N_BLOCKS;
			int tmp = le32_to_cpu(*p);
			*p = 0;

			stegfs_free_blocks (inode, tmp, 1);
		}
next_block:
	}
	return retry;
}

static int trunc_indirect (struct inode * inode, int offset, u32 * p,
			struct buffer_head *dind_bh, char *dind_bp)
{
	struct buffer_head * ind_bh;
	char * ind_bp;
	int i, tmp, retry = 0;
	unsigned long block_to_free = 0, free_count = 0;
	int indirect_block, addr_per_block, blocks;
	struct stegfs_btable btab;

	tmp = le32_to_cpu(*p);
	if (!tmp)
		return 0;
	ind_bh = bread (inode->i_dev, tmp, inode->i_sb->s_blocksize);
	if (tmp != le32_to_cpu(*p)) {
		brelse (ind_bh);
		return 1;
	}
	/* A read failure? Report error and clear slot (should be rare). */ 
	if (!ind_bh) {
		stegfs_error(inode->i_sb, "trunc_indirect",
			"Read failure, inode=%ld, block=%d",
			inode->i_ino, tmp);
		*p = 0;
		if (dind_bh) {
/*
No point doing encryption here?
			if (STEGFS_IS_HID_INO(inode->i_ino)) {
				stegfs_encrypt_cbc2(inode->i_sb,
						    STEGFS_INO_LVL(inode->i_ino),
						    dind_bh->b_blocknr,
						    dind_bp,
						    dind_bh->b_data,
						    dind_bh->b_size);
				/ * only redo btab when we brelse * /
			}
*/
			if (!STEGFS_IS_HID_INO(inode->i_ino))
				mark_buffer_dirty(dind_bh, 1);
		}
		else
			mark_inode_dirty(inode);
		return 0;
	}

	blocks = inode->i_sb->s_blocksize / 512;
	addr_per_block = EXT2_ADDR_PER_BLOCK(inode->i_sb);
	indirect_block = INDIRECT_BLOCK(inode, offset);
	if (indirect_block < 0)
		indirect_block = 0;
	if (STEGFS_IS_HID_INO(inode->i_ino)) {
		stegfs_get_btab(inode->i_sb, tmp, &btab);
		stegfs_decrypt_btab(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    tmp, &btab);
		if (btab.magic1 != 0 || btab.magic2 != 0 ||
		    btab.ino != inode->i_ino ||
		    btab.bchecksum != stegfs_chksum(ind_bh->b_data,
						   ind_bh->b_size)) {
			stegfs_debug("bad ind block\n");
			brelse(ind_bh);
			return 0;
		}
		ind_bp = kmalloc(ind_bh->b_size, GFP_KERNEL);
		stegfs_decrypt_cbc2(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    ind_bh->b_blocknr,
				    ind_bh->b_data,
				    ind_bp,
				    ind_bh->b_size, btab.iv);
	}
	else {
		btab.iv = 0;
		ind_bp = ind_bh->b_data;
	}
	for (i = indirect_block ; i < addr_per_block ; i++) {
		u32 * ind = i + (u32 *) ind_bp;
		struct buffer_head * bh;

		wait_on_buffer(ind_bh);
		tmp = le32_to_cpu(*ind);
		if (!tmp)
			continue;
		/*
		 * Use find_buffer so we don't block here.
		 */
		bh = find_buffer(inode->i_dev, tmp, inode->i_sb->s_blocksize);
		if (bh) {
			bh->b_count++;
			if (bh->b_count != 1 || buffer_locked(bh)) {
				brelse (bh);
				retry = 1;
				continue;
			}
		}

		*ind = 0;
		inode->i_blocks -= blocks;
		mark_inode_dirty(inode);
		bforget(bh);
/*
		if (STEGFS_IS_HID_INO(inode->i_ino))
			stegfs_encrypt_cbc2(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
					    ind_bh->b_blocknr,
					    ind_bp,
					    ind_bh->b_data,
					    ind_bh->b_size);
*/
		if (!STEGFS_IS_HID_INO(inode->i_ino))
			mark_buffer_dirty(ind_bh, 1);

		/* accumulate blocks to free if they're contiguous */
		if (free_count == 0)
			goto free_this;
		else if (block_to_free == tmp - free_count)
			free_count++;
		else {
			stegfs_free_blocks (inode, block_to_free, free_count);
		free_this:
			block_to_free = tmp;
			free_count = 1;
		}
	}
	if (free_count > 0)
		stegfs_free_blocks (inode, block_to_free, free_count);
	/*
	 * Check the block and dispose of the ind_bh buffer.
	 */
	retry |= check_block_empty(inode, ind_bh, ind_bp, btab.iv, p,
				   dind_bh, dind_bp);

	return retry;
}

static int trunc_dindirect (struct inode * inode, int offset, u32 * p,
			struct buffer_head * tind_bh, char * tind_bp)
{
	struct buffer_head * dind_bh;
	char * dind_bp;
	int i, tmp, retry = 0;
	int dindirect_block, addr_per_block;
	struct stegfs_btable btab;

	tmp = le32_to_cpu(*p);
	if (!tmp)
		return 0;
	dind_bh = bread (inode->i_dev, tmp, inode->i_sb->s_blocksize);
	if (tmp != le32_to_cpu(*p)) {
		brelse (dind_bh);
		return 1;
	}
	/* A read failure? Report error and clear slot (should be rare). */ 
	if (!dind_bh) {
		stegfs_error(inode->i_sb, "trunc_dindirect",
			"Read failure, inode=%ld, block=%d",
			inode->i_ino, tmp);
		*p = 0;
		if (tind_bh) {
/*
			if (STEGFS_IS_HID_INO(inode->i_ino))
				stegfs_encrypt_cbc2(inode->i_sb,
						    STEGFS_INO_LVL(inode->i_ino),
						    tind_bh->b_blocknr,
						    tind_bp,
						    tind_bh->b_data,
						    tind_bh->b_size);

*/
			if (!STEGFS_IS_HID_INO(inode->i_ino))
				mark_buffer_dirty(tind_bh, 1);
		}
		else
			mark_inode_dirty(inode);
		return 0;
	}

	addr_per_block = EXT2_ADDR_PER_BLOCK(inode->i_sb);
	dindirect_block = DINDIRECT_BLOCK(inode, offset);
	if (dindirect_block < 0)
		dindirect_block = 0;
	if (STEGFS_IS_HID_INO(inode->i_ino)) {
		stegfs_get_btab(inode->i_sb, tmp, &btab);
		stegfs_decrypt_btab(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    tmp, &btab);
		if (btab.magic1 != 0 || btab.magic2 != 0 ||
		    btab.ino != inode->i_ino ||
		    btab.bchecksum != stegfs_chksum(dind_bh->b_data,
						   dind_bh->b_size)) {
			stegfs_debug("bad dind block\n");
			brelse(dind_bh);
			return 0;
		}
		dind_bp = kmalloc(dind_bh->b_size, GFP_KERNEL);
		stegfs_decrypt_cbc2(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    dind_bh->b_blocknr,
				    dind_bh->b_data,
				    dind_bp,
				    dind_bh->b_size, btab.iv);
	}
	else {
		btab.iv = 0;
		dind_bp = dind_bh->b_data;
	}
	for (i = dindirect_block ; i < addr_per_block ; i++) {
		u32 * dind = i + (u32 *) dind_bp;

		retry |= trunc_indirect(inode,
					offset + (i * addr_per_block),
					dind, dind_bh, dind_bp);
	}
	/*
	 * Check the block and dispose of the dind_bh buffer.
	 */
	retry |= check_block_empty(inode, dind_bh, dind_bp, btab.iv, p,
				   tind_bh, tind_bp);

	return retry;
}

static int trunc_tindirect (struct inode * inode, int ncopy)
{
	u32 * p = inode->u.stegfs_i.i_x->i_data + EXT2_TIND_BLOCK +
		ncopy * EXT2_N_BLOCKS;
	struct buffer_head * tind_bh;
	char * tind_bp;
	int i, tmp, retry = 0;
	int tindirect_block, addr_per_block, offset;
	struct stegfs_btable btab;

	if (!(tmp = *p))
		return 0;
	tind_bh = bread (inode->i_dev, le32_to_cpu(tmp),
			 inode->i_sb->s_blocksize);
	if (tmp != *p) {
		brelse (tind_bh);
		return 1;
	}
	/* A read failure? Report error and clear slot (should be rare). */ 
	if (!tind_bh) {
		stegfs_error(inode->i_sb, "trunc_tindirect",
			"Read failure, inode=%ld, block=%d",
			inode->i_ino, le32_to_cpu(tmp));
		*p = 0;
		mark_inode_dirty(inode);
		return 0;
	}

	addr_per_block = EXT2_ADDR_PER_BLOCK(inode->i_sb);
	offset = EXT2_NDIR_BLOCKS + addr_per_block + 
		(addr_per_block * addr_per_block);
	tindirect_block = TINDIRECT_BLOCK(inode, offset);
	if (tindirect_block < 0)
		tindirect_block = 0;
	if (STEGFS_IS_HID_INO(inode->i_ino)) {
		stegfs_get_btab(inode->i_sb, le32_to_cpu(tmp), &btab);
		stegfs_decrypt_btab(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    le32_to_cpu(tmp), &btab);
		if (btab.magic1 != 0 || btab.magic2 != 0 ||
		    btab.ino != inode->i_ino ||
		    btab.bchecksum != stegfs_chksum(tind_bh->b_data,
						   tind_bh->b_size)) {
			stegfs_debug("bad tind block: copy %u\n", ncopy);
			brelse(tind_bh);
			return 0;
		}
		tind_bp = kmalloc(tind_bh->b_size, GFP_KERNEL);
		stegfs_decrypt_cbc2(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    tind_bh->b_blocknr,
				    tind_bh->b_data,
				    tind_bp,
				    tind_bh->b_size, btab.iv);
	}
	else {
		btab.iv = 0;
		tind_bp = tind_bh->b_data;
	}
	for (i = tindirect_block ; i < addr_per_block ; i++) {
		u32 * tind = i + (u32 *) tind_bp;

		retry |= trunc_dindirect(inode,
				offset + (i * addr_per_block * addr_per_block),
				tind, tind_bh, tind_bp);
	}
	/*
	 * Check the block and dispose of the tind_bh buffer.
	 */
	retry |= check_block_empty(inode, tind_bh, tind_bp, btab.iv,
				   p, NULL, NULL);

	return retry;
}
		
void stegfs_truncate (struct inode * inode)
{
	int err, cerr, offset, retry, i;
	unsigned short iv;
	int ncopies = STEGFS_IS_HID_INO(inode->i_ino) ?
		inode->u.stegfs_i.i_bcopies : 1;

	stegfs_debug("ino: %lu size: %lu\n", inode->i_ino, inode->i_size);

	if (!(S_ISREG(inode->i_mode) || S_ISDIR(inode->i_mode) ||
	    S_ISLNK(inode->i_mode)))
		return;
	if (IS_APPEND(inode) || IS_IMMUTABLE(inode))
		return;

	stegfs_remove_suid(inode);
	stegfs_discard_prealloc(inode);
	while (1) {
		retry = trunc_direct(inode);
		for (i=0; i<ncopies; i++)
			retry |= trunc_indirect (inode, 
				EXT2_IND_BLOCK,
				(u32 *) &inode->u.stegfs_i.i_x->i_data[EXT2_IND_BLOCK + i*EXT2_N_BLOCKS],
				NULL, NULL);
		for (i=0; i<ncopies; i++)
			retry |= trunc_dindirect (inode,
				EXT2_IND_BLOCK+EXT2_ADDR_PER_BLOCK(inode->i_sb),
				(u32 *)&inode->u.stegfs_i.i_x->i_data[EXT2_DIND_BLOCK + i*EXT2_N_BLOCKS],
				NULL, NULL);
		for (i=0; i<ncopies; i++)
			retry |= trunc_tindirect (inode, i);
		if (!retry)
			break;
		if (IS_SYNC(inode) && (inode->i_state & I_DIRTY))
			stegfs_sync_inode (inode);
		run_task_queue(&tq_disk);
		current->policy |= SCHED_YIELD;
		schedule ();
	}
	/*
	 * If the file is not being truncated to a block boundary, the
	 * contents of the partial block following the end of the file
	 * must be zeroed in case it ever becomes accessible again due
	 * to subsequent file growth.
	 */
	offset = inode->i_size & (inode->i_sb->s_blocksize - 1);
	if (offset) {
		struct buffer_head * bh;
		char * bufplain;
		struct stegfs_btable btab;
		bh = stegfs_bread (inode,
		     inode->i_size >> EXT2_BLOCK_SIZE_BITS(inode->i_sb),
				   &iv, 0, &err);
		if (bh) {
			if (!STEGFS_IS_HID_INO(inode->i_ino)) {
				memset (bh->b_data + offset, 0,
					inode->i_sb->s_blocksize - offset);
				mark_buffer_dirty (bh, 0);
				brelse (bh);
			}
			else {
				bufplain = kmalloc(bh->b_size, GFP_KERNEL);
				stegfs_decrypt_cbc2(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
					    bh->b_blocknr, bh->b_data,
					    bufplain, bh->b_size, iv);
				memset (bufplain + offset, 0,
					inode->i_sb->s_blocksize - offset);
				brelse(bh);
				for(i=0; i<inode->u.stegfs_i.i_bcopies; i++) {
					bh = stegfs_getblkn(inode,
					     inode->i_size >> EXT2_BLOCK_SIZE_BITS(inode->i_sb),
							    &btab.iv, 1, &err,
							    i, 1, &cerr);
					stegfs_encrypt_cbc2(inode->i_sb,
						 STEGFS_INO_LVL(inode->i_ino),
							    bh->b_blocknr,
							    bufplain,
							    bh->b_data,
							    bh->b_size,
							    &btab.iv);
					mark_buffer_dirty (bh, 0);
					btab.magic1 = 0;
					btab.magic2 = 0;
					btab.ino = inode->i_ino;
					btab.bchecksum =
						stegfs_chksum(bh->b_data,
							      bh->b_size);
					stegfs_encrypt_btab(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
							    bh->b_blocknr,
							    &btab);
					stegfs_put_btab(inode->i_sb,
							bh->b_blocknr,
							&btab);
					brelse(bh);
				}
				kfree(bufplain);
			}
		}
	}
	inode->i_mtime = inode->i_ctime = CURRENT_TIME;
	mark_inode_dirty(inode);
}
