/* StegFS - The Steganographic File System */
/* Copyright (c)1999 Andrew McDonald <andrew@mcdonald.org.uk> */

/* Activate additional levels in a StegFS file system. */
/* $Id: addlevel.c,v 1.2 1999/11/20 13:21:39 admcd Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "stegfs.h"
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <linux/major.h>
#include <string.h>
#include "sha1.h"

void printusage(char *);

int main(int argc, char* argv[]) {

	struct stegfs_fs *fssys;
	int i,j,c;
	char *passwds[STEGFS_MAX_LEVELS-1];
	FILE *devrand;
	struct stat statbuf;

	char cipherstr[35];
	int ic, bc;
	int currmax, ltoadd;
	char promptbuf[45];
	char *pass;
	char enckey[16];
	int slevel;

	fssys = (struct stegfs_fs *) malloc(sizeof(struct stegfs_fs));

	printf("StegFS - (c)1999 ADM\n");


	/* defaults */
	ic = 5;
	bc = 5;
	strncpy(cipherstr, "serpent", 35);

	while (1) {

		c = getopt(argc, argv, "n:c:i:b:");

		if (c == -1)
			break;

		switch (c)
		{
		case 'c':
			strncpy(cipherstr, optarg, 35);
			cipherstr[34] = '\0';
			break;
		case 'i':
			ic = atoi(optarg);
			break;
		case 'b':
			bc = atoi(optarg);
			break;

		default:
			printusage(argv[0]);
		}
	}

	if(optind != argc-4) {
		printf("Something not specified\n");
		printusage(argv[0]);
	}

	/* validate args */
	if (ic < 1 || ic > STEGFS_MAX_INO_COPIES) {
		printf("Inode copies: invalid value\n");
		exit(1);
	}
	if (bc < 1 || bc > STEGFS_MAX_BLOCK_COPIES) {
		printf("Block copies: invalid value\n");
		exit(1);
	}

	currmax = atoi(argv[optind+2]);
	ltoadd = atoi(argv[optind+3]);

	if (currmax < 0 || currmax+ltoadd > STEGFS_MAX_LEVELS-1) {
		printf("No. of levels: invalid value\n");
		exit(1);
	}

	/* Check random number source is what we think it is */
	i = stat(RAND_DEV, &statbuf);
	if (i == -1) {
		printf("Error accessing %s\n", RAND_DEV);
		exit(3);
	}
	if (!S_ISCHR(statbuf.st_mode) ||
	    major(statbuf.st_rdev) != MEM_MAJOR ||
	    minor(statbuf.st_rdev) != 9) {
		printf("WARNING!!! %s is not char device major %u minor 9!\n",
		       RAND_DEV, MEM_MAJOR);
	}

	if (!open_fs(fssys, argv[optind], argv[optind+1], cipherstr, 0)) {
		printf("Failed to open device or file\n");
		exit(2);
	}

	if(fssys->btabver != 1) {
		printf("Only for use on version 1 block table files.\n");
		printf("Aborted.\n");
		exit(2);
	}

	/* get passphrase for existing levels */
	sprintf(promptbuf, "Enter passphrase for existing level %u: ", currmax);
	pass = getpass(promptbuf);
	if(pass[0] == '\0') {
		printf("Aborted.\n");
		exit(1);
	}
	passwds[currmax-1] = malloc(strlen(pass)+1);
	strncpy(passwds[currmax-1], pass, strlen(pass)+1);
	memset(pass, 0, strlen(pass));

	/* Get keys */
	get_skeys(fssys, currmax, passwds[currmax-1]);
	if(!verify_skeys(fssys, currmax)) {
		printf("Invalid passphrase.\n");
		exit(1);
	}

	j = 0;
	for (i=currmax; i<currmax+ltoadd; i++) {
		sprintf(promptbuf, "Enter passphrase for new level %u: ", i+1);
		pass = getpass(promptbuf);
		if(pass[0] == '\0') {
			printf("Aborted.\n");
			exit(1);
		}
		passwds[i] = malloc(strlen(pass)+1);
		strncpy(passwds[i], pass, strlen(pass)+1);
		memset(pass, 0, strlen(pass));

		sprintf(promptbuf, "Re-enter passphrase for new level %u: ",
			i+1);
		pass = getpass(promptbuf);
		if (strcmp(pass,passwds[i]) != 0) {
			memset(pass, 0, strlen(pass));
			printf("Passphrases differ.\n");
			i--;
			j++;
			if (j > 2) {
				printf("Failed after %d attempts.\n", j);
				exit(1);
			}
		}
		else {
			memset(pass, 0, strlen(pass));
			j = 0;
		}
	}
	passwds[i] = NULL;

	slevel = currmax+ltoadd;
	if(ltoadd == 0)
		printf("This operation will simply destroy any existing "
		       "levels above level %d.\n", currmax);
	else {
		printf("Adding level");
		if(ltoadd == 1)
			printf(" %d.\n", currmax+1);
		else
			printf("s %d to %d.\n", currmax+1, currmax+ltoadd);
	}
	if (slevel == 14)
		printf("Level 15");
	else if (slevel < 14)
		printf("Levels %d to 15", slevel+1);
	if (slevel < 15)
		printf(" will not be accessible.\n");

	printf("     **********\n");
	printf("WARNING: This operation cannot be undone.\n");
	if (currmax == 14 && ltoadd == 1)
		printf("If level 15");
	if (currmax < 14)
		printf("If any of levels %d to 15", currmax+1);
	printf(" already exist, their contents will be permanently lost.\n");
	printf("     **********\n");

	printf("Are you sure you want to proceed? (y,n) ");
	if (getchar() != 'y') {
		printf("Aborted.\n");
		exit(1);
	}

	devrand = fopen(RAND_DEV, "r");
	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}
	fseek(fssys->btabfile,
	      fssys->e2fs->super->s_blocks_count*sizeof(struct stegfs_btable), SEEK_SET);
	fseek(fssys->btabfile,
	      .5*(currmax+1)*currmax*sizeof(struct stegfs_btable),
	      SEEK_CUR);

	/* Write new encrypted keys. */
	for (i=currmax; i<currmax+ltoadd; i++) {
		fssys->slkeys[i] = malloc(16);
		fread(fssys->slkeys[i], 16, 1, devrand);
		for (j=0; j<(i+1); j++) {
			encrypt_skey(fssys, passwds[i],
				     fssys->slkeys[j], enckey);
			fwrite(enckey, sizeof(enckey), 1, fssys->btabfile);
		}
	}
	/* Overwrite any further old keys. */
	for (i=currmax+ltoadd; i<STEGFS_MAX_LEVELS-1; i++) {
		for (j=0; j<(i+1); j++) {
			fread(enckey, 16, 1, devrand);
			fwrite(enckey, 16, 1, fssys->btabfile);
		}
	}
	fclose(devrand);

	if(ltoadd > 0) {
		printf("Writing root directories: ");
		for (i=currmax; i<currmax+ltoadd; i++) {
			mkroot(fssys, i+1, passwds[i], ic, bc);
		}
		printf("Done\n");
	}

	close_fs(fssys);
	printf("Done.\n");

	return 0;
}




void printusage(char *progname) {
	printf("\nUsage:\n");
	printf("%s /dev/fsdevice btabfile current_max_level levels_to_add\n"
	       "         [-c cipher] [-i inocopies] [-b blkcopies]\n",
	       basename(progname));
	printf("Note: levels_to_add is the number of additional levels to add.\n"
	       "      current_max_level + levels_to_add must be less than or\n"
	       "      equal to %d.\n",
	       STEGFS_MAX_LEVELS-1);
	exit(1);
}
