/* StegFS - The Steganographic File System */
/* Copyright (c)1999 Andrew McDonald <andrew@mcdonald.org.uk> */

/* Take an ext2 fs and make it ready for data hiding */
/* - Randomize contents of unused blocks
   - Create root inode(s)
   - Create stegfs information table
*/
/* $Id: mkstegfs.c,v 1.20 1999/11/20 13:22:03 admcd Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "stegfs.h"
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <linux/major.h>
#include <string.h>


void printusage(char *);
void initbtab(struct stegfs_fs *fs);


int main(int argc, char* argv[]) {

	struct stegfs_fs *fssys;
	int i,j,c;
	char *passwds[STEGFS_MAX_LEVELS-1];
	FILE *devrand;
	struct stat statbuf;

	char cipherstr[35];
	int ic, bc;
	int levels;
	char promptbuf[35];
	char *pass;
	char enckey[16];

	fssys = (struct stegfs_fs *) malloc(sizeof(struct stegfs_fs));

	printf("StegFS - (c)1999 ADM\n");

#ifdef DEVEL_ZEROFILL
#warning Development option DEVEL_ZEROFILL selected. File systems created \
	will not be secure.

	printf("Warning! INSECURE: Development option "
	       "DEVEL_ZEROFILL selected.\n");
#endif

	/* defaults */
	ic = 5;
	bc = 5;
	levels = STEGFS_MAX_LEVELS - 1;
	strncpy(cipherstr, "serpent", 35);

	while (1) {

		c = getopt(argc, argv, "n:c:i:b:");

		if (c == -1)
			break;

		switch (c)
		{
		case 'n':
			levels = atoi(optarg);
			break;
		case 'c':
			strncpy(cipherstr, optarg, 35);
			cipherstr[34] = '\0';
			break;
		case 'i':
			ic = atoi(optarg);
			break;
		case 'b':
			bc = atoi(optarg);
			break;

		default:
			printusage(argv[0]);
		}
	}

	if(optind != argc-2) {
		printf("Block device or block table file not specified\n");
		printusage(argv[0]);
	}

	/* validate args */
	if (ic < 1 || ic > STEGFS_MAX_INO_COPIES) {
		printf("Inode copies: invalid value\n");
		exit(1);
	}
	if (bc < 1 || bc > STEGFS_MAX_BLOCK_COPIES) {
		printf("Block copies: invalid value\n");
		exit(1);
	}
	if (levels < 0 || levels > STEGFS_MAX_LEVELS-1) {
		printf("No. of levels: invalid value\n");
		exit(1);
	}

	/* Check random number source is what we think it is */
	i = stat(RAND_DEV, &statbuf);
	if (i == -1) {
		printf("Error accessing %s\n", RAND_DEV);
		exit(3);
	}
	if (!S_ISCHR(statbuf.st_mode) ||
	    major(statbuf.st_rdev) != MEM_MAJOR ||
	    minor(statbuf.st_rdev) != 9) {
		printf("WARNING!!! %s is not char device major %u minor 9!\n",
		       RAND_DEV, MEM_MAJOR);
	}

	if (!open_fs(fssys, argv[optind], argv[optind+1], cipherstr, 1)) {
		printf("Failed to open device or file\n");
		exit(2);
	}
  

	/* get passphrases */
	j = 0;
	for (i=0; i<levels; i++) {
		sprintf(promptbuf, "Enter passphrase for level %u: ", i+1);
		pass = getpass(promptbuf);
		if(pass[0] == '\0') {
			levels = i;
			break;
		}
		passwds[i] = malloc(strlen(pass)+1);
		strncpy(passwds[i], pass, strlen(pass)+1);
		memset(pass, 0, strlen(pass));

		sprintf(promptbuf, "Re-enter passphrase for level %u: ", i+1);
		pass = getpass(promptbuf);
		if (strcmp(pass,passwds[i]) != 0) {
			memset(pass, 0, strlen(pass));
			printf("Passphrases differ.\n");
			i--;
			j++;
			if (j > 2) {
				printf("Failed after %d attempts.\n", j);
				exit(1);
			}
		}
		else {
			memset(pass, 0, strlen(pass));
			j = 0;
		}
	}
	passwds[i] = NULL;

	if (levels == 14)
		printf("Level 15");
	else if (levels < 14)
		printf("Levels %d to 15", i+1);
	if (levels<15)
		printf(" will not be accessible.\n");


	initbtab(fssys);

	randomize_empties(fssys);

	devrand = fopen(RAND_DEV, "r");
	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}

	if(fssys->btabver > 1) {
		fseek(fssys->btabfile, sizeof(struct stegfs_btable_head)+
		      fssys->e2fs->super->s_blocks_count*
		      sizeof(struct stegfs_btable), SEEK_SET);
		for (i=0; i<levels; i++) {
			fssys->slkeys[i] = malloc(STEGFS_KEY_BYTES);
			fread(fssys->slkeys[i], STEGFS_KEY_BYTES, 1, devrand);
			for (j=0; j<(i+1); j++) {
				encrypt_skey(fssys, passwds[i],
					     fssys->slkeys[j], enckey);
				fwrite(enckey, sizeof(enckey), 1, fssys->btabfile);
			}
			fseek(fssys->btabfile, (STEGFS_MAX_LEVELS-1-(i+1))*
			      STEGFS_KEY_BYTES, SEEK_CUR);
		}	
	}
	else {
		fseek(fssys->btabfile,
		      fssys->e2fs->super->s_blocks_count*
		      sizeof(struct stegfs_btable), SEEK_SET);
		for (i=0; i<levels; i++) {
			fssys->slkeys[i] = malloc(STEGFS_KEY_BYTES);
			fread(fssys->slkeys[i], STEGFS_KEY_BYTES, 1, devrand);
			for (j=0; j<(i+1); j++) {
				encrypt_skey(fssys, passwds[i],
					     fssys->slkeys[j], enckey);
				fwrite(enckey, sizeof(enckey), 1, fssys->btabfile);
			}
		}	
	}

	fclose(devrand);

	printf("Writing root directories: ");
	for (i=0; i<levels; i++) {
		printf("%3d%%", (100*i)/levels);
		mkroot(fssys, i+1, passwds[i], ic, bc);
		printf("\b\b\b\b");
	}
	printf("Done\n");

	close_fs(fssys);

	return 0;
}




void printusage(char *progname) {
	char **ciphernames;
	int numciphers, i;

	printf("\nUsage:\n");
	printf("%s /dev/fsdevice btabfile [-n num_levels] [-c cipher]\n"
	       "         [-i inocopies] [-b blkcopies]\n", basename(progname));
	printf("Note: num_levels is the number of passphrases to prompt for.\n"
	       "      All higher levels than this are given random keys\n"
	       "      which are not retained.\n"
	       "      Keys are always written for all %d levels.\n",
	       STEGFS_MAX_LEVELS-1);

	numciphers = get_numciphers();
	ciphernames = (char **)malloc(numciphers * sizeof(char *));
	numciphers = get_ciphers(ciphernames);
	if (!numciphers)
		printf("*** No ciphers available ***\n");
	else {
		printf("Available ciphers:");
		for(i=0; i<numciphers; i++) {
			printf(" %s", ciphernames[i]);
		}
		printf("\n");
	}
	exit(1);
}



void initbtab(struct stegfs_fs *fs) {

	int i,j;
	struct stegfs_btable btab;
	struct stegfs_btable_head btabh;
	FILE *devrand;
	char key[16];

#ifdef DEVEL_ZEROFILL
	devrand = fopen("/dev/zero", "r");
#else
	devrand = fopen(RAND_DEV, "r");
#endif

	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}

	/* create the stegfstab file */
	printf("Writing block table: ");
	if(fs->btabver > 1) {
		btabh.magic = STEGFS_BTAB_MAGIC;
		btabh.version = fs->btabver;
		btabh.pad1 = 0;
		btabh.pad2 = 0;
		btabh.pad3 = 0;
		fwrite(&btabh, sizeof(struct stegfs_btable_head), 1,
		       fs->btabfile);
	}

	for(i=0; i<fs->e2fs->super->s_blocks_count; i++) {
		printf("%3d%%", (100*i)/fs->e2fs->super->s_blocks_count);

		fread(&btab, sizeof(struct stegfs_btable), 1, devrand);

		fwrite(&btab, sizeof(struct stegfs_btable), 1, fs->btabfile);
		printf("\b\b\b\b");
	}

	for (i=1; i<STEGFS_MAX_LEVELS; i++) {
		for (j=0; j<i; j++) {
			fread(&key, sizeof(key), 1, devrand);
			fwrite(&key, sizeof(key), 1, fs->btabfile);
		}
	}
	printf("Done\n");
	fclose(devrand);
	return;

}

