/* StegFS - The Steganographic File System */
/* Copyright (c)1999 Andrew McDonald <andrew@mcdonald.org.uk> */

/* Take an ext2 fs and make it ready for data hiding */
/* - Randomize contents of unused blocks
   - Create root inode(s)
   - Create stegfs information table
*/
/* $Id: mkstegfs.c,v 1.16 1999/09/22 14:19:34 adm36 Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "stegfs.h"
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <linux/major.h>
#include <string.h>


void printusage(char *);
void initbtab(struct stegfs_fs *fs);
void mkroot(struct stegfs_fs *fs, int level, char *passwd, int ic, int bc);


int main(int argc, char* argv[]) {

	struct stegfs_fs *fssys;
	int i,j,c;
	char *passwds[STEGFS_MAX_LEVELS-1];
	FILE *devrand;
	struct stat statbuf;

	char cipherstr[35];
	int ic, bc;
	int levels;
	char promptbuf[35];
	char *pass;
	char enckey[16];

	fssys = (struct stegfs_fs *) malloc(sizeof(struct stegfs_fs));

	printf("StegFS - (c)1999 ADM\n");

#ifdef DEVEL_ZEROFILL
#warning Development option DEVEL_ZEROFILL selected. File systems created \
	will not be secure.

	printf("Warning! INSECURE: Development option "
	       "DEVEL_ZEROFILL selected.\n");
#endif

	/* defaults */
	ic = 5;
	bc = 5;
	levels = STEGFS_MAX_LEVELS - 1;
	strncpy(cipherstr, "serpent", 35);

	while (1) {

		c = getopt(argc, argv, "n:c:i:b:");

		if (c == -1)
			break;

		switch (c)
		{
		case 'n':
			levels = atoi(optarg);
			break;
		case 'c':
			strncpy(cipherstr, optarg, 35);
			cipherstr[34] = '\0';
			break;
		case 'i':
			ic = atoi(optarg);
			break;
		case 'b':
			bc = atoi(optarg);
			break;

		default:
			printusage(argv[0]);
		}
	}

	if(optind != argc-2) {
		printf("Block device or block table file not specified\n");
		printusage(argv[0]);
	}

	/* validate args */
	if (ic < 1 || ic > STEGFS_MAX_INO_COPIES) {
		printf("Inode copies: invalid value\n");
		exit(1);
	}
	if (bc < 1 || bc > STEGFS_MAX_BLOCK_COPIES) {
		printf("Block copies: invalid value\n");
		exit(1);
	}
	if (levels < 0 || levels > STEGFS_MAX_LEVELS-1) {
		printf("No. of levels: invalid value\n");
		exit(1);
	}

	/* Check random number source is what we think it is */
	i = stat(RAND_DEV, &statbuf);
	if (i == -1) {
		printf("Error accessing %s\n", RAND_DEV);
		exit(3);
	}
	if (!S_ISCHR(statbuf.st_mode) ||
	    major(statbuf.st_rdev) != MEM_MAJOR ||
	    minor(statbuf.st_rdev) != 9) {
		printf("WARNING!!! %s is not char device major %u minor 9!\n",
		       RAND_DEV, MEM_MAJOR);
	}

	if (!open_fs(fssys, argv[optind], argv[optind+1], cipherstr, 1)) {
		printf("Failed to open device or file\n");
		exit(2);
	}
  

	/* get passphrases */
	j = 0;
	for (i=0; i<levels; i++) {
		sprintf(promptbuf, "Enter passphrase for level %u: ", i+1);
		pass = getpass(promptbuf);
		if(pass[0] == '\0') {
			levels = i;
			break;
		}
		passwds[i] = malloc(strlen(pass)+1);
		strncpy(passwds[i], pass, strlen(pass)+1);
		memset(pass, 0, strlen(pass));

		sprintf(promptbuf, "Re-enter passphrase for level %u: ", i+1);
		pass = getpass(promptbuf);
		if (strcmp(pass,passwds[i]) != 0) {
			memset(pass, 0, strlen(pass));
			printf("Passphrases differ.\n");
			i--;
			j++;
			if (j > 2) {
				printf("Failed after %d attempts.\n", j);
				exit(1);
			}
		}
		else {
			memset(pass, 0, strlen(pass));
			j = 0;
		}
	}
	passwds[i] = NULL;

	fssys->slevel = levels;
	if (levels == 14)
		printf("Level 15");
	else if (levels < 14)
		printf("Levels %d to 15", i+1);
	if (levels<15)
		printf(" will not be accessible.\n");


	initbtab(fssys);

	get_bbpos(fssys);

	randomize_empties(fssys);

	devrand = fopen(RAND_DEV, "r");
	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}
	fseek(fssys->btabfile,
	      fssys->sb.s_blocks_count*sizeof(struct stegfs_btable), SEEK_SET);
	for (i=0; i<levels; i++) {
		fssys->slkeys[i] = malloc(16);
		fread(fssys->slkeys[i], 16, 1, devrand);
		for (j=0; j<(i+1); j++) {
			encrypt_skey(fssys, passwds[i],
				     fssys->slkeys[j], enckey);
			fwrite(enckey, sizeof(enckey), 1, fssys->btabfile);
		}
	}
	fclose(devrand);

	printf("Writing root directories: ");
	for (i=0; i<levels; i++) {
		printf("%3d%%", (100*i)/levels);
		mkroot(fssys, i+1, passwds[i], ic, bc);
		printf("\b\b\b\b");
	}
	printf("Done\n");

	close_fs(fssys);

	return 0;
}




void printusage(char *progname) {
	printf("\nUsage:\n");
	printf("%s /dev/fsdevice btabfile [-n num_levels] [-c cipher]\n"
	       "         [-i inocopies] [-b blkcopies]\n", basename(progname));
	printf("Note: num_levels is the number of passphrases to prompt for.\n"
	       "      All higher levels than this are given random keys\n"
	       "      which are not retained.\n"
	       "      Keys are always written for all %d levels.\n",
	       STEGFS_MAX_LEVELS-1);
	exit(1);
}



void initbtab(struct stegfs_fs *fs) {

	int i,j;
	struct stegfs_btable btab;
	FILE *devrand;
	char key[16];

#ifdef DEVEL_ZEROFILL
	devrand = fopen("/dev/zero", "r");
#else
	devrand = fopen(RAND_DEV, "r");
#endif

	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}

	/* create the stegfstab file */
	printf("Writing block table: ");
	for(i=0; i<fs->sb.s_blocks_count; i++) {
		printf("%3d%%", (100*i)/fs->sb.s_blocks_count);

		fread(&btab, sizeof(struct stegfs_btable), 1, devrand);

		fwrite(&btab, sizeof(struct stegfs_btable), 1, fs->btabfile);
		printf("\b\b\b\b");
	}

	for (i=1; i<STEGFS_MAX_LEVELS; i++) {
		for (j=0; j<i; j++) {
			fread(&key, sizeof(key), 1, devrand);
			fwrite(&key, sizeof(key), 1, fs->btabfile);
		}
	}
	printf("Done\n");
	fclose(devrand);
	return;

}


void mkroot(struct stegfs_fs *fs, int level, char *passwd, int ic, int bc) {

	unsigned int inonum;
	int i;
	struct timeval tv;
	struct timezone tz;
	struct stegfs_btable btab, btab2;
	struct stegfs_inode inode, inode2;
	char *datablock, *datablock2;
	struct ext2_dir_entry_2 *dirent;
	unsigned int *iblks;

	inonum = EXT2_ROOT_INO | STEGFS_INO_MASK | (level)<<(STEGFS_LVL_SHIFT);

	btab.magic1 = 0;
	btab.magic2 = 1;
	btab.iv = 0;
	btab.ino = inonum;

	inode.i_mode = S_IFDIR|S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH;
	inode.i_uid = fs->sb.s_def_resuid;
	inode.i_size = EXT2_BLOCK_SIZE(&(fs->sb));
	gettimeofday(&tv, &tz);
	inode.i_atime = inode.i_ctime = inode.i_mtime = tv.tv_sec;
	inode.i_dtime = 0;
	inode.i_gid = fs->sb.s_def_resgid;
	inode.i_links_count = 3; /* /, . and .. */
	inode.i_blocks = 1;
	inode.i_flags = 0;
	inode.osd1.linux1.l_i_reserved1 = 0;

	inode.i_version = 0;
	inode.i_file_acl = 0;
	inode.i_dir_acl = 0;
	inode.i_faddr = 0;
	inode.osd2.linux2.l_i_frag = 0;
	inode.osd2.linux2.l_i_fsize = 0;
	inode.osd2.linux2.i_pad1 = 0;
	inode.osd2.linux2.l_i_reserved2[0] = 0;
	inode.osd2.linux2.l_i_reserved2[1] = 0;

	inode.i_icopies = ic;
	inode.i_bcopies = bc;
	inode.i_pad0 = 0;

	for(i=0; i<EXT2_N_BLOCKS*STEGFS_MAX_BLOCK_COPIES+STEGFS_MAX_INO_COPIES;
	    i++)
		inode.i_inode[i] = 0;

	iblks = malloc(inode.i_icopies * sizeof(u_int32_t));
	get_new_inode_blocks(fs, level, inonum, inode.i_icopies, iblks);

	for(i=0; i<inode.i_icopies; i++) {
		inode.i_inode[i] = iblks[i];
		encrypt_btab2(fs, level, inode.i_inode[i], &btab, &btab2);
		put_btab(fs, inode.i_inode[i], &btab2);
		/*
		  printf("i %u\n", inode.i_inode[i]);
		*/
	}
	free(iblks);

	btab.magic2 = 0;
	for(i=0; i<inode.i_bcopies; i++) {
		inode.i_block[i*EXT2_N_BLOCKS] = get_new_block(fs);
		encrypt_btab2(fs, level, inode.i_block[i*EXT2_N_BLOCKS],
			      &btab, &btab2);
		put_btab(fs, inode.i_block[i*EXT2_N_BLOCKS], &btab2);
		/*
		  printf("d %u\n", inode.i_block[i*EXT2_N_BLOCKS]);
		*/
	}

	btab.magic2 = 1;

	for(i=0; i<inode.i_icopies; i++) {
		encrypt_cbc2(fs, level, inode.i_inode[i],
			     (char *)&inode, (char *)&inode2,
			     sizeof(struct stegfs_inode), btab.iv);
		fseek(fs->fsfile, EXT2_BLOCK_SIZE(&(fs->sb))*inode.i_inode[i],
		      SEEK_SET);
		fwrite(&inode2, sizeof(struct stegfs_inode), 1, fs->fsfile);
		btab.bchecksum = stegfs_chksum((unsigned char *)&inode2,
					       sizeof(struct stegfs_inode));
		encrypt_btab2(fs, level, inode.i_inode[i], &btab, &btab2);
		put_btab(fs, inode.i_inode[i], &btab2);
	}

	datablock = malloc( EXT2_BLOCK_SIZE(&(fs->sb)) );
	datablock2 = malloc( EXT2_BLOCK_SIZE(&(fs->sb)) );
	dirent = (struct ext2_dir_entry_2 *)
		malloc(sizeof(struct ext2_dir_entry_2) * 2);
	dirent[0].rec_len = 12;
	dirent[0].name_len = 1;
	dirent[0].file_type = 0;
	strncpy(dirent[0].name, ".", EXT2_NAME_LEN);
	dirent[0].inode = inonum;

	dirent[1].rec_len = EXT2_BLOCK_SIZE(&(fs->sb)) - dirent[0].rec_len;
	dirent[1].name_len = 2;
	dirent[1].file_type = 0;
	strncpy(dirent[1].name, "..", EXT2_NAME_LEN);
	dirent[1].inode = inonum;

	memcpy(datablock, &dirent[0], sizeof(struct ext2_dir_entry_2));
	memcpy(datablock+dirent[0].rec_len, &dirent[1],
	       sizeof(struct ext2_dir_entry_2));

	btab.magic2 = 0;
	for(i=0; i<inode.i_bcopies; i++) {
		fseek(fs->fsfile,
		      EXT2_BLOCK_SIZE(&(fs->sb))*inode.i_block[i*EXT2_N_BLOCKS],
		      SEEK_SET);
		encrypt_cbc2(fs, level, inode.i_block[i*EXT2_N_BLOCKS],
			     datablock, datablock2,
			     EXT2_BLOCK_SIZE(&(fs->sb)), btab.iv);
		fwrite(datablock2, EXT2_BLOCK_SIZE(&(fs->sb)), 1, fs->fsfile);
		btab.bchecksum = stegfs_chksum(datablock2,
					       EXT2_BLOCK_SIZE(&(fs->sb)));
		encrypt_btab2(fs, level, inode.i_block[i*EXT2_N_BLOCKS],
			      &btab, &btab2);
		put_btab(fs, inode.i_block[i*EXT2_N_BLOCKS], &btab2);

	}

	free(dirent);
	free(datablock);
	free(datablock2);
}


