/* Stegfs Helper Functions */
/* Sort of in the vein of libext2, but not as useful or complete */

/* Copyright (c)1999 Andrew McDonald (andrew@mcdonald.org.uk) */
/* $Id: stegfs.c,v 1.16 1999/09/22 14:19:58 adm36 Exp $ */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include "stegfs.h"
#include "serpent.h"
#include "nullenc.h"
#include "rc6.h"
#include "sha1.h"

static struct stegfs_cipher serpent_ops = {
	"serpent",
	serpent_set_key,
	serpent_encrypt,
	serpent_decrypt
};

static struct stegfs_cipher rc6_ops = {
	"rc6",
	rc6_set_key,
	rc6_encrypt,
	rc6_decrypt
};

static struct stegfs_cipher nullenc_ops = {
	"nullenc",
	nullenc_set_key,
	nullenc_encrypt,
	nullenc_decrypt
};

#define NUMCIPHERS 3
static struct stegfs_cipher *stegfs_ciphers[NUMCIPHERS] = {
  &serpent_ops,
  &rc6_ops,
  &nullenc_ops
};

static int nibblemap[] = {4, 3, 3, 2, 3, 2, 2, 1, 3, 2, 2, 1, 2, 1, 1, 0};


int open_fs(struct stegfs_fs *fs, char *fsname, char *btabname, char *cipher,
	    int newfs) {

	int i;
	struct stat statbuf;

	i = stat(fsname, &statbuf);
	if (i == -1 && errno == ENOENT) {
		printf("Block device %s does not exist\n", fsname);
		return 0;
	}
	if (!S_ISBLK(statbuf.st_mode)) {
		printf("Warning: %s is not a block device\n", fsname);
	}
	fs->fsfile = fopen(fsname, "r+");
	if (!fs->fsfile)
		return 0;
	if (fseek(fs->fsfile, BLOCK_SIZE, SEEK_SET) != 0
	    || fread(&(fs->sb), sizeof(struct ext2_super_block),
		     1, fs->fsfile) != 1) {
		perror(fsname);
		fclose(fs->fsfile);
		return 0;
	}
	if(fs->sb.s_magic != EXT2_SUPER_MAGIC) {
		printf("%s: Bad Ext2fs magic\n", fsname);
		fclose (fs->fsfile);
		return 0;
	}
	if(fs->sb.s_state & EXT2_ERROR_FS) {
		printf("Ext2 file system on %s has errors. Run e2fsck.\n",
		       fsname);
		fclose(fs->fsfile);
		return 0;
	}
	if(!(fs->sb.s_state & EXT2_VALID_FS)) {
		printf("Error with %s. File system mounted or not "
		       "unmounted cleanly?\n", fsname);
		fclose(fs->fsfile);
		return 0;
	}

	if (newfs) {
		i = stat(btabname, &statbuf);
		if (i == 0) {
			printf("Block table file %s already exists.\n",
			       btabname);
			fclose(fs->fsfile);
			return 0;
		}
		fs->btabfile = fopen(btabname, "w+");
	}
	else
		fs->btabfile = fopen(btabname, "r+");

	if (!fs->btabfile)
		return 0;

	fs->cipher = NULL;
	for (i=0; i<NUMCIPHERS; i++) {
		if(!strcmp(cipher, stegfs_ciphers[i]->id))
			fs->cipher = stegfs_ciphers[i];
	}
	if (!fs->cipher) {
		printf("Invalid cipher name: %s\n", cipher);
		return 0;
	}

  return 1;
}

int close_fs(struct stegfs_fs *fs) {
	fclose(fs->fsfile);
	fclose(fs->btabfile);
	return 1;
}

void get_skeys(struct stegfs_fs *fs, int level, char *pass) {

	int i;
	char enckey[16];

	fs->slevel = level;

	fseek(fs->btabfile, fs->sb.s_blocks_count*sizeof(struct stegfs_btable),
	      SEEK_SET);
	fseek(fs->btabfile, .5*level*(level-1)*sizeof(struct stegfs_btable),
	      SEEK_CUR);

	for (i=0; i<level; i++) {
		fs->slkeys[i] = malloc(16);
		fread(&enckey, 16, 1, fs->btabfile);
		decrypt_skey(fs, pass, enckey, fs->slkeys[i]);
	}

	return;
}

int get_stegfs_inodes(struct stegfs_fs *fs, struct inoblocks **inolist) {

	int i,j,l;
	struct stegfs_btable btab, btab2;
	struct inoblocks *inob, *inobp, *inobtmp;
	int addedblk;

	fseek(fs->btabfile, 0, SEEK_SET);

	for(i=0; i<fs->sb.s_blocks_count; i++) {

		fread(&btab2, sizeof(struct stegfs_btable), 1,
		      fs->btabfile);
		for(l=0; l<fs->slevel; l++) {
			decrypt_btab2(fs, l+1, i, &btab2, &btab);

			if(btab.magic1 == 0 && btab.magic2 == 1) {
				inob = inolist[l];
				inobp = inob;
				addedblk = 0;
				while(inob != NULL) {
					if(inob->inode == btab.ino) {
						for(j=0; j<STEGFS_MAX_INO_COPIES; j++) {
							if(inob->block[j] == 0) {
								inob->block[j] = i;
								addedblk = 1;
								break;
							}
						}
						if(!addedblk) {
							printf("Too many inode copies: inode %u block %u\n", inob->inode, i);
							addedblk = 1;
						}
						break;
					}

					inobp = inob;
					inob = inob->next;
				}
				if(!addedblk) {

					inobtmp = (struct inoblocks*)malloc(sizeof(struct inoblocks));
					if(inobtmp == NULL) {
						printf("can't get free memory\n");
						return 0;
					}
					if(inobp == NULL) {

						inolist[l] = inobtmp;
						inobp = inolist[l];
						inobp->inode = btab.ino;
						inobp->block[0] = i;
						inobp->next = NULL;

						for(j=1; j<STEGFS_MAX_INO_COPIES; j++) {
							inobp->block[j] = 0;
						}

					}
					else {

						inobp->next = inobtmp;
						inobp->next->inode = btab.ino;
						inobp->next->block[0] = i;
						inobp->next->next = NULL;
						for(j=1; j<STEGFS_MAX_INO_COPIES; j++)
							inobp->next->block[j] = 0;

					}
				}
			}
		}
	}

  return 1;
}

int get_bbpos(struct stegfs_fs *fs) {

	struct ext2_group_desc gd;
	int i;

	fs->groups = fs->sb.s_inodes_count / fs->sb.s_inodes_per_group;

	fs->bbarray = (u_int32_t *) malloc(fs->groups * sizeof(u_int32_t));

	for(i=0; i<fs->groups; i++) {
    
		if(fread(&gd, sizeof(struct ext2_group_desc), 1,
			 fs->fsfile) != 1) {
			perror("GD read error");
			return 0;
		}

		fs->bbarray[i] = gd.bg_block_bitmap;
	}

	return 1;
}

void randomize_empties(struct stegfs_fs *fs /*, int avoidhidden*/) {
  /* read block group block bitmap then randomize contents of empties */

	unsigned char bitmap[1024]; /* FIXME: hard coded sizes? */
	unsigned char buff1[1024];
	unsigned char buff8[8192];

	unsigned int sum = 0;
	unsigned int i, n, c, k;
	unsigned char b;
	FILE *devrand;

#ifdef DEVEL_ZEROFILL
	devrand = fopen("/dev/zero", "r");
#else
	devrand = fopen(RAND_DEV, "r");
#endif
	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}

	for(k=0; k<8192; k++)
		buff8[k] = 0;
	for(k=0; k<1024; k++)
		buff1[k] = 0;


	for(n=0; n<fs->groups; n++) {
                /* seek to start of bitmap block */
		fseek(fs->fsfile, fs->bbarray[n]*1024, SEEK_SET);

		fread(bitmap, 1024, 1, fs->fsfile);

		for (i = 0; i < 1024; i++)
			sum += nibblemap[bitmap[i] & 0xf] +
				nibblemap[(bitmap[i] >> 4) & 0xf];
	}

	printf("Filling empty blocks: ");

	for(n=0; n<fs->groups; n++) {
		fseek(fs->fsfile, fs->bbarray[n]*1024, SEEK_SET);
		fread(bitmap, 1024, 1, fs->fsfile);

		c=0;
		fseek(fs->fsfile, n*8192*1024+1024, SEEK_SET);
		for(i=0; i<1024; i++) {
			printf("%3d%%", (100*(n*8192+i))/fs->sb.s_blocks_count);
			if(bitmap[i] == 0xFF) {
				c++;
			}
			else {
				fseek(fs->fsfile, c*8192, SEEK_CUR);
				c=0;
				if(!bitmap[i]) {
					/* 8 blocks together */
					fread(buff8, 1, 8192, devrand);
					fwrite(buff8, 1, 8192, fs->fsfile);
				}
				else {
					b = bitmap[i];
					for(k=0; k<8; k++) {
						if(!(b & 0x01)) {
							fread(buff1, 1, 1024, devrand);
							fwrite(buff1, 1, 1024, fs->fsfile);
						}
						else {
							fseek(fs->fsfile, 1024, SEEK_CUR);
						}
						b = b>>1;
					}
				}
			}
			printf("\b\b\b\b");
		}

	}
	printf("Done\n");
	fclose(devrand);
	return;
}

u_int8_t ext2_isblockused(struct stegfs_fs *fs, u_int32_t blocknum) {

	unsigned char bitmap[1024]; /* FIXME */
	unsigned char bitc;
	u_int32_t bn, gbn, bng;
  
	/* is it in range */
	if (blocknum < 1 || blocknum > fs->sb.s_blocks_count-1) {
		/* out of range */
		return -1;
	}

	/* work out which block group it is in */

	bn = blocknum-1; /* first block of fs is reserved */

	gbn = bn / 8192;

	if (gbn+1 > fs->groups) {
		/* invalid block */
		return -1;
	}

	/* get blocks bitmap */
	fseek(fs->fsfile, fs->bbarray[gbn]*1024, SEEK_SET); /* seek to start of
							       bitmap block */
	fread(bitmap, 1024, 1, fs->fsfile);

	/* check bit */
	bng = (bn % 8192) / 8; /* block number within group */
	bitc = bitmap[bng];

	bng = (bn % 8192) % 8;   /* bit within char */
	bitc = bitc >> bng;
  
	return (bitc & 0x01);
}

u_int8_t stegfs_isblockused(struct stegfs_fs *fs, u_int32_t blocknum) {

	struct stegfs_btable btab, btab2;
	int i;

	get_btab(fs, blocknum, &btab);

	for (i=1; i<(fs->slevel+1); i++) {
		decrypt_btab2(fs, i, blocknum, &btab, &btab2);
		if(btab2.magic1 == 0 &&
		   (btab2.magic2 == 0 || btab2.magic2 == 1))
			return 1;
	}

	return 0;
}

u_int32_t get_new_block(struct stegfs_fs *fs) {

	FILE *devrand;
	u_int32_t scale, maxblock;
	u_int32_t randbs, randbsx;
	int k;

	devrand = fopen(RAND_DEV, "r");
	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}

	maxblock = fs->sb.s_blocks_count - 1;
	scale = ((unsigned int)-1) / maxblock;

	fread(&randbs, sizeof(u_int32_t), 1, devrand);

	k=0;
	randbsx = randbs / scale;
	while (randbsx > maxblock || randbsx < 1 ||
	       ext2_isblockused(fs,randbsx) ||
	       stegfs_isblockused(fs,randbsx)) {
		k++;
		if(k>10000) {
			return 0;
		}
		fread(&randbs, sizeof(u_int32_t), 1, devrand);
		randbsx = randbs / scale;
	}

	fclose(devrand);

	return randbsx;
}

int get_new_inode_blocks(struct stegfs_fs *fs, int slevel, int ino,
			 int numcopies, unsigned int *blocks) {

	u_int32_t scale, maxblock;
	u_int32_t randbsx;
	int k,i;
	unsigned int digest[5];
	SHA1_CTX sha1ctx;
#ifdef STEGFS_OLD_INO_POS
	unsigned char *keystr;
	int slen = 16+sizeof(int);
#else
	unsigned int keystr[6];
#endif
	int blks = 0;

	maxblock = fs->sb.s_blocks_count - 1;
	scale = ((unsigned int)-1) / maxblock;

#ifdef STEGFS_OLD_INO_POS
	keystr = malloc(slen);
	memcpy(keystr, fs->slkeys[slevel-1], 16);
	memcpy(keystr+16, &ino, sizeof(int));

	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, keystr, slen);
	SHA1Final((unsigned char *)digest, &sha1ctx);

	free(keystr);
#else
	memcpy((unsigned char*)keystr, fs->slkeys[slevel-1], 16);
	keystr[4] = ino;
	keystr[5] = 0;

        SHA1Init(&sha1ctx);
        SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
        SHA1Final((unsigned char *)digest, &sha1ctx);
#endif

	k=0;

	while (1) {

		for (i=0; i<5; i++) {
			randbsx = digest[i] / scale;
			if (!(randbsx > maxblock || randbsx < 1 ||
			      ext2_isblockused(fs,randbsx) ||
			      stegfs_isblockused(fs,randbsx))) {
				blocks[blks++] = randbsx;
			}
			if (blks == numcopies)
				return 1;
		}

		SHA1Init(&sha1ctx);
#ifdef STEGFS_OLD_INO_POS
		SHA1Update(&sha1ctx, (unsigned char *)digest,
			   5*sizeof(unsigned int));
#else
		keystr[5]++;
		SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
#endif
		SHA1Final((unsigned char *)digest, &sha1ctx);

	}


}

void put_btab(struct stegfs_fs *fs, u_int32_t bnum,
	      struct stegfs_btable *btab) {


	fseek(fs->btabfile, bnum*sizeof(struct stegfs_btable), SEEK_SET);

	fwrite(btab, sizeof(struct stegfs_btable), 1, fs->btabfile);

	return;

}

void get_btab(struct stegfs_fs *fs, u_int32_t bnum,
	      struct stegfs_btable *btab) {


	fseek(fs->btabfile, bnum*sizeof(struct stegfs_btable), SEEK_SET);

	fread(btab, sizeof(struct stegfs_btable), 1, fs->btabfile);

	return;

}

unsigned int stegfs_chksum(unsigned char * data, int size) {
#ifdef STEGFS_PARANOID_XSUM
	SHA1_CTX sha1ctx;
	unsigned int digest[5];
  
	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, data, size);
	SHA1Final((unsigned char *)digest, &sha1ctx);

	return digest[0];
#else
        unsigned char *datax;

        datax = data + size - sizeof(unsigned int);

        return *datax;

#endif
}

char *make_key(struct stegfs_fs *fs, int level, unsigned int blocknum) {
#ifdef STEGFS_PARANOID
	char *keystr;
	SHA1_CTX sha1ctx;
	unsigned int digest[5];
	char *key;
	int slen = 16+1+sizeof(unsigned int);

	keystr = malloc(slen);
	strncpy(keystr, fs->slkeys[level-1], slen);
	memcpy(keystr+16+1, &blocknum, sizeof(unsigned int));

	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, keystr, slen);
	SHA1Final((unsigned char *)digest, &sha1ctx);
	free(keystr);

	key = malloc(16);
	memcpy(key, digest, 16);

	return key;
#else
	char *key;
	unsigned int *key2;
	key = malloc(16);
	memcpy(key, fs->slkeys[level-1], 16);
	key2 = (unsigned int *)key;
	key2[3] = key2[3] ^ blocknum;

	return key;
#endif
}

void encrypt_skey(struct stegfs_fs *fs, char *passwd, char *skey,
		  char *skeyout) {
	SHA1_CTX sha1ctx;
	unsigned int digest[5];
	char *keys;

	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, passwd, strlen(passwd));
	SHA1Final((unsigned char *)digest, &sha1ctx);

	keys = fs->cipher->set_key((char *)digest, 128);
	fs->cipher->encrypt(keys, skey, skeyout);
	free(keys);
}

void decrypt_skey(struct stegfs_fs *fs, char *passwd, char *skey,
		  char *skeyout) {
	SHA1_CTX sha1ctx;
	unsigned int digest[5];
	char *keys;

	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, passwd, strlen(passwd));
	SHA1Final((unsigned char *)digest, &sha1ctx);

	keys = fs->cipher->set_key((char *)digest, 128);
	fs->cipher->decrypt(keys, skey, skeyout);
	free(keys);
}

void encrypt_cbc(struct stegfs_fs *fs, int level, unsigned int bnum,
		 char *data, int size, unsigned short iv) {

	char *key, *keys;
	char outb[16], inb[16];
	char *p;
	int i;

	key = make_key(fs, level, bnum);
	keys = fs->cipher->set_key(key, 128);
	for(p=data; p<(data+size); p+=16) {
		if (p>data) {
			for(i=0; i<16; i++)
				inb[i] = p[i]^outb[i];
		}
		else {
			memcpy(inb, p, 16);
                        ((unsigned short *)inb)[0] ^= iv;
		}
		fs->cipher->encrypt(keys, inb, outb);
		memcpy(p, outb, 16);
	}
	free(key);
	free(keys);

}

void encrypt_cbc2(struct stegfs_fs *fs, int level, unsigned int bnum,
		  char *data, char *data2, int size, unsigned short iv) {

	char *key, *keys;
	char *p, *p2;
	char inb[16];
	int i;

	key = make_key(fs, level, bnum);
	keys = fs->cipher->set_key(key, 128);
	for(p=data, p2=data2; p<(data+size); p+=16, p2+=16) {
		if (p>data) {
			for(i=0; i<16; i++)
				inb[i] = p[i]^(p2-16)[i];
		}
		else {
			memcpy(inb, p, 16);
                        ((unsigned short *)inb)[0] ^= iv;
		}
		fs->cipher->encrypt(keys, inb, p2);
	}
	free(key);
	free(keys);

}

void decrypt_cbc(struct stegfs_fs *fs, int level, unsigned int bnum,
		 char *data, int size, unsigned short iv) {

	char *key, *keys;
	char inb[16], inb2[16], outb[16];
	char *p;
	int i;

	key = make_key(fs, level, bnum);
	keys = fs->cipher->set_key(key, 128);
	for(p=data; p<(data+size); p+=16) {
		if (p>data) {
			memcpy(inb2, inb, 16);
			memcpy(inb, p, 16);
		}
		fs->cipher->decrypt(keys, p, outb);
		if (p>data) {
			for(i=0; i<16; i++)
				p[i] = outb[i]^inb2[i];
		}
		else {
                        ((unsigned short *)outb)[0] ^= iv;
			memcpy(inb, p, 16);
			memcpy(p, outb, 16);
		}
	}
	free(key);
	free(keys);

}

void decrypt_cbc2(struct stegfs_fs *fs, int level, unsigned int bnum,
		  char *data, char *data2, int size, unsigned short iv) {

	char *key, *keys;
	char *p, *p2;
	int i;

	key = make_key(fs, level, bnum);
	keys = fs->cipher->set_key(key, 128);
	for(p=data, p2=data2; p<(data+size); p+=16, p2+=16) {
		fs->cipher->decrypt(keys, p, p2);
		if (p>data) {
			for(i=0; i<16; i++)
				p2[i] = p2[i]^(p-16)[i];
		}
		else
                        ((unsigned short *)p2)[0] ^= iv;
	}
	free(key);
	free(keys);

}

void encrypt_btab(struct stegfs_fs *fs, int slevel, unsigned int bnum,
		  struct stegfs_btable *btab) {

        char *tmpdata;
        char *key, *keys;

        tmpdata = malloc(sizeof(struct stegfs_btable));

        memcpy(tmpdata, btab, sizeof(struct stegfs_btable));

        key = make_key(fs, slevel, bnum);
        keys = fs->cipher->set_key(key, 128);
        free(key);
        fs->cipher->encrypt(keys, tmpdata, (char *)btab);
        free(keys);
        free(tmpdata);

        return;
}

void encrypt_btab2(struct stegfs_fs *fs, int slevel, unsigned int bnum,
		   struct stegfs_btable *btab, struct stegfs_btable *btab2) {

        char *key, *keys;

        key = make_key(fs, slevel, bnum);
        keys = fs->cipher->set_key(key, 128);
        free(key);
        fs->cipher->encrypt(keys, (char *)btab, (char *)btab2);
        free(keys);

        return;
}

void decrypt_btab(struct stegfs_fs *fs, int slevel, unsigned int bnum,
		  struct stegfs_btable *btab) {

        char *tmpdata;
        char *key, *keys;

        tmpdata = malloc(sizeof(struct stegfs_btable));

        memcpy(tmpdata, btab, sizeof(struct stegfs_btable));

        key = make_key(fs, slevel, bnum);
        keys = fs->cipher->set_key(key, 128);
        free(key);
        fs->cipher->decrypt(keys, tmpdata, (char *)btab);
        free(keys);
        free(tmpdata);

        return;
}


void decrypt_btab2(struct stegfs_fs *fs, int slevel, unsigned int bnum,
		   struct stegfs_btable *btab, struct stegfs_btable *btab2) {

        char *key, *keys;

        key = make_key(fs, slevel, bnum);
        keys = fs->cipher->set_key(key, 128);
        free(key);
        fs->cipher->decrypt(keys, (char *)btab, (char *)btab2);
        free(keys);

        return;
}
