/*
 *      $Source: /home/fergia/CVS//tcfs-openbsd/tcfslib-1.0.0/lib/tcfs_database.c,v $
 *      $Revision: 1.1.1.1 $
 *      $Date: 2000/06/07 12:21:31 $
 *      $State: Exp $
 *      $Author: fergia $
 *      $Lockers$
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <gdbm.h>
#include "tcfs_global.h"
#include "tcfs_database.h"
#include "tcfs_key.h"
#include "tcfs_errno.h"

#ifdef TCFS_DEBUG
#include <errno.h>
#endif

/* It gets memory for a tcfspwdb struct. On error it returns a null pointer. On success it returns a pointer to the memory. */
tcfspwdb* 
tcfs_pwdb_alloc(void) {

 	tcfspwdb* pointer;

	pointer=(tcfspwdb*) malloc(sizeof(tcfspwdb));
	if (pointer==NULL) {
		tcfs_errno=ER_MEM;
		return NULL;
	}

	tcfs_errno=OK;
	return (tcfspwdb*) pointer;
}

/* It gets memory for a tcfsgpwdb struct. On error it returns a null pointer. On success it returns a pointer to the memory. */
tcfsgpwdb* 
tcfs_gpwdb_alloc(void) {

 	tcfsgpwdb* pointer;

	pointer=(tcfsgpwdb*) malloc(sizeof(tcfsgpwdb));
	if (pointer==NULL) {
		tcfs_errno=ER_MEM;
		return NULL;
	}

	tcfs_errno=OK;
	return (tcfsgpwdb*) pointer;
}

/* It frees memory area allocated with tcfs_pwdballoc */
void
tcfs_pwdb_free(tcfspwdb *src) {
	free((void*)src);
	tcfs_errno=OK;
}

/* It frees memory area allocated with tcfs_gpwdballoc */
void
tcfs_gpwdb_free(tcfsgpwdb *src) {
	free((void*)src);
	tcfs_errno=OK;
}

/* It allows to edit a tcfspwdb struct. The field to edit are specified by flag. On success return 0.*/
int 
tcfs_pwdb_edit(tcfspwdb* data, int flag, ...) {

	va_list list;
	char *arg;

	va_start(list,flag);

	if ((flag & TCFS_DBUSER)!=0) {
		arg=va_arg(list,char*);
		if (strlen(arg)>UserLen) 
			return(tcfs_errno=ER_MAXUSER);
		strcpy(data->user,arg);
	}
	
	if ((flag & TCFS_DBKEY)!=0) {
		arg=va_arg(list,char*);
		if (strlen(arg)>KeyLen) 
			return(tcfs_errno=ER_MAXKEY);
		strcpy(data->key,arg);
	}
	
	va_end(list);
		
	return(tcfs_errno=OK);	
}	

/* It allows to edit a tcfsgpwdb struct. The field to edit are specified by flag. On success return the modified tcfsgpwdb struct.*/
int 
tcfs_gpwdb_edit(tcfsgpwdb* data, int flag, ...) {

	va_list list;
	char *arg;
	gid_t arg_gid;
	unsigned int arg_uint;

	va_start(list,flag);

	if ((flag & TCFS_DBUSER)!=0) {
		arg=va_arg(list,char*);
		if (strlen(arg)>UserLen) 
			return(tcfs_errno=ER_MAXUSER);
		strcpy(data->user,arg);
	}
	
	if ((flag & TCFS_DBKEY)!=0) {
		arg=va_arg(list,char*);
		if (strlen(arg)>KeyLen) 
			return(tcfs_errno=ER_MAXKEY);
		
		strcpy(data->groupkey,arg);
	}

	if ((flag & TCFS_DBGID)!=0) {
		arg_gid=va_arg(list,gid_t);
		data->gid=arg_gid;
	}

	if ((flag & TCFS_DBPARTS)!=0) {
		arg_uint=va_arg(list,unsigned int);
		data->numofparts=arg_uint;
	}
	
	if ((flag & TCFS_DBTHRSHO)!=0) {
		arg_uint=va_arg(list,unsigned int);
		data->threshold=arg_uint;
	}
	
	va_end(list);
		
	return(tcfs_errno=OK);	
}

/* It fetch's a tcfspwdb struct. On success return 0 */
int
tcfs_pwdb_fetch(tcfspwdb* data, int flag, ...) {
	
	va_list list;
	char *user,*key;

	va_start(list,flag);

	if (flag & TCFS_DBUSER){
		user=va_arg(list,char*);
		memset(user,0,UserLen+1);
		strcpy(user,data->user);
	}						

	if (flag & TCFS_DBKEY) {
		key=va_arg(list,char*);
		memset(key,0,KeyLen+1);
		strcpy(key,data->key);
	}

	va_end(list);
	return(tcfs_errno=OK); 	

}

/* It fetch's a tcfsgpwdb struct. On success return 0 */
/* To do: i must decide if the function will manage various errors. */	
int
tcfs_gpwdb_fetch(tcfsgpwdb* data, int flag, ...) {
	
	va_list list;
	char *user,*gkey;
	gid_t *gid;
	unsigned int *num,*thr;

	va_start(list,flag);

	if (flag & TCFS_DBUSER){
		user=va_arg(list,char*);
		memset(user,0,UserLen+1);
		strcpy(user,data->user);
	}						

	if (flag & TCFS_DBKEY) {
		gkey=va_arg(list,char*);
		memset(gkey,0,KeyLen+1);
		strcpy(gkey,data->groupkey);
	}

	if (flag & TCFS_DBGID) {
		gid=va_arg(list,gid_t*);
		*gid=data->gid;

#ifdef TCFS_DEBUG
	fprintf(stderr,"\ntcfs_gpwdbfecth: value of gid and *gid:%p %d.Bye\n",			gid,*gid);
#endif
	}	

	if (flag & TCFS_DBPARTS) {
		num=va_arg(list,unsigned int*);
		*num=data->numofparts;
	}

	if (flag & TCFS_DBTHRSHO) {
		thr=va_arg(list,unsigned int*);
		*thr=data->threshold;
	}

	va_end(list);
	return(tcfs_errno=OK); 	

}
			
/* Insert/Modify/Delete an entry in the GDBM key's file. Return 0 on success. */
int
tcfs_pwfile_store(char *user, tcfspwdb *src, unsigned int cmd) {
	
	GDBM_FILE pwdb;
	int flag=GDBM_WRITER,store_flag;
	datum key, content;

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_pwfile_store: I'm in tcfs_gdbm_storepw\n");
#endif

	if (access(TCFS_PWDBPATH,F_OK)<0) 	
		flag=GDBM_NEWDB;
		
	pwdb=gdbm_open(TCFS_PWDBPATH,TCFS_PWDBSIZE,flag,TCFS_PWDBMODE,NULL);
	
	if (pwdb==NULL) { 

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_pwfile_store: %s\n",gdbm_strerror(gdbm_errno));
#endif

		return(tcfs_errno=ER_GDBMOPEN);
	}

	key.dptr=user;
	key.dsize=strlen(user);	
	
	if (cmd!=TCFS_DBDELETE) {
	
		content.dptr=(char*)src;
		content.dsize=sizeof(tcfspwdb);
	
		switch(cmd) {
		
			case TCFS_DBINSERT:
				store_flag=GDBM_INSERT;
				break;
			case TCFS_DBREPLACE:
				if (gdbm_exists(pwdb,key)==0) 
					return(tcfs_errno=ER_GDBMNODATA);
				store_flag=GDBM_REPLACE;
				break;
			default:
				return(tcfs_errno=ER_UNKGDBMOPER); 
			
		}
				
		if (gdbm_store(pwdb,key,content,store_flag)!=0) {
			gdbm_close(pwdb);
			return(tcfs_errno=ER_GDBMSTORE);
		}
	}	

	else 
		if (gdbm_delete(pwdb,key)!=0) {
			gdbm_close(pwdb);
			return(tcfs_errno=ER_GDBMDELETE);
		}

	gdbm_reorganize(pwdb);	
	gdbm_close(pwdb);
	return(tcfs_errno=OK);
}

/* Insert/Modify/Delete an entry in the GDBM group key's file. Return 0 on success. */
int
tcfs_gpwfile_store(char *user, gid_t gid, tcfsgpwdb *src, unsigned int cmd) {
	
	GDBM_FILE pwdb;
	int flag=GDBM_WRITER,store_flag;
	datum key, content;
	

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_gpwfile_store: I'm in tcfs_gdbm_gstorepw\n");
#endif

	if (access(TCFS_GPWDBPATH,F_OK)<0) 	
		flag=GDBM_NEWDB;
		
	pwdb=gdbm_open(TCFS_GPWDBPATH,TCFS_GPWDBSIZE,flag,TCFS_GPWDBMODE,NULL);
	
	if (pwdb==NULL) { 

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_gpwfile_store: %s\n",gdbm_strerror(gdbm_errno));
#endif

		return(tcfs_errno=ER_GDBMOPEN);
	}

	key.dptr=(char*) malloc(strlen(user)+1+TCFS_DIMGID+1); /* user + \033 + gid + \0 */
	sprintf(key.dptr,"%s\033%d",user,gid);
	key.dsize=strlen(key.dptr);	
	
#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_gpwfile_store: Value of key.dptr and key.dsize:%s %d\n",
key.dptr, key.dsize);
#endif

	if (cmd!=TCFS_DBDELETE) {
	
		content.dptr=(char*)src;
		content.dsize=sizeof(tcfsgpwdb);
	
		switch(cmd) {
		
			case TCFS_DBINSERT:
				store_flag=GDBM_INSERT;
				break;
			case TCFS_DBREPLACE:
				if (gdbm_exists(pwdb,key)==0) {
					free(key.dptr);
					return(tcfs_errno=ER_GDBMNODATA);
				}
				store_flag=GDBM_REPLACE;
				break;
			default:
				free(key.dptr);
				return(tcfs_errno=ER_UNKGDBMOPER); 
			
		}
				
		if (gdbm_store(pwdb,key,content,store_flag)!=0) {
			gdbm_close(pwdb);
			free(key.dptr);
			return(tcfs_errno=ER_GDBMSTORE);
		}
	}	

	else 
		if (gdbm_delete(pwdb,key)!=0) {
			gdbm_close(pwdb);
			free(key.dptr);
			return(tcfs_errno=ER_GDBMDELETE);
		}

	gdbm_reorganize(pwdb);	
	gdbm_close(pwdb);
	free(key.dptr);	
	return(tcfs_errno=OK);
}

/* It fetchs an entry in the GDBM key's file. It returns a pointer to the tcfspwdb struct fetched. The tcfspwdb struct fetched must be freed by the user. The function returns the null pointer on error. */	
tcfspwdb*
tcfs_pwfile_fetch(char *user) {

	GDBM_FILE pwdb;
	datum key, content;
	tcfspwdb *result=NULL;
	
	if (access(TCFS_PWDBPATH,F_OK)<0) {
		tcfs_errno=ER_GDBMEXIST;
		return NULL;
	}		
							
	pwdb=gdbm_open(TCFS_PWDBPATH,TCFS_PWDBSIZE,GDBM_READER,TCFS_PWDBMODE,
			NULL);
	
	if (pwdb==NULL) {
		tcfs_errno=ER_GDBMOPEN;
		return NULL;
	}

	key.dptr=user;
	key.dsize=strlen(user);

	content=gdbm_fetch(pwdb,key);
	if (content.dptr==NULL) {
		gdbm_close(pwdb);
		tcfs_errno=ER_GDBMNODATA;
		return NULL;
	}	

	result=tcfs_pwdb_alloc();
	
	result=memcpy(result,content.dptr,sizeof(tcfspwdb));

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_pwfile_fetch: result->user -%s- result->key -%s-\n",result->user,result->key);
#endif
	
	gdbm_close(pwdb);
	free(content.dptr);
	tcfs_errno=OK;
	return (tcfspwdb*) result;
}

/* It fetchs an entry in the GDBM group key's file. It returns a pointer to the tcfsgpwdb struct fetched. The tcfsgpwdb struct fetched must be freed by the user. The function returns the null pointer on error. */
tcfsgpwdb*
tcfs_gpwfile_fetch(char *user,gid_t gid) {

	GDBM_FILE pwdb;
	datum key, content;
	tcfsgpwdb *result=NULL;
	
	if (access(TCFS_GPWDBPATH,F_OK)<0) {
		tcfs_errno=ER_GDBMEXIST;
		return NULL;
	}		
							
	pwdb=gdbm_open(TCFS_GPWDBPATH,TCFS_GPWDBSIZE,GDBM_READER,TCFS_GPWDBMODE,
			NULL);
	
	if (pwdb==NULL) {
		tcfs_errno=ER_GDBMOPEN;
		return NULL;
	}

	key.dptr=(char*) malloc(strlen(user)+1+TCFS_DIMGID+1);  /* user + \033 + gid + \0 */
	sprintf(key.dptr,"%s\033%d",user,gid);
	key.dsize=strlen(key.dptr);

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_gpwfile_fetch: Value of key.dptr and key.dsize:%s %d\n",
key.dptr, key.dsize);
#endif

	content=gdbm_fetch(pwdb,key);
	if (content.dptr==NULL) {
		gdbm_close(pwdb);
		free(key.dptr);
		tcfs_errno=ER_GDBMNODATA;
		return NULL;
	}	

	result=tcfs_gpwdb_alloc();
	
	result=memcpy(result,content.dptr,sizeof(tcfsgpwdb));
	
	gdbm_close(pwdb);
	free(content.dptr);
	free(key.dptr);
	tcfs_errno=OK;
	return (tcfsgpwdb*) result;
}

/* It removes all the user that are in the group in input. On success return 0. It's used when the user calls tcfsrmgroup */
int
tcfs_gpwfile_scan(gid_t gid, int *found, int remove) {

	GDBM_FILE pwdb;
	datum key,nextkey;
	char *search,*gidstring,*tmp;
	int flag=GDBM_READER;
	
	*found=0;
	
	if (access(TCFS_GPWDBPATH,F_OK)<0) {
        	tcfs_errno=ER_GDBMEXIST;
                return 1;
	}
                
	if (remove==TCFS_DBDELETE)
		flag=GDBM_WRITER;

        if ((pwdb=gdbm_open(TCFS_GPWDBPATH,TCFS_GPWDBSIZE,flag,TCFS_GPWDBMODE,NULL))==NULL) {
		tcfs_errno=ER_GDBMOPEN;
                return 1;
	}

	gidstring=(char*)malloc(1+TCFS_DIMGID+1); /* \033 + gid + \0 */
	sprintf(gidstring,"\033%d",gid);
	key=gdbm_firstkey(pwdb);
	
	while (key.dptr!=NULL) {

		nextkey=gdbm_nextkey(pwdb,key);

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_gpwfile_scan: Value of key.dptr and key.dsize:%p=%s %d\n", key.dptr,key.dptr, key.dsize);
	fprintf(stderr,"tcfs_gpwfile_scan: Value of gid:%s - size:%d\n",gidstring,strlen(gidstring));
#endif
		
		tmp=malloc(key.dsize+1);
		strncpy(tmp,key.dptr,key.dsize);
		tmp[key.dsize]='\0';
		search=strchr(tmp,'\033');	

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_gpwfile_scan: Value of search:%s size:%d\n",search,strlen(search));
#endif

		if (strcmp(search,gidstring)==0) {
			
			*found=1;

			if (remove==TCFS_DBDELETE) { 
				
#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_gpwfile_scan: I must delete an entry.\n");
#endif

				if (gdbm_delete(pwdb,key)!=0) {
					gdbm_close(pwdb);
					free(tmp);
					free(gidstring);
					free(key.dptr);
					free(nextkey.dptr);
					found=0;
					return(tcfs_errno=ER_GDBMDELETE);
				}
			}
			else {
			
					free(tmp);
					free(gidstring);
					free(key.dptr);
					free(nextkey.dptr);
					gdbm_close(pwdb);
					return(tcfs_errno=OK);
			}
		}

		free(tmp);
		free(key.dptr);
		key=nextkey;
	}

	free(gidstring);
	free(key.dptr);

	if (remove==TCFS_DBDELETE)
		gdbm_reorganize(pwdb);
	gdbm_close(pwdb);

	return(tcfs_errno=OK);
} 		
 
/* It re-encrypts the user's tcfskey crypted with oldpassword, with new password. On success it returns 0.*/        
int tcfs_change_userkey(char* user, char* oldpassword, char* newpassword) {

	tcfspwdb *entry;
	char *old_key, *key, *new_key;

	if ((entry=tcfs_pwfile_fetch(user))==NULL) return(tcfs_errno);

	key=malloc(UUKEYSIZE);
	tcfs_pwdb_fetch(entry,TCFS_DBKEY,key);
	if (strlen(key)!=0) {
		if ((old_key=tcfs_decrypt_key(key,oldpassword,TCFS_USER_KEY))==NULL) {
			free(key);
			return(tcfs_errno);
		}
		if ((new_key=tcfs_encrypt_key(old_key,newpassword,TCFS_USER_KEY))==NULL) {
			free(key);
			free(old_key);
			return(tcfs_errno);
		}

#ifdef TCFS_DEBUG
	fprintf(stderr,"Old crypted key: %s\n",key);
	fprintf(stderr,"New crypted key: %s\n",new_key);
#endif

		if (tcfs_pwdb_edit(entry,TCFS_DBKEY,new_key)!=0) {
			free(key);
			free(old_key);
			free(new_key);
			return(tcfs_errno);
		}
		
		if (tcfs_pwfile_store(user,entry,TCFS_DBREPLACE)!=0) {
			free(key);
			free(old_key);
			free(new_key);
			return(tcfs_errno);
		}
	}
	
	return(tcfs_errno=OK);
}


/* It re-encrypts the group's tcfskey crypted with oldpassword, with new password. On success it returns 0.*/        
int tcfs_change_groupkey(char* user, char* oldpassword, char* newpassword) {
	
	char *old_key, *new_key;
	GDBM_FILE pwdb;
	datum key, nextkey, content;
	int flag=GDBM_WRITER;
	
	if (access(TCFS_GPWDBPATH,F_OK)<0) {
        	tcfs_errno=ER_GDBMEXIST;
                return 1;
	}
                
        if ((pwdb=gdbm_open(TCFS_GPWDBPATH,TCFS_GPWDBSIZE,flag,TCFS_GPWDBMODE,NULL))==NULL) {
		tcfs_errno=ER_GDBMOPEN;
                return 1;
	}

	key=gdbm_firstkey(pwdb);
	
	while (key.dptr!=NULL) {

		nextkey=gdbm_nextkey(pwdb,key);
		
		if (strncmp(user,key.dptr,strlen(user))==0) {
			
			content=gdbm_fetch(pwdb,key);
		
			if ((old_key=tcfs_decrypt_key(((tcfsgpwdb*) content.dptr)->groupkey,oldpassword,TCFS_GROUP_KEY))==NULL) {
				free(key.dptr);
				free(nextkey.dptr);
				free(content.dptr);			
				return(tcfs_errno);
			}
			if ((new_key=tcfs_encrypt_key(old_key,newpassword,TCFS_GROUP_KEY))==NULL) {
				free(key.dptr);
				free(nextkey.dptr);
				free(content.dptr);			
				free(old_key);
				return(tcfs_errno);
			}

#ifdef TCFS_DEBUG
	fprintf(stderr,"Old crypted key: %s\n",((tcfsgpwdb*) content.dptr)->groupkey);
	fprintf(stderr,"New crypted key: %s\n",new_key);
#endif
		
			memcpy(((tcfsgpwdb*) content.dptr)->groupkey,new_key,UUKEYSIZE);
		
			free(old_key);
			free(new_key);
			
			if (gdbm_store(pwdb,key,content,GDBM_REPLACE)!=0) {
				free(key.dptr);
				free(nextkey.dptr);
				free(content.dptr);			
				tcfs_errno=ER_GDBMSTORE;
				return(tcfs_errno);
			}
		
			free(content.dptr);			
		}						
						
		free(key.dptr);
		key=nextkey;
	}

	free(key.dptr);
	gdbm_reorganize(pwdb);
	gdbm_close(pwdb);

	return(tcfs_errno=OK);
}
