/* 
 * 	$Source: /home/fergia/CVS//tcfs-openbsd/tcfslib-1.0.0/lib/tcfs_key.c,v $
 *	$Revision: 1.1.1.1 $
 * 	$Date: 2000/06/07 12:21:31 $
 *	$State: Exp $
 *      $Author: fergia $
 *      $Lockers$	
 */


#include <stdio.h>
#include <string.h>
#include <sys/param.h>
#include <sys/mount.h>
#include <dlfcn.h>
#include <stdlib.h>

#include "tcfs_global.h"
#include "tcfs_errno.h"
#include "tcfs_key.h"


/* This is initialized with the dynamic library's name */
static char libname[LINE]="\0"; 

static void* handle;
static void* (*init_key)(char*);
static char* (*encrypt)(char*,int,void*,int*);
static char* (*decrypt)(char*,int,void*,int*);
static void (*delete_key)(void*);

extern char* tcfs_base64_encode(char*, int);
extern char* tcfs_base64_decode(char*, int*);

/* It fetch the dynamic library's name of the cipher to use. The result is put in libname. On success it return 0. It is also an internal function. */
static int _tcfs_get_library_name(void) {

	FILE* conf;
	char buf[LINE],lib[LINE],*temp;
	unsigned short state=0;

	tcfs_errno=OK;


#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_get_library_name: Execute _tcfs_get_library_name.\n");
#endif

	if ((conf=fopen(TCFS_CIPHER_CONF,"r"))==NULL)  
		return(tcfs_errno=ER_CONF_FILE);

		
	while (fgets(buf,LINE,conf)!=NULL) { 

		if (*buf!=TCFS_CIPHER_COMMENT && *buf!='\n') {
			
			temp=strtok(buf,DELIM);

			if (temp==NULL) continue; /* The line is empty */
			if (*temp==TCFS_CIPHER_COMMENT) continue;
	
			if (state==0) {
				if (strcmp(temp,TCFS_CIPHER_KEYWORD)==0) {
					temp=strtok(NULL,DELIM);
					if (temp==NULL) break; 
					strcpy(lib,temp);
					state=1;

#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_get_library_name: library: %s\n",lib);
#endif

					continue;	
				}
				else break;
			}
			if (state==1) {
				if (strcmp(temp,lib)==0) {
					temp=strtok(NULL,DELIM);
					if (temp==NULL) break; 
					strcpy(libname,temp);

#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_get_library_name: library name: %s\n",libname);
#endif
					break;
				}
			}
		}
	}
	if (strlen(libname)==0)  
		tcfs_errno=ER_BAD_CONF_FILE;
	
	
	fclose(conf);
	return (tcfs_errno);
}

/* It is used to load function from the dynamic library. On success it returns 0. It is also an internal function. */
static int _tcfs_load_cipher_function(void) {
	
#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_load_cipher_function: in tcfs_load_cipher_function.\n");
#endif

	handle=dlopen(libname,DL_LAZY);

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_encrypt_key: Value of handle %p\n",handle);
#endif
	
	if (handle==NULL) 
		return(tcfs_errno=ER_LOAD_LIBRARY);
	
#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_load_cipher_function: Loading init_key\n");
#endif

	init_key=(void* (*)(char*)) dlsym(handle,TCFS_INIT_KEY);
	if (dlerror()!=NULL) {
		dlclose(handle);
		return(tcfs_errno=ER_LOAD_FUNCTION);
	}

#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_load_cipher_function: init_key: %p\n",init_key);
#endif

	encrypt=(char* (*)(char*, int, void*, int*)) dlsym(handle,TCFS_ENCRYPT);		
#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_load_cipher_function: encrypt: %p\n",encrypt);
#endif
	if (dlerror()!=NULL) {
		dlclose(handle);
		return(tcfs_errno=ER_LOAD_FUNCTION);
	}

#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_load_cipher_function: loading %s\n",TCFS_DECRYPT);
#endif

	decrypt=(char* (*)(char*, int, void*, int*)) dlsym(handle,TCFS_DECRYPT);		
	if (dlerror()!=NULL) {
		dlclose(handle);
		return(tcfs_errno=ER_LOAD_FUNCTION);
	}
	
#ifdef TCFS_DEBUG
	fprintf(stderr,"_tcfs_load_cipher_function: loading %s.\n",TCFS_DELETE_KEY);
#endif

	delete_key=(void (*)(void*)) dlsym(handle,TCFS_DELETE_KEY);
	if (dlerror()!=NULL) {
		dlclose(handle);
		return(tcfs_errno=ER_LOAD_FUNCTION);
	}
	return(tcfs_errno=OK);
}

/* It is used to unload the dynamic library. It's an internal function.*/
static void _tcfs_unload_cipher_function(void) {

	dlclose(handle);

}
	
/* It is used to encrypt the tcfs key of an user. The encrypt's key is the user's password. On success it returns the encrypted tcfs_key. On error it returns NULL. Flag is used to distinguish the user key from the group key. Note that the returned string can be greater than the input, but it's a base 64 encoding string. So it has not the '\0' character. */
char* 
tcfs_encrypt_key (char* tcfs_key, char *password, int flag)
{

	void *ks;
	char *encrypted,*encoded;
	char pass[_PASSWORD_LEN];
	int keysize=(flag==TCFS_USER_KEY)?KEYSIZE:GROUPKEYSIZE,len;

#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_encrypt_key: Execute tcfs_encrypt_key.\n");
	fprintf(stderr,"tcfs_encrypt_key: init_key %X\n",(unsigned int) *init_key);
	fprintf(stderr,"tcfs_encrypt_key: Value of handle %p\n",handle);
	fprintf(stderr,"tcfs_encrypt_key: Value of libname %s\n",libname);
#endif

	tcfs_errno=OK;

	if (strlen(libname)==0) { 
		if (_tcfs_get_library_name()!=0) return (char*)0;
  
#ifdef TCFS_DEBUG
	fprintf(stderr,"tcfs_encrypt_key: _tcfs_get_library_name return success.\n");
#endif
	
	}

	if (_tcfs_load_cipher_function()!=0) return (char*)0;
	
	strcpy (pass, password);
  	while (strlen(pass)<KEYSIZE)
      		strcat (pass, password);
   	
#ifdef TCFS_DEBUG
	fprintf(stderr,"\ntcfs_encrypt_key: Before init_key pass=%s, strlen(pass)=%d",pass,strlen(pass));
	fprintf(stderr,"\ntcfs_encrypt_key: init_key %X",(unsigned int) *init_key);
#endif

	ks=init_key(pass);
	encrypted=encrypt(tcfs_key,keysize,ks,&len);
	delete_key(ks);

#ifdef TCFS_DEBUG
	{
	int i=0;
	fprintf(stderr,"\ntcfs_encrypt_key: tcfs_key crypted = ");
	for (;i<len;i++)
		fprintf(stderr,"%X ",(unsigned char) encrypted[i]);
	fprintf(stderr,"\n");
	fprintf(stderr,"\ntcfs_encrypt_key: Value of handle %p",handle);
	}
#endif
	
	_tcfs_unload_cipher_function();		

  	encoded=(char*) tcfs_base64_encode (encrypted,len);

	return(encoded);
}


/* It is used to decrypt the tcfs key of an user. The decrypt's key is the user's password. On success it returns the decrypted tcfs_key. On error it returns NULL. Flag is used to distinguish the user key from the group key. Note that the returned string must be keysize long. */
char * 
tcfs_decrypt_key (char *tcfs_key, char *password, int flag)
{
  	char pass[_PASSWORD_LEN];
	char *decrypted,*decoded;
	void *ks;
	int len,keysize=(flag==TCFS_USER_KEY)?KEYSIZE:GROUPKEYSIZE,decryptlen;

#ifdef TCFS_DEBUG
	fprintf(stderr,"\ntcfs_decrypt_key: Execute tcfs_decrypt_key");
	fprintf(stderr,"\ntcfs_decrypt_key: Value of libname %s",libname);
#endif

	tcfs_errno=OK;

	if (strlen(libname)==0) { 
		if (_tcfs_get_library_name()!=0) return (char*)0;
  
#ifdef TCFS_DEBUG
	fprintf(stderr,"\ntcfs_decrypt_key: _tcfs_get_library_name return success.");
#endif
	
	}
	
	if (_tcfs_load_cipher_function()!=0) return (char*)0;
	

#ifdef TCFS_DEBUG
	fprintf(stderr,"\ntcfs_decrypt_key: before tcfs_base64_decode, value of tcfs_key: %s",tcfs_key);
#endif

	if ((decoded = (char*) tcfs_base64_decode (tcfs_key,&len))==NULL) {
		_tcfs_unload_cipher_function();	
		return (char*) 0;
	}
	

	strcpy (pass, password);
  	while (strlen(pass)<KEYSIZE)
      		strcat (pass, password);

	ks=init_key(pass);

#ifdef TCFS_DEBUG
	fprintf(stderr,"\ntcfs_decrypt_key: after init_key, value of len: %d",len);
#endif

	/* On success, decryptlen must ne egual to keysize */
	decrypted=decrypt(decoded,len,ks,&decryptlen);
	delete_key(ks);
		
#ifdef TCFS_DEBUG
	fprintf(stderr,"\ntcfs_decrypt_key: Len of the decrypted string: %d\n",decryptlen);
#endif

	_tcfs_unload_cipher_function();	
	free(decoded);

	if (decryptlen!=keysize) {
		tcfs_errno=ER_TCFS_DECRYPT;
		return (char*)0;
	}

	return(decrypted);
}



/* It's the interface to the mount function. It allows the call of the mount function on the tcfs filesystems. On success it returns 0. It's an internal function. */

static int _tcfs_mount_interface(const char *filesystem, struct tcfs_args* data) { 
	
	struct statfs info;
	
	tcfs_errno=OK;


	if (statfs(filesystem,&info)!=0)
		return (tcfs_errno=ER_TCFS_OPEN);


	if (strcmp(info.f_fstypename,TCFS_FSTYPE)!=0)
		return (tcfs_errno=ER_NO_TCFS);

	if (mount(TCFS_FSTYPE,filesystem,MNT_UPDATE,(void*) data)!=0)
		return(tcfs_errno=ER_MOUNT);

	return(tcfs_errno);
	
}

/* It passes the tcfs_key of an user to the kernel. The user_name is used to get the uid. On success it returns 0.*/

int tcfs_user_enable (const char* filesystem, uid_t uid, char *tcfskey) {
  
	struct tcfs_args data;
		
	tcfs_errno=OK;	

	memcpy(data.tcfs_key,tcfskey,sizeof(data.tcfs_key));
	data.cmd=TCFS_PUT_UIDKEY;
	data.user=uid;
	

	if (_tcfs_mount_interface(filesystem, &data)!=0)
		if (tcfs_errno==ER_MOUNT) 
			tcfs_errno=ER_ENABLE;


 	 return(tcfs_errno);	
	
}

/* It removes the tcfs_key from the kernel; uid is the user id. On success it returns 0. */

int tcfs_user_disable(const char* filesystem, uid_t uid) {

	struct tcfs_args data;

	tcfs_errno=OK;

	data.cmd=TCFS_RM_UIDKEY;     
        data.user=uid;	
  
	if (_tcfs_mount_interface(filesystem, &data)!=0)
                if (tcfs_errno==ER_MOUNT)
                        tcfs_errno=ER_DISABLE;

         return(tcfs_errno);
}


/* It passes a tcfs_key of a process to the kernel. uid is the user id of the process' owner, pid is the process id and tcfskey is the TCFS key to pass to the kernel. On success it returns 0.*/ 

int tcfs_process_enable(const char* filesystem, uid_t uid, pid_t pid, char* tcfskey) {

	struct tcfs_args data;

	tcfs_errno=OK;

	data.cmd=TCFS_PUT_PIDKEY;
	data.user=uid;
	data.proc=pid;
	memcpy(data.tcfs_key,tcfskey,sizeof(data.tcfs_key));

	if (_tcfs_mount_interface(filesystem,&data)!=0)
		if (tcfs_errno==ER_MOUNT)
			tcfs_errno=ER_ENABLE;

	return(tcfs_errno);
}
	

/* It removes a tcfskey of a process from the kernel. uid is the user id of the process' owner and pid is the process id of the process. On success it returns 0. */

int tcfs_process_disable(const char *filesystem, uid_t uid, pid_t pid) {

	struct tcfs_args data;

	tcfs_errno=OK;

	data.cmd=TCFS_RM_PIDKEY;
	data.user=uid;
	data.proc=pid;

	if (_tcfs_mount_interface(filesystem,&data)!=0)
		if (tcfs_errno==ER_MOUNT)
			tcfs_errno=ER_DISABLE;

	return(tcfs_errno);
} 

/* It passes the tcfs group key's piece to the kernel. uid is the user id and gid is the group id of the group to wich the user belongs. threshold is the minimum number of user to get the key. The key to pass to the kernel is tcfskey. On success it returns 0. */ 

int tcfs_group_enable(const char* filesystem, uid_t uid, gid_t gid, int threshold, char* tcfskey) {

	struct tcfs_args data;
	
	tcfs_errno=OK;	

	data.cmd=TCFS_PUT_GIDKEY;
	data.user=uid;
	data.group=gid;
	data.treshold=threshold;
	memcpy(data.tcfs_key,tcfskey,sizeof(data.tcfs_key));

	if (_tcfs_mount_interface(filesystem,&data)!=0)
                if (tcfs_errno==ER_MOUNT)
                        tcfs_errno=ER_ENABLE; 

        return(tcfs_errno);
	
}


/* It is used to remove the TCFS group key's piece from the kernel. uid is the user id and gid is the group id of group to which users belongs. On success it returns 0. */

int tcfs_group_disable(const char* filesystem, uid_t uid, gid_t gid) {
		
	struct tcfs_args data;
  
	tcfs_errno=OK;	
	
	data.cmd=TCFS_RM_GIDKEY;
	data.user=uid;
	data.group=gid;

	if (_tcfs_mount_interface(filesystem,&data)!=0)
                if (tcfs_errno==ER_MOUNT)
                        tcfs_errno=ER_DISABLE;

        return(tcfs_errno);
}

/* It gives back information about the filesystem: it gives back the status, the number of user keys, the number of group keys, the TCFS version, the cipher, the cipher key's length and the cipher's version. On success it returns 0. */ 

int tcfs_getstatus(const char* filesystem, tcfs_status* status) {

	struct tcfs_args data;

	tcfs_errno=OK;

	data.cmd=TCFS_GET_STATUS;

	if (_tcfs_mount_interface(filesystem,&data)!=0)
                if (tcfs_errno==ER_MOUNT)
                        tcfs_errno=ER_STATUS;

	*status=data.st;

        return(tcfs_errno);
 
}
