/*
Validation routines
This file is in the public domain
*/

#include <fstream>
#include <cctype>
#include <string>
#include <vector>

#include <opencl/encoder.h>
using namespace OpenCL_types;

OpenCL::Filter* lookup(const std::string&, const std::string&,
                       const std::string&);

bool failed_test(const std::string&, const std::string&,
                 const std::string&, const std::string&,
                 const std::string&);

std::vector<std::string> parse(const std::string&);

u32bit do_validation_tests(const std::string& filename)
   {
   std::ifstream test_data(filename.c_str());

   if(!test_data)
       {
       std::cout << "Couldn't open test file " << filename << std::endl;
       std::exit(1);
       }

   u32bit errors = 0, alg_count = 0;
   std::string algorithm;
   while(!test_data.eof())
      {
      if(test_data.bad() || test_data.fail())
         {
         std::cout << "File I/O error." << std::endl;
         std::exit(1);
         }

      std::string line;
      std::getline(test_data, line);
      if(line.size() == 0 || line[0] == '#')
         continue;

      // check for lines consisting of only <space> and/or <tab>: ignore
      bool ignoreline = true;
      for(u32bit j = 0; j != line.size(); j++)
         if(line[j] != ' ' && line[j] != '\t')
            ignoreline = false;
      if(ignoreline) continue;

      while(line[line.size()-1] == '\\' && !test_data.eof())
         {
         line.replace(line.size()-1, 1, "");
         std::string nextline;
         std::getline(test_data, nextline);
         line += nextline;
         }

      if(line[0] == '[' && line[line.size() - 1] == ']')
         {
         algorithm = line.substr(1, line.size() - 2);
         alg_count = 0;
         continue;
         }

      std::vector<std::string> substr = parse(line);

      alg_count++;
      if(failed_test(algorithm, substr[0], substr[1], substr[2], substr[3]))
         {
         std::cout << "ERROR: \"" << algorithm << "\" failed test #"
                   << alg_count << std::endl;
         errors++;
         }
      }
   return errors;
   }

std::vector<std::string> parse(const std::string& line)
   {
   const char DELIMITER = ':';
   std::vector<std::string> substr;
   std::string::size_type start = 0, end = line.find(DELIMITER);
   while(end != std::string::npos)
      {
      substr.push_back(line.substr(start, end-start));
      start = end+1;
      end = line.find(DELIMITER, start);
      }
   if(line.size() > start)
      substr.push_back(line.substr(start));
   while(substr.size() <= 4) // at least 4 substr, some possibly empty
      substr.push_back("");
   return substr;
   }

bool failed_test(const std::string& algo,     const std::string& in,
                 const std::string& expected, const std::string& key,
                 const std::string& iv)
   {
   OpenCL::Filter* test = lookup(algo, key, iv);
   if(test == 0)
      {
      std::cout << "ERROR: \"" + algo + "\" is not a known algorithm name."
                << std::endl;
      std::exit(1);
      }

   OpenCL::Pipe pipe(new OpenCL::HexDecoder, test, new OpenCL::HexEncoder);
   pipe.write((byte*)in.c_str(), in.size());
   pipe.close();
   OpenCL::SecureVector<byte> output = pipe.read_all();

   if(output.size() != expected.size())
      return true;

   for(u32bit j = 0; j != output.size(); j++)
      if(output[j] != expected[j] && output[j] != std::toupper(expected[j]))
         return true;

   return false;
   }
