/*************************************************
* Cipher Modes Header File                       *
* (C) 1999-2001 The OpenCL Project               *
*************************************************/

#ifndef OPENCL_CIPHER_MODES_H__
#define OPENCL_CIPHER_MODES_H__

#include <opencl/filtbase.h>
#include <opencl/symkey.h>

namespace OpenCL {

/*************************************************
* Cipher Mode Base Class                         *
*************************************************/
class CipherMode : public Filter { };

/*************************************************
* CBC w/Padding Mode Encryption                  *
*************************************************/
template<typename B>
class CBC_wPadding_Encryption : public CipherMode
   {
   public:
      static const u32bit BLOCKSIZE = B::BLOCKSIZE, IVSIZE = B::BLOCKSIZE,
                          KEYLENGTH = B::KEYLENGTH;
      void write(const byte[], u32bit);
      void final();
      CBC_wPadding_Encryption(const BlockCipherKey&,
                              const BlockCipherModeIV&);
   private:
      void process_block();
      B cipher;
      SecureBuffer<byte, BLOCKSIZE> state;
      u32bit position;
   };

/*************************************************
* CBC w/Padding Mode Decryption                  *
*************************************************/
template<typename B>
class CBC_wPadding_Decryption : public CipherMode
   {
   public:
      static const u32bit BLOCKSIZE = B::BLOCKSIZE, IVSIZE = B::BLOCKSIZE,
                          KEYLENGTH = B::KEYLENGTH;
      void write(const byte[], u32bit);
      void final();
      CBC_wPadding_Decryption(const BlockCipherKey&,
                              const BlockCipherModeIV&);
   private:
      void process_block();
      B cipher;
      SecureBuffer<byte, BLOCKSIZE> state, buffer, temp;
      u32bit position;
   };

/*************************************************
* CFB Mode Encryption                            *
*************************************************/
template<typename B>
class CFB_Encryption : public CipherMode
   {
   public:
      static const u32bit BLOCKSIZE = B::BLOCKSIZE, IVSIZE = B::BLOCKSIZE,
                          KEYLENGTH = B::KEYLENGTH;
      void write(const byte[], u32bit);
      CFB_Encryption(const BlockCipherKey&, const BlockCipherModeIV&,
                     u32bit = BLOCKSIZE);
   private:
      void feedback();
      const u32bit FEEDBACKSIZE;
      B cipher;
      SecureBuffer<byte, BLOCKSIZE> state, reg;
      u32bit position;
   };

/*************************************************
* CFB Mode Decryption                            *
*************************************************/
template<typename B>
class CFB_Decryption : public CipherMode
   {
   public:
      static const u32bit BLOCKSIZE = B::BLOCKSIZE, IVSIZE = B::BLOCKSIZE,
                          KEYLENGTH = B::KEYLENGTH;
      void write(const byte[], u32bit);
      CFB_Decryption(const BlockCipherKey&, const BlockCipherModeIV&,
                     u32bit = BLOCKSIZE);
   private:
      void feedback();
      const u32bit FEEDBACKSIZE;
      B cipher;
      SecureBuffer<byte, BLOCKSIZE> state, reg;
      u32bit position;
   };

/*************************************************
* OFB Mode                                       *
*************************************************/
template<typename B>
class OFB : public CipherMode
   {
   public:
      static const u32bit BLOCKSIZE = B::BLOCKSIZE, IVSIZE = B::BLOCKSIZE,
                          KEYLENGTH = B::KEYLENGTH;
      void write(const byte[], u32bit);
      OFB(const BlockCipherKey&, const BlockCipherModeIV&);
   private:
      void feedback() { cipher.encrypt(state); position = 0; }
      B cipher;
      SecureBuffer<byte, BLOCKSIZE> state, buffer;
      u32bit position;
   };

/*************************************************
* Counter Mode                                   *
*************************************************/
template<typename B>
class Counter : public CipherMode
   {
   public:
      static const u32bit BLOCKSIZE = B::BLOCKSIZE, IVSIZE = B::BLOCKSIZE,
                          KEYLENGTH = B::KEYLENGTH;
      void write(const byte[], u32bit);
      Counter(const BlockCipherKey&, const BlockCipherModeIV&);
   private:
      void increment_counter();
      B cipher;
      SecureBuffer<byte, BLOCKSIZE> state, counter;
      u32bit position;
   };

}

#endif

#ifndef OPENCL_MODES_ICC__
#define OPENCL_MODES_ICC__
#include <opencl/modes.icc>
#endif
