/*************************************************
* OpenCL Basic Interface Header File             *
* (C) 1999-2001 The OpenCL Project               *
*************************************************/

#ifndef OPENCL_OPENCL_H__
#define OPENCL_OPENCL_H__

#include <string>
#include <opencl/exceptn.h>
#include <opencl/secmem.h>
#include <opencl/version.h>

namespace OpenCL {

/*************************************************
* Symmetric Algorithm Base Class                 *
*************************************************/
class SymmetricAlgorithm
   {
   public:
      virtual void set_key(const byte[], u32bit) throw(InvalidKeyLength) = 0;
      virtual bool valid_keylength(u32bit) const;
      SymmetricAlgorithm(u32bit, u32bit = 0, u32bit = 1);
      virtual ~SymmetricAlgorithm() {}
   private:
      const u32bit max, min, mod;
   };

/*************************************************
* Block Cipher Base Class                        *
*************************************************/
class BlockCipher : public SymmetricAlgorithm
   {
   public:
      const u32bit BLOCKSIZE, KEYLENGTH;
      virtual void encrypt(const byte[], byte[]) const = 0;
      virtual void decrypt(const byte[], byte[]) const = 0;
      virtual void encrypt(byte block[]) const = 0;
      virtual void decrypt(byte block[]) const = 0;
      virtual void clear() throw() = 0;
      BlockCipher(u32bit, u32bit, u32bit = 0, u32bit = 1);
      virtual ~BlockCipher() {}
   };

/*************************************************
* Stream Cipher Base Class                       *
*************************************************/
class StreamCipher : public SymmetricAlgorithm
   {
   public:
      const u32bit KEYLENGTH;
      void encrypt(const byte[], byte[], u32bit);
      void decrypt(const byte[], byte[], u32bit);
      void encrypt(byte[], u32bit);
      void decrypt(byte[], u32bit);
      virtual void clear() throw() = 0;
      StreamCipher(u32bit, u32bit = 0, u32bit = 1);
      virtual ~StreamCipher() {}
   private:
      virtual void cipher(const byte[], byte[], u32bit) = 0;
   };

/*************************************************
* Random Access Stream Cipher Base Class         *
*************************************************/
struct RandomAccessStreamCipher : public StreamCipher
   {
   virtual void seek(u32bit) = 0;
   RandomAccessStreamCipher(u32bit, u32bit = 0, u32bit = 1);
   virtual ~RandomAccessStreamCipher() {}
   };

/*************************************************
* Hash Function Base Class                       *
*************************************************/
struct HashFunction
   {
   const u32bit HASHLENGTH;
   virtual void process(const byte[], u32bit, byte[]);
   virtual void update(const byte[], u32bit) = 0;
   virtual void final(byte[]) = 0;
   virtual void clear() throw() = 0;
   HashFunction(u32bit hashlen) : HASHLENGTH(hashlen) {}
   virtual ~HashFunction() {}
   };

/*************************************************
* Message Authentication Code Base Class         *
*************************************************/
struct MessageAuthCode : public SymmetricAlgorithm
   {
   const u32bit MACLENGTH, KEYLENGTH;
   virtual void process(const byte[], u32bit, byte[]);
   virtual void update(const byte[], u32bit) = 0;
   virtual void final(byte[]) = 0;
   virtual void clear() throw() = 0;
   MessageAuthCode(u32bit, u32bit, u32bit = 0, u32bit = 1);
   virtual ~MessageAuthCode() {}
   };

/*************************************************
* Random Number Generator Base Class             *
*************************************************/
struct RandomNumberGenerator
   {
   virtual byte random() = 0;
   virtual void randomize(byte[], u32bit);
   virtual void add_entropy(const byte[], u32bit) throw() = 0;
   virtual void add_entropy(class EntropySource&, bool) = 0;
   virtual void clear() throw() = 0;
   virtual ~RandomNumberGenerator() {}
   };

/*************************************************
* Entropy Source Base Class                      *
*************************************************/
struct EntropySource
   {
   virtual u32bit get_entropy(byte[], u32bit) = 0;
   virtual ~EntropySource() {}
   };

}

#endif
