/*************************************************
* Blowfish Source File                           *
* (C) 1999-2001 The OpenCL Project               *
*************************************************/

#include <opencl/blowfish.h>

namespace OpenCL {

/*************************************************
* Blowfish Encryption                            *
*************************************************/
void Blowfish::encrypt(const byte in[BLOCKSIZE], byte out[BLOCKSIZE]) const
   {
   u32bit left  = make_u32bit(in[0], in[1], in[2], in[3]),
          right = make_u32bit(in[4], in[5], in[6], in[7]);
   round(left, right, 0); round(right, left, 1); round(left, right, 2);
   round(right, left, 3); round(left, right, 4); round(right, left, 5);
   round(left, right, 6); round(right, left, 7); round(left, right, 8);
   round(right, left, 9); round(left, right,10); round(right, left,11);
   round(left, right,12); round(right, left,13); round(left, right,14);
   round(right, left,15); left ^= PBox[16]; right ^= PBox[17];
   out[0] = get_byte(0, right); out[1] = get_byte(1, right);
   out[2] = get_byte(2, right); out[3] = get_byte(3, right);
   out[4] = get_byte(0, left);  out[5] = get_byte(1, left);
   out[6] = get_byte(2, left);  out[7] = get_byte(3, left);
   }

/*************************************************
* Blowfish Decryption                            *
*************************************************/
void Blowfish::decrypt(const byte in[BLOCKSIZE], byte out[BLOCKSIZE]) const
   {
   u32bit left  = make_u32bit(in[0], in[1], in[2], in[3]),
          right = make_u32bit(in[4], in[5], in[6], in[7]);
   round(left, right,17); round(right, left,16); round(left, right,15);
   round(right, left,14); round(left, right,13); round(right, left,12);
   round(left, right,11); round(right, left,10); round(left, right, 9);
   round(right, left, 8); round(left, right, 7); round(right, left, 6);
   round(left, right, 5); round(right, left, 4); round(left, right, 3);
   round(right, left, 2); left ^= PBox[1]; right ^= PBox[0];
   out[0] = get_byte(0, right); out[1] = get_byte(1, right);
   out[2] = get_byte(2, right); out[3] = get_byte(3, right);
   out[4] = get_byte(0, left);  out[5] = get_byte(1, left);
   out[6] = get_byte(2, left);  out[7] = get_byte(3, left);
   }


/*************************************************
* Blowfish Round                                 *
*************************************************/
void Blowfish::round(u32bit& left, u32bit& right, u32bit round) const
   {
   left  ^= PBox[round];
   right ^= ((SBox1[get_byte(0, left)]  + SBox2[get_byte(1, left)]) ^
              SBox3[get_byte(2, left)]) + SBox4[get_byte(3, left)];
   }

/*************************************************
* Blowfish Key Setup                             *
*************************************************/
void Blowfish::set_key(const byte key[], u32bit length) throw(InvalidKeyLength)
   {
   if(!valid_keylength(length))
      throw InvalidKeyLength("Blowfish", length);
   clear();
   for(u32bit j = 0, k = 0; j != 18; j++, k += 4)
      PBox[j] ^= make_u32bit(key[(k  ) % length], key[(k+1) % length],
                             key[(k+2) % length], key[(k+3) % length]);
   u32bit left = 0, right = 0;
   generate_sbox(PBox,  18,  left, right);
   generate_sbox(SBox1, 256, left, right);
   generate_sbox(SBox2, 256, left, right);
   generate_sbox(SBox3, 256, left, right);
   generate_sbox(SBox4, 256, left, right);
   }

/*************************************************
* Generate one of the SBoxes                     *
*************************************************/
void Blowfish::generate_sbox(u32bit Box[], u32bit size,
                             u32bit& left, u32bit& right) const
   {
   for(u32bit j = 0; j != size; j += 2)
      {
      round(left, right, 0); round(right, left, 1); round(left, right, 2);
      round(right, left, 3); round(left, right, 4); round(right, left, 5);
      round(left, right, 6); round(right, left, 7); round(left, right, 8);
      round(right, left, 9); round(left, right,10); round(right, left,11);
      round(left, right,12); round(right, left,13); round(left, right,14);
      round(right, left,15); u32bit temp = right; right = left ^ PBox[16];
      left = temp ^ PBox[17]; Box[j] = left; Box[j+1] = right;
      }
   }

/*************************************************
* Clear memory of sensitive data                 *
*************************************************/
void Blowfish::clear() throw()
   {
   PBox.copy(PBOX, 18);
   SBox1.copy(SBOX1, 256);
   SBox2.copy(SBOX2, 256);
   SBox3.copy(SBOX3, 256);
   SBox4.copy(SBOX4, 256);
   }

}
