/*************************************************
* Checksum Source File                           *
* (C) 1999-2001 The OpenCL Project               *
*************************************************/

#include <opencl/checksum.h>

namespace OpenCL {

/*************************************************
* Adler32 Hash                                   *
*************************************************/
void Adler32::hash(const byte input[], u32bit length)
   {
   for(u32bit j = 0; j != length; j++)
      {
      S1 += input[j];
      S2 += S1;
      }
   S1 %= 65521; S2 %= 65521;
   }

/*************************************************
* Update a Adler32 Hash                          *
*************************************************/
void Adler32::update(const byte input[], u32bit length)
   {
   static const u32bit PROCESS_AMOUNT = 5551;
   while(length >= PROCESS_AMOUNT)
      {
      hash(input, PROCESS_AMOUNT);
      input += PROCESS_AMOUNT;
      length -= PROCESS_AMOUNT;
      }
   hash(input, length);
   }

/*************************************************
* Finalize an Adler32 Hash                       *
*************************************************/
void Adler32::final(byte output[HASHLENGTH])
   {
   output[0] = get_byte(2, S2);
   output[1] = get_byte(3, S2);
   output[2] = get_byte(2, S1);
   output[3] = get_byte(3, S1);
   clear();
   }

/*************************************************
* Update a CRC24 Hash                            *
*************************************************/
void CRC24::update(const byte in[], u32bit length)
   {
   for(u32bit j = 0; j != length; j++)
      crc_val = ((crc_val << 8) ^ TABLE[(crc_val >> 16) ^ in[j]]) & 0xFFFFFF;
   }

/*************************************************
* Finalize a CRC24 Hash                          *
*************************************************/
void CRC24::final(byte output[HASHLENGTH])
   {
   for(u32bit j = 0; j != HASHLENGTH; j++)
      output[j] = get_byte(j+1, crc_val);
   clear();
   }

/*************************************************
* Update a CRC32 Hash                            *
*************************************************/
void CRC32::update(const byte input[], u32bit length)
   {
   for(u32bit j = 0; j != length; j++)
      crc_val = TABLE[(crc_val ^ input[j]) & 0xFF] ^ (crc_val >> 8);
   }

/*************************************************
* Finalize a CRC32 Hash                          *
*************************************************/
void CRC32::final(byte output[HASHLENGTH])
   {
   crc_val ^= 0xFFFFFFFF;
   for(u32bit j = 0; j != HASHLENGTH; j++)
      output[j] = get_byte(j, crc_val);
   clear();
   }

}
