/*************************************************
* OpenCL Base Interface Source File              *
* (C) 1999-2001 The OpenCL Project               *
*************************************************/

#include <opencl/opencl.h>

namespace OpenCL {

/*************************************************
* SymmetricAlgorithm                             *
*************************************************/
SymmetricAlgorithm::SymmetricAlgorithm(u32bit min_len, u32bit max_len,
                                       u32bit mod_len) :
   max(max_len ? max_len : min_len), min(min_len), mod(mod_len) {}

/*************************************************
* Query if the keylength is valid                *
*************************************************/
bool SymmetricAlgorithm::valid_keylength(u32bit length) const
   {
   return (length >= min && length <= max && length % mod == 0);
   }

/*************************************************
* BlockCipher                                    *
*************************************************/
BlockCipher::BlockCipher(u32bit block, u32bit min, u32bit max, u32bit mod) :
   SymmetricAlgorithm(min, max, mod),
   BLOCKSIZE(block), KEYLENGTH(max ? max : min) {}

/*************************************************
* StreamCipher                                   *
*************************************************/
StreamCipher::StreamCipher(u32bit min, u32bit max, u32bit mod) :
   SymmetricAlgorithm(min, max, mod), KEYLENGTH(max) {}

/*************************************************
* Encrypt/decrypt with a StreamCipher            *
*************************************************/
void StreamCipher::encrypt(const byte in[], byte out[], u32bit len)
   { cipher(in, out, len); }

void StreamCipher::encrypt(byte in[], u32bit len)
   { cipher(in, in, len); }

void StreamCipher::decrypt(const byte in[], byte out[], u32bit len)
   { cipher(in, out, len); }

void StreamCipher::decrypt(byte in[], u32bit len)
   { cipher(in, in, len); }

/*************************************************
* RandomAccessStreamCipher                       *
*************************************************/
RandomAccessStreamCipher::RandomAccessStreamCipher(u32bit min, u32bit max,
                                                   u32bit mod) :
   StreamCipher(min, max, mod) {}

/*************************************************
* Construct a MAC                                *
*************************************************/
MessageAuthCode::MessageAuthCode(u32bit maclen, u32bit min,
                                 u32bit max, u32bit mod) :
   SymmetricAlgorithm(min, max, mod),
   MACLENGTH(maclen), KEYLENGTH(max ? max : min) {}

/*************************************************
* Hash/MAC a complete message                    *
*************************************************/
void HashFunction::process(const byte in[], u32bit len, byte hash[])
   { update(in, len); final(hash); }

void MessageAuthCode::process(const byte in[], u32bit len, byte mac[])
   { update(in, len); final(mac); }

/*************************************************
* Randomize an array of bytes                    *
*************************************************/
void RandomNumberGenerator::randomize(byte out[], u32bit len)
   {
   for(u32bit j = 0; j != len; j++)
      out[j] ^= random();
   }

/*************************************************
* Constructor for InvalidKeyLength               *
*************************************************/
InvalidKeyLength::InvalidKeyLength(const std::string& name, u32bit length)
   {
   if(length == 0)
      set_msg(name + " cannot accept a key of zero length");
   else
      {
      std::string lenstr;
      while(length > 0)
         {
         lenstr = (char)('0' + length % 10) + lenstr;
         length /= 10;
         set_msg(name + " cannot accept a key of length " + lenstr);
         }
      }
   }

}
