/*************************************************
* Randpool Source File                           *
* (C) 1999-2001 The OpenCL Project               *
*************************************************/

#include <opencl/randpool.h>

namespace OpenCL {

/*************************************************
* Generate a random byte                         *
*************************************************/
byte Randpool::random()
   {
   if(position == BUFFERSIZE)
      generate(system_clock());
   return out[position++];
   }

/*************************************************
* Refill the buffer                              *
*************************************************/
void Randpool::generate(u64bit input)
   {
   RANDPOOL_HASH hash;
   hash.update((byte*)&input, 8);
   hash.update(pool, POOLSIZE);
   hash.final(out);
   xor_buf(pool, out, BUFFERSIZE);
   mix_pool();
   position = 0;
   }

/*************************************************
* Mix up the pool                                *
*************************************************/
void Randpool::mix_pool()
   {
   RANDPOOL_HASH hash;
   for(u32bit j = 0; j != POOLSIZE; j += BUFFERSIZE)
      {
      xor_buf(pool + (j + BUFFERSIZE) % POOLSIZE, pool + j, BUFFERSIZE);
      hash.process(pool + j, std::min(3 * BUFFERSIZE, POOLSIZE - j), pool + j);
      }
   }

/*************************************************
* Add entropy to internal state                  *
*************************************************/
void Randpool::add_entropy(const byte data[], u32bit length) throw()
   {
   while(length)
      {
      u32bit added = std::min(pool.size(), length);
      xor_buf(pool, data, added);
      generate(system_clock());
      length -= added;
      data += added;
      }
   generate(system_time());
   }

/*************************************************
* Add entropy to internal state                  *
*************************************************/
void Randpool::add_entropy(EntropySource& source, bool slowpoll)
   {
   SecureVector<byte> entropy(slowpoll ? (POOLSIZE) : (POOLSIZE / 8));
   u32bit returned = source.get_entropy(entropy, entropy.size());
   add_entropy(entropy, returned);
   }

/*************************************************
* Clear memory of sensitive data                 *
*************************************************/
void Randpool::clear() throw()
   {
   pool.clear();
   out.clear();
   position = 0;
   }

}
