/*************************************************
* SAFER-SK128 Source File                        *
* (C) 1999-2001 The OpenCL Project               *
*************************************************/

#include <opencl/safer_sk.h>

namespace OpenCL {

/*************************************************
* SAFER-SK128 Encryption                         *
*************************************************/
void SAFER_SK128::encrypt(const byte in[BLOCKSIZE], byte out[BLOCKSIZE]) const
   {
   byte A = in[0], B = in[1], C = in[2], D = in[3],
        E = in[4], F = in[5], G = in[6], H = in[7];
   for(u32bit j = 0; j != 16*ROUNDS; j += 16)
      {
      A = EXP[A ^ EK[j  ]]; B = LOG[B + EK[j+1]];
      C = LOG[C + EK[j+2]]; D = EXP[D ^ EK[j+3]];
      E = EXP[E ^ EK[j+4]]; F = LOG[F + EK[j+5]];
      G = LOG[G + EK[j+6]]; H = EXP[H ^ EK[j+7]];
      A += EK[j+ 8]; B ^= EK[j+ 9]; C ^= EK[j+10]; D += EK[j+11];
      E += EK[j+12]; F ^= EK[j+13]; G ^= EK[j+14]; H += EK[j+15];
      PHT(A, B); PHT(C, D); PHT(E, F); PHT(G, H); PHT(A, C); PHT(E, G);
      PHT(B, D); PHT(F, H); PHT(A, E); PHT(B, F); PHT(C, G); PHT(D, H);
      byte T = B; B = E; E = C; C = T; T = D; D = F; F = G; G = T;
      }
   A ^= EK[16*ROUNDS+0]; B += EK[16*ROUNDS+1]; C += EK[16*ROUNDS+2];
   D ^= EK[16*ROUNDS+3]; E ^= EK[16*ROUNDS+4]; F += EK[16*ROUNDS+5];
   G += EK[16*ROUNDS+6]; H ^= EK[16*ROUNDS+7];
   out[0] = A; out[1] = B; out[2] = C; out[3] = D;
   out[4] = E; out[5] = F; out[6] = G; out[7] = H;
   }

/*************************************************
* SAFER-SK128 Decryption                         *
*************************************************/
void SAFER_SK128::decrypt(const byte in[BLOCKSIZE], byte out[BLOCKSIZE]) const
   {
   byte A = in[0], B = in[1], C = in[2], D = in[3],
        E = in[4], F = in[5], G = in[6], H = in[7];
   A ^= EK[16*ROUNDS+0]; B -= EK[16*ROUNDS+1]; C -= EK[16*ROUNDS+2];
   D ^= EK[16*ROUNDS+3]; E ^= EK[16*ROUNDS+4]; F -= EK[16*ROUNDS+5];
   G -= EK[16*ROUNDS+6]; H ^= EK[16*ROUNDS+7];
   for(s32bit j = ROUNDS-1; j >= 0; j--)
      {
      byte T = E; E = B; B = C; C = T; T = F; F = D; D = G; G = T;
      IPHT(A, E); IPHT(B, F); IPHT(C, G); IPHT(D, H); IPHT(A, C); IPHT(E, G);
      IPHT(B, D); IPHT(F, H); IPHT(A, B); IPHT(C, D); IPHT(E, F); IPHT(G, H);
      A = LOG[A - EK[16*j+8 ] + 256]; B = EXP[B ^ EK[16*j+9 ]];
      C = EXP[C ^ EK[16*j+10]];       D = LOG[D - EK[16*j+11] + 256];
      E = LOG[E - EK[16*j+12] + 256]; F = EXP[F ^ EK[16*j+13]];
      G = EXP[G ^ EK[16*j+14]];       H = LOG[H - EK[16*j+15] + 256];
      A ^= EK[16*j+0]; B -= EK[16*j+1]; C -= EK[16*j+2]; D ^= EK[16*j+3];
      E ^= EK[16*j+4]; F -= EK[16*j+5]; G -= EK[16*j+6]; H ^= EK[16*j+7];
      }
   out[0] = A; out[1] = B; out[2] = C; out[3] = D;
   out[4] = E; out[5] = F; out[6] = G; out[7] = H;
   }

/*************************************************
* SAFER-SK128 Key Setup                          *
*************************************************/
void SAFER_SK128::set_key(const byte key[], u32bit length)
   throw(InvalidKeyLength)
   {
   if(!valid_keylength(length))
      throw InvalidKeyLength("SAFER-SK128", length);
   SecureBuffer<byte, 9> KA, KB;
   for(u32bit j = 0; j != 8; j++)
      {
      KA[8] ^= KA[j] = rotate_left(key[j], 5);
      KB[8] ^= KB[j] = EK[j] = key[j+8];
      }
   for(u32bit j = 1; j != ROUNDS + 1; j++)
      {
      for(u32bit k = 0; k != 9; k++)
         {
         KA[k] = rotate_left(KA[k], 6);
         KB[k] = rotate_left(KB[k], 6);
         }
      for(u32bit k = 0; k != 8; k++)
         EK[16*j+k-8] = (byte)(KA[(2*j+k-1) % 9] + EXP[EXP[18*j+k+1]]);
      for(u32bit k = 0; k != 8; k++)
         EK[16*j+k] = (byte)(KB[(2*j+k) % 9] + EXP[EXP[18*j+k+10]]);
      }
   }

/*************************************************
* SAFER-SK128 Constructor                        *
*************************************************/
SAFER_SK128::SAFER_SK128(u32bit r) : BlockCipher(BLOCKSIZE, KEYLENGTH),
                                     ROUNDS(r), EK(16*ROUNDS + 8)
   {
   if(ROUNDS > 13 || ROUNDS == 0)
      throw InvalidArgument("SAFER-SK128: Invalid number of rounds");
   }
}
