/*************************************************
* SEAL Source File                               *
* (C) 1999-2001 The OpenCL Project               *
*************************************************/

#include <opencl/seal.h>
#include <opencl/sha1.h>

namespace OpenCL {

/*************************************************
* Generate SEAL Cipher Stream                    *
*************************************************/
void SEAL::cipher(const byte in[], byte out[], u32bit length)
   {
   while(length >= STATESIZE - position)
      {
      xor_buf(out, in, state + position, STATESIZE - position);
      length -= (STATESIZE - position);
      in += (STATESIZE - position);
      out += (STATESIZE - position);
      generate(counter++);
      }
   xor_buf(out, in, state + position, length);
   position += length;
   }

/*************************************************
* Generate the internal state                    *
*************************************************/
void SEAL::generate(u32bit n)
   {
   u32bit A, B, C, D, P, Q;
   SecureBuffer<u32bit, 4> N, X;
   for(u32bit j = 0; j != STATESIZE / 1024; j++)
      {
      A = n ^ R[4*j];                     B = rotate_right(n,  8) ^ R[4*j+1];
      C = rotate_right(n, 16) ^ R[4*j+2]; D = rotate_right(n, 24) ^ R[4*j+3];
      P = A & 0x7FC; B += T[P/4]; A = rotate_right(A, 9);
      P = B & 0x7FC; C += T[P/4]; B = rotate_right(B, 9);
      P = C & 0x7FC; D += T[P/4]; C = rotate_right(C, 9);
      P = D & 0x7FC; A += T[P/4]; D = rotate_right(D, 9);
      P = A & 0x7FC; B += T[P/4]; A = rotate_right(A, 9);
      P = B & 0x7FC; C += T[P/4]; B = rotate_right(B, 9);
      P = C & 0x7FC; D += T[P/4]; C = rotate_right(C, 9);
      P = D & 0x7FC; A += T[P/4]; D = rotate_right(D, 9);
      N[0] = D; N[1] = B; N[2] = A; N[3] = C;
      P = A & 0x7FC; B += T[P/4]; A = rotate_right(A, 9);
      P = B & 0x7FC; C += T[P/4]; B = rotate_right(B, 9);
      P = C & 0x7FC; D += T[P/4]; C = rotate_right(C, 9);
      P = D & 0x7FC; A += T[P/4]; D = rotate_right(D, 9);
      for(u32bit k = 0; k != 64; k++)
         {
         P = A       & 0x7FC; B += T[P/4]; A = rotate_right(A, 9); B ^= A;
         Q = B       & 0x7FC; C ^= T[Q/4]; B = rotate_right(B, 9); C += B;
         P = (P + C) & 0x7FC; D += T[P/4]; C = rotate_right(C, 9); D ^= C;
         Q = (Q + D) & 0x7FC; A ^= T[Q/4]; D = rotate_right(D, 9); A += D;
         P = (P + A) & 0x7FC; B ^= T[P/4]; A = rotate_right(A, 9);
         Q = (Q + B) & 0x7FC; C += T[Q/4]; B = rotate_right(B, 9);
         P = (P + C) & 0x7FC; D ^= T[P/4]; C = rotate_right(C, 9);
         Q = (Q + D) & 0x7FC; A += T[Q/4]; D = rotate_right(D, 9);
         X[0] = B + S[4*k  ]; X[1] = C ^ S[4*k+1];
         X[2] = D + S[4*k+2]; X[3] = A ^ S[4*k+3];
         for(u32bit l = 0; l != 16; l++)
            state[1024*j+16*k+l] = get_byte(l%4, X[l/4]);
         if(k % 2 == 0) { A += N[0]; B += N[1]; C ^= N[0]; D ^= N[1]; }
         else           { A += N[2]; B += N[3]; C ^= N[2]; D ^= N[3]; }
         }
      }
   position = 0;
   }

/*************************************************
* SEAL Key Setup                                 *
*************************************************/
void SEAL::set_key(const byte key[], u32bit length) throw(InvalidKeyLength)
   {
   class Gamma
      {
      public:
         Gamma(const byte key[20])
            {
            last_index = 0xFFFFFFFF;
            for(u32bit j = 0; j != 20; j++)
               K[j/4] = (K[j/4] << 8) + key[j];
            }
         u32bit operator()(u32bit index)
            {
            u32bit new_index = index / 5;
            if(new_index != last_index)
               {
               for(u32bit j = 0; j != 4; j++)
                  M[j] = get_byte(j, new_index);
               sha1.digest = K;
               sha1.hash(M);
               last_index = new_index;
               }
            return sha1.digest[index % 5];
            }
      private:
         SHA1 sha1;
         SecureBuffer<u32bit, 5> K;
         SecureBuffer<byte, 64> M;
         u32bit last_index;
      };
   if(!valid_keylength(length))
      throw InvalidKeyLength("SEAL", length);
   clear();
   Gamma gamma(key);
   for(u32bit j = 0; j != 512; j++)
      T[j] = gamma(j);
   for(u32bit j = 0; j != 256; j++)
      S[j] = gamma(0x1000 + j);
   for(u32bit j = 0; j != STATESIZE / 256; j++)
      R[j] = gamma(0x2000 + j);
   generate(counter++);
   }

/*************************************************
* Seek a new position in the cipher stream       *
*************************************************/
void SEAL::seek(u32bit new_position)
   {
   counter = (new_position / STATESIZE) + START;
   generate(counter++);
   position = new_position % STATESIZE;
   }

/*************************************************
* Clear memory of sensitive data                 *
*************************************************/
void SEAL::clear() throw()
   {
   state.clear();
   T.clear();
   S.clear();
   R.clear();
   position = 0;
   counter = START;
   }

}
