using System;
using System.Text;

namespace Org.BouncyCastle.Asn1
{
    /**
     * Der IA5String object - this is an ascii string.
     */
    public class DerIA5String
        : DerStringBase
    {
        private readonly string str;

        /**
         * return a IA5 string from the passed in object
         *
         * @exception ArgumentException if the object cannot be converted.
         */
        public static DerIA5String GetInstance(
            object obj)
        {
            if (obj == null || obj is DerIA5String)
            {
                return (DerIA5String)obj;
            }

            if (obj is Asn1OctetString)
            {
                return new DerIA5String(((Asn1OctetString)obj).GetOctets());
            }

            if (obj is Asn1TaggedObject)
            {
                return GetInstance(((Asn1TaggedObject)obj).GetObject());
            }

            throw new ArgumentException("illegal object in GetInstance: " + obj.GetType().Name);
        }

        /**
         * return an IA5 string from a tagged object.
         *
         * @param obj the tagged object holding the object we want
         * @param explicitly true if the object is meant to be explicitly
         *              tagged false otherwise.
         * @exception ArgumentException if the tagged object cannot
         *               be converted.
         */
        public static DerIA5String GetInstance(
            Asn1TaggedObject	obj,
            bool				explicitly)
        {
            return GetInstance(obj.GetObject());
        }

        /**
         * basic constructor - with bytes.
         */
        public DerIA5String(
            byte[] str)
			: this(Encoding.ASCII.GetString(str), false)
        {
        }

		/**
		* basic constructor - without validation.
		*/
		public DerIA5String(
			string str)
			: this(str, false)
		{
		}

		/**
		* Constructor with optional validation.
		*
		* @param string the base string to wrap.
		* @param validate whether or not to check the string.
		* @throws IllegalArgumentException if validate is true and the string
		* contains characters that should not be in an IA5String.
		*/
		public DerIA5String(
			string	str,
			bool	validate)
		{
			if (str == null)
				throw new ArgumentNullException("str");
			if (validate && !IsIA5String(str))
				throw new ArgumentException("string contains illegal characters", "str");

			this.str = str;
		}

		public override string GetString()
        {
            return str;
        }

		public byte[] GetOctets()
        {
			return Encoding.ASCII.GetBytes(str);
        }

		internal override void Encode(
            DerOutputStream derOut)
        {
            derOut.WriteEncoded(Asn1Tags.IA5String, this.GetOctets());
        }

		protected override int Asn1GetHashCode()
		{
            return this.str.GetHashCode();
        }

		protected override bool Asn1Equals(
            Asn1Object obj)
        {
			DerIA5String other = obj as DerIA5String;

			if (other == null)
				return false;

			return this.str.Equals(other.str);
        }

		/**
		 * return true if the passed in String can be represented without
		 * loss as an IA5String, false otherwise.
		 *
		 * @return true if in printable set, false otherwise.
		 */
		public static bool IsIA5String(
			string str)
		{
			foreach (char ch in str)
			{
				if (ch > 0x007f)
				{
					return false;
				}
			}

			return true;
		}
	}
}
