using System;

using NUnit.Framework;

using Org.BouncyCastle.Math;
using Org.BouncyCastle.Utilities;

namespace Org.BouncyCastle.Math.Tests
{
	[TestFixture]
	public class BigIntegerTest
	{
		private static Random random = new Random();

		[Test]
		public void TestAbs()
		{
			Assert.AreEqual(zero, zero.Abs());

			Assert.AreEqual(one, one.Abs());
			Assert.AreEqual(one, minusOne.Abs());

			Assert.AreEqual(two, two.Abs());
			Assert.AreEqual(two, minusTwo.Abs());
		}

		[Test]
		public void TestAdd()
		{
			for (int i = -10; i <= 10; ++i)
			{
				for (int j = -10; j <= 10; ++j)
				{
					Assert.AreEqual(
						val(i + j),
						val(i).Add(val(j)),
						"Problem: " + i + ".Add(" + j + ") should be " + (i + j));
				}
			}
		}

		[Test]
		public void TestAnd()
		{
			for (int i = -10; i <= 10; ++i)
			{
				for (int j = -10; j <= 10; ++j)
				{
					Assert.AreEqual(
						val(i & j),
						val(i).And(val(j)),
						"Problem: " + i + " AND " + j + " should be " + (i & j));
				}
			}
		}

		[Test]
		public void TestAndNot()
		{
			for (int i = -10; i <= 10; ++i)
			{
				for (int j = -10; j <= 10; ++j)
				{
					Assert.AreEqual(
						val(i & ~j),
						val(i).AndNot(val(j)),
						"Problem: " + i + " AND NOT " + j + " should be " + (i & ~j));
				}
			}
		}

		[Test]
		public void TestBitCount()
		{
			for (int i = 0; i < 10; ++i)
			{
				BigInteger test = new BigInteger(128, 0, random);
				int bitCount = 0;

				for (int bit = 0; bit < test.BitLength; ++bit)
				{
					if (test.TestBit(bit))
					{
						++bitCount;
					}
				}

				Assert.AreEqual(bitCount, test.BitCount);
			}
		}

		[Test]
		public void TestBitLength()
		{
			for (int i = 0; i < 128; ++i)
			{
				Assert.AreEqual(i + 1, one.ShiftLeft(i).BitLength);
			}
		}

		[Test]
		public void TestClearBit()
		{
			Assert.AreEqual(zero, zero.ClearBit(0));
			Assert.AreEqual(zero, one.ClearBit(0));
			Assert.AreEqual(two, two.ClearBit(0));

			Assert.AreEqual(zero, zero.ClearBit(1));
			Assert.AreEqual(one, one.ClearBit(1));
			Assert.AreEqual(zero, two.ClearBit(1));

			// TODO Tests for clearing bits in negative numbers

			// TODO Tests for clearing extended bits
		}

		[Test]
		public void TestCompareTo()
		{
			Assert.AreEqual(0, minusTwo.CompareTo(minusTwo));
			Assert.AreEqual(-1, minusTwo.CompareTo(minusOne));
			Assert.AreEqual(-1, minusTwo.CompareTo(zero));
			Assert.AreEqual(-1, minusTwo.CompareTo(one));
			Assert.AreEqual(-1, minusTwo.CompareTo(two));

			Assert.AreEqual(1, minusOne.CompareTo(minusTwo));
			Assert.AreEqual(0, minusOne.CompareTo(minusOne));
			Assert.AreEqual(-1, minusOne.CompareTo(zero));
			Assert.AreEqual(-1, minusOne.CompareTo(one));
			Assert.AreEqual(-1, minusOne.CompareTo(two));

			Assert.AreEqual(1, zero.CompareTo(minusTwo));
			Assert.AreEqual(1, zero.CompareTo(minusOne));
			Assert.AreEqual(0, zero.CompareTo(zero));
			Assert.AreEqual(-1, zero.CompareTo(one));
			Assert.AreEqual(-1, zero.CompareTo(two));

			Assert.AreEqual(1, one.CompareTo(minusTwo));
			Assert.AreEqual(1, one.CompareTo(minusOne));
			Assert.AreEqual(1, one.CompareTo(zero));
			Assert.AreEqual(0, one.CompareTo(one));
			Assert.AreEqual(-1, one.CompareTo(two));

			Assert.AreEqual(1, two.CompareTo(minusTwo));
			Assert.AreEqual(1, two.CompareTo(minusOne));
			Assert.AreEqual(1, two.CompareTo(zero));
			Assert.AreEqual(1, two.CompareTo(one));
			Assert.AreEqual(0, two.CompareTo(two));
		}

		[Test]
		public void TestConstructors()
		{
			Assert.AreEqual(BigInteger.Zero, new BigInteger(new byte[]{ 0 }));
			Assert.AreEqual(BigInteger.Zero, new BigInteger(new byte[]{ 0, 0 }));

			// TODO Other constructors
		}

		[Test]
		public void TestDivide()
		{
			for (int i = -5; i <= 5; ++i)
			{
				try
				{
					val(i).Divide(zero);
					Assert.Fail("expected ArithmeticException");
				}
				catch (ArithmeticException) {}
			}

			int product = 1 * 2 * 3 * 4 * 5 * 6 * 7 * 8 * 9;
			int productPlus = product + 1;

			BigInteger bigProduct = val(product);
			BigInteger bigProductPlus = val(productPlus);

			for (int divisor = 1; divisor < 10; ++divisor)
			{
				// Exact division
				BigInteger expected = val(product / divisor);

				Assert.AreEqual(expected, bigProduct.Divide(val(divisor)));
				Assert.AreEqual(expected.Negate(), bigProduct.Negate().Divide(val(divisor)));
				Assert.AreEqual(expected.Negate(), bigProduct.Divide(val(divisor).Negate()));
				Assert.AreEqual(expected, bigProduct.Negate().Divide(val(divisor).Negate()));

				expected = val((product + 1)/divisor);

				Assert.AreEqual(expected, bigProductPlus.Divide(val(divisor)));
				Assert.AreEqual(expected.Negate(), bigProductPlus.Negate().Divide(val(divisor)));
				Assert.AreEqual(expected.Negate(), bigProductPlus.Divide(val(divisor).Negate()));
				Assert.AreEqual(expected, bigProductPlus.Negate().Divide(val(divisor).Negate()));
			}

			for (int rep = 0; rep < 10; ++rep)
			{
				BigInteger a = new BigInteger(100 - rep, 0, random);
				BigInteger b = new BigInteger(100 + rep, 0, random);
				BigInteger c = new BigInteger(10 + rep, 0, random);
				BigInteger d = a.Multiply(b).Add(c);
				BigInteger e = d.Divide(a);

				Assert.AreEqual(b, e);
			}
		}

		[Test]
		public void TestDivideAndRemainder()
		{
			// TODO Basic tests

			for (int rep = 0; rep < 10; ++rep)
			{
				BigInteger a = new BigInteger(100 - rep, 0, random);
				BigInteger b = new BigInteger(100 + rep, 0, random);
				BigInteger c = new BigInteger(10 + rep, 0, random);
				BigInteger d = a.Multiply(b).Add(c);
				BigInteger[] es = d.DivideAndRemainder(a);

				Assert.AreEqual(b, es[0]);
				Assert.AreEqual(c, es[1]);
			}
		}

		[Test]
		public void TestFlipBit()
		{
			for (int i = 0; i < 10; ++i)
			{
				BigInteger a = new BigInteger(128, 0, random);
				BigInteger b = a;

				for (int x = 0; x < 100; ++x)
				{
					// Note: Intentionally greater than initial size
					int pos = random.Next(256);

					a = a.FlipBit(pos);
					b = b.TestBit(pos) ? b.ClearBit(pos) : b.SetBit(pos);
				}

				Assert.AreEqual(a, b);
			}
		}

		[Test]
		public void TestGcd()
		{
			// TODO
			Assert.Ignore();
		}

		[Test]
		public void TestGetLowestSetBit()
		{
			for (int i = 0; i < 10; ++i)
			{
				BigInteger test = new BigInteger(128, 0, random).Add(one);
				int bit1 = test.GetLowestSetBit();
				Assert.AreEqual(test, test.ShiftRight(bit1).ShiftLeft(bit1));
				int bit2 = test.ShiftLeft(i + 1).GetLowestSetBit();
				Assert.AreEqual(i + 1, bit2 - bit1);
			}
		}

		[Test]
		public void TestIntValue()
		{
			int[] tests = new int[]{ int.MinValue, -1234, -10, -1, 0, ~0, 1, 10, 5678, int.MaxValue };

			foreach (int test in tests)
			{
				Assert.AreEqual(test, val(test).IntValue);
			}

			// TODO Tests for large numbers
		}

		[Test]
		public void TestIsProbablePrime()
		{
			Assert.IsFalse(zero.IsProbablePrime(100));
			Assert.IsFalse(zero.IsProbablePrime(100));
			Assert.IsTrue(zero.IsProbablePrime(0));
			Assert.IsTrue(zero.IsProbablePrime(-10));
			Assert.IsFalse(minusOne.IsProbablePrime(100));
			Assert.IsTrue(minusTwo.IsProbablePrime(100));
			Assert.IsTrue(val(-17).IsProbablePrime(100));
			Assert.IsTrue(val(67).IsProbablePrime(100));
			Assert.IsTrue(val(773).IsProbablePrime(100));

			foreach (int p in firstPrimes)
			{
				Assert.IsTrue(val(p).IsProbablePrime(100));
				Assert.IsTrue(val(-p).IsProbablePrime(100));
			}

			foreach (int c in nonPrimes)
			{
				Assert.IsFalse(val(c).IsProbablePrime(100));
				Assert.IsFalse(val(-c).IsProbablePrime(100));
			}

			foreach (int e in mersennePrimeExponents)
			{
				Assert.IsTrue(mersenne(e).IsProbablePrime(100));
				Assert.IsTrue(mersenne(e).Negate().IsProbablePrime(100));
			}

			foreach (int e in nonPrimeExponents)
			{
				Assert.IsFalse(mersenne(e).IsProbablePrime(100));
				Assert.IsFalse(mersenne(e).Negate().IsProbablePrime(100));
			}

			// TODO Other examples of 'tricky' values?
		}

		[Test]
		public void TestLongValue()
		{
			long[] tests = new long[]{ long.MinValue, -1234, -10, -1, 0L, ~0L, 1, 10, 5678, long.MaxValue };

			foreach (long test in tests)
			{
				Assert.AreEqual(test, val(test).LongValue);
			}

			// TODO Tests for large numbers
		}

		[Test]
		public void TestMax()
		{
			for (int i = -10; i <= 10; ++i)
			{
				for (int j = -10; j <= 10; ++j)
				{
					Assert.AreEqual(val(System.Math.Max(i, j)), val(i).Max(val(j)));
				}
			}
		}

		[Test]
		public void TestMin()
		{
			for (int i = -10; i <= 10; ++i)
			{
				for (int j = -10; j <= 10; ++j)
				{
					Assert.AreEqual(val(System.Math.Min(i, j)), val(i).Min(val(j)));
				}
			}
		}

		[Test]
		public void TestMod()
		{
			// TODO Basic tests

			for (int rep = 0; rep < 10; ++rep)
			{
				BigInteger a = new BigInteger(100 - rep, 0, random);
				BigInteger b = new BigInteger(100 + rep, 0, random);
				BigInteger c = new BigInteger(10 + rep, 0, random);
				BigInteger d = a.Multiply(b).Add(c);
				BigInteger e = d.Mod(a);

				Assert.AreEqual(c, e);
			}
		}

		[Test]
		public void TestModInverse()
		{
			// TODO
			Assert.Ignore();
		}

		[Test]
		public void TestModPow()
		{
			try
			{
				two.ModPow(one, zero);
				Assert.Fail("expected ArithmeticException");
			}
			catch (ArithmeticException) {}

			Assert.AreEqual(zero, two.ModPow(one, one));
			Assert.AreEqual(one, two.ModPow(zero, two));
		}

		[Test]
		public void TestMultiply()
		{
			// TODO
			Assert.Ignore();
		}

		[Test]
		public void TestNegate()
		{
			for (int i = -10; i <= 10; ++i)
			{
				Assert.AreEqual(val(-i), val(i).Negate());
			}
		}

		[Test]
		public void TestNextProbablePrime()
		{
			BigInteger firstPrime = BigInteger.ProbablePrime(32, random);
			BigInteger nextPrime = firstPrime.NextProbablePrime();

			Assert.IsTrue(firstPrime.IsProbablePrime(10));
			Assert.IsTrue(nextPrime.IsProbablePrime(10));

			BigInteger check = firstPrime.Add(one);

			while (check.CompareTo(nextPrime) < 0)
			{
				Assert.IsFalse(check.IsProbablePrime(10));
				check = check.Add(one);
			}
		}

		[Test]
		public void TestNot()
		{
			for (int i = -10; i <= 10; ++i)
			{
				Assert.AreEqual(
					val(~i),
					val(i).Not(),
					"Problem: ~" + i + " should be " + ~i);
			}
		}

		[Test]
		public void TestOr()
		{
			for (int i = -10; i <= 10; ++i)
			{
				for (int j = -10; j <= 10; ++j)
				{
					Assert.AreEqual(
						val(i | j),
						val(i).Or(val(j)),
						"Problem: " + i + " OR " + j + " should be " + (i | j));
				}
			}
		}

		[Test]
		public void TestPow()
		{
			Assert.AreEqual(one, zero.Pow(0));
			Assert.AreEqual(zero, zero.Pow(123));
			Assert.AreEqual(one, one.Pow(0));
			Assert.AreEqual(one, one.Pow(123));

			BigInteger n = new BigInteger("1234567890987654321");
			BigInteger result = one;

			for (int i = 0; i < 10; ++i)
			{
				try
				{
					val(i).Pow(-1);
					Assert.Fail("expected ArithmeticException");
				}
				catch (ArithmeticException) {}

				Assert.AreEqual(result, n.Pow(i));

				result = result.Multiply(n);
			}
		}

		[Test]
		public void TestRemainder()
		{
			// TODO
			Assert.Ignore();
		}

		[Test]
		public void TestSetBit()
		{
			// TODO
			Assert.Ignore();
		}

		[Test]
		public void TestShiftLeft()
		{
			// TODO
			Assert.Ignore();
		}

		[Test]
		public void TestShiftRight()
		{
			// TODO
			Assert.Ignore();
		}

		[Test]
		public void TestSignValue()
		{
			for (int i = -10; i <= 10; ++i)
			{
				Assert.AreEqual(i < 0 ? -1 : i > 0 ? 1 : 0, val(i).SignValue);
			}
		}

		[Test]
		public void TestSubtract()
		{
			for (int i = -10; i <= 10; ++i)
			{
				for (int j = -10; j <= 10; ++j)
				{
					Assert.AreEqual(
						val(i - j),
						val(i).Subtract(val(j)),
						"Problem: " + i + ".Subtract(" + j + ") should be " + (i - j));
				}
			}
		}

		[Test]
		public void TestTestBit()
		{
			// TODO
			Assert.Ignore();
		}

		[Test]
		public void TestToByteArray()
		{
			byte[] z = BigInteger.Zero.ToByteArray();
			Assert.IsTrue(Arrays.AreEqual(new byte[1], z));

			for (int i = 16; i <= 48; ++i)
			{
				BigInteger x = BigInteger.ProbablePrime(i, random);
				byte[] b = x.ToByteArray();
				Assert.AreEqual((i / 8 + 1), b.Length);
				BigInteger y = new BigInteger(b);
				Assert.AreEqual(x, y);
			}
		}

		[Test]
		public void TestToByteArrayUnsigned()
		{
			byte[] z = BigInteger.Zero.ToByteArrayUnsigned();
			Assert.IsTrue(Arrays.AreEqual(new byte[0], z));

			for (int i = 16; i <= 48; ++i)
			{
				BigInteger x = BigInteger.ProbablePrime(i, random);
				byte[] b = x.ToByteArrayUnsigned();
				Assert.AreEqual((i + 7) / 8, b.Length);
				BigInteger y = new BigInteger(1, b);
				Assert.AreEqual(x, y);
				y = new BigInteger(1, b);
				Assert.AreEqual(x.ToString(2), y.ToString(2));
			}
		}

		[Test]
		public void TestToString()
		{
			string s = "12345667890987654321";

			Assert.AreEqual(s, new BigInteger(s).ToString());
			Assert.AreEqual(s, new BigInteger(s, 10).ToString(10));
			Assert.AreEqual(s, new BigInteger(s, 16).ToString(16));
		}

		[Test]
		public void TestValueOf()
		{
			Assert.AreEqual(-1, BigInteger.ValueOf(-1).SignValue);
			Assert.AreEqual(0, BigInteger.ValueOf(0).SignValue);
			Assert.AreEqual(1, BigInteger.ValueOf(1).SignValue);

			for (long i = -5; i < 5; ++i)
			{
				Assert.AreEqual(i, BigInteger.ValueOf(i).IntValue);
			}
		}

		[Test]
		public void TestXor()
		{
			for (int i = -10; i <= 10; ++i)
			{
				for (int j = -10; j <= 10; ++j)
				{
					Assert.AreEqual(
						val(i ^ j),
						val(i).Xor(val(j)),
						"Problem: " + i + " XOR " + j + " should be " + (i ^ j));
				}
			}
		}

		private static BigInteger val(long n)
		{
			return BigInteger.ValueOf(n);
		}

		private static BigInteger mersenne(int e)
		{
			return two.Pow(e).Subtract(one);
		}

		private static readonly BigInteger minusTwo = BigInteger.Two.Negate();
		private static readonly BigInteger minusOne = BigInteger.One.Negate();
		private static readonly BigInteger zero = BigInteger.Zero;
		private static readonly BigInteger one = BigInteger.One;
		private static readonly BigInteger two = BigInteger.Two;

		private static int[] firstPrimes = { 2, 3, 5, 7, 11, 13, 17, 19, 23, 29 };
		private static int[] nonPrimes = { 0, 1, 4, 10, 20, 21, 22, 25, 26, 27 };

		private static int[] mersennePrimeExponents = { 2, 3, 5, 7, 13, 17, 19, 31, 61, 89 };
		private static int[] nonPrimeExponents = { 1, 4, 6, 9, 11, 15, 23, 29, 37, 41 };
	}
}
