using System;
using System.Collections;
using System.IO;

using Org.BouncyCastle.Bcpg.OpenPgp;

namespace Org.BouncyCastle.Bcpg.OpenPgp.Examples
{
    /**
    * A simple utility class that creates clear signed files and verifies them.
    * <p>
    * To sign a file: ClearSignedFileProcessor -s fileName secretKey passPhrase.<br>
    * If -a is specified the output file will be "ascii-armored".
    * <p>
    * To decrypt: ClearSignedFileProcessor -v fileName signatureFile publicKeyFile.
    * <p>
    * Note: This example does not dash escape the input on signing or look for dash escaping on verification. See section 7 of RFC 2440 for further details.
    */
    public sealed class ClearSignedFileProcessor
    {
        private ClearSignedFileProcessor()
        {
        }

        /**
        * A simple routine that opens a key ring file and loads the first available key suitable for
        * signature generation.
        *
        * @param in
        * @return
        * @m_out
        * @
        */
        private static PgpSecretKey ReadSecretKey(
            Stream inputStream)
        {
            PgpSecretKeyRingBundle        pgpSec = new PgpSecretKeyRingBundle(inputStream);

            //
            // we just loop through the collection till we find a key suitable for encryption, in the real
            // world you would probably want to be a bit smarter about this.
            //

            //
            // iterate through the key rings.
            //
            foreach (PgpSecretKeyRing kRing in pgpSec.GetKeyRings())
            {
                foreach (PgpSecretKey k in kRing.GetSecretKeys())
                {
                    if (k.IsSigningKey)
                    {
                        return k;
                    }
                }
            }

            throw new ArgumentException("Can't find signing key in key ring.");
        }

        /**
        * verify a SHA1 clear text signed file
        */
        private static void VerifyFile(
            Stream        inputStream,
            Stream        keyIn)

        {
            ArmoredInputStream    aIn = new ArmoredInputStream(inputStream);

            //
            // read the input, making sure we ingore the last newline.
            //
            int ch, lastCh;
            bool newLine = false;
            MemoryStream bOut = new MemoryStream();

			lastCh = 0;

			while ((ch = aIn.ReadByte()) >= 0 && aIn.IsClearText())
            {
				if (lastCh == '\r' && ch == '\n')
				{
					continue;
				}

				if (newLine)
                {
                    bOut.WriteByte((byte)lastCh);
                    newLine = false;
                }

				if (ch == '\r' || ch == '\n')
                {
					lastCh = ch;
                    newLine = true;
                    continue;
                }

                bOut.WriteByte((byte)ch);
				lastCh = ch;
            }

			PgpPublicKeyRingBundle pgpRings = new PgpPublicKeyRingBundle(keyIn);

            PgpObjectFactory	pgpFact = new PgpObjectFactory(aIn);
            PgpSignatureList	p3 = (PgpSignatureList) pgpFact.NextPgpObject();
            PgpSignature		sig = p3[0];

            sig.InitVerify(pgpRings.GetPublicKey(sig.KeyId));

            byte[] data = bOut.ToArray();

            sig.Update(data, 0, data.Length);

            if (sig.Verify())
            {
                Console.WriteLine("signature verified.");
            }
            else
            {
                Console.WriteLine("signature verification failed.");
            }
        }

        /**
        * create a clear text signed file.
        */
        private static void SignFile(
            string	fileName,
            Stream	keyIn,
            Stream	outputStream,
            char[]	pass,
			string	digestName)
        {
			HashAlgorithmTag digest;

			if (digestName.Equals("SHA256"))
			{
				digest = HashAlgorithmTag.Sha256;
			}
			else if (digestName.Equals("SHA384"))
			{
				digest = HashAlgorithmTag.Sha384;
			}
			else if (digestName.Equals("SHA512"))
			{
				digest = HashAlgorithmTag.Sha512;
			}
			else if (digestName.Equals("MD5"))
			{
				digest = HashAlgorithmTag.MD5;
			}
			else if (digestName.Equals("RIPEMD160"))
			{
				digest = HashAlgorithmTag.RipeMD160;
			}
			else
			{
				digest = HashAlgorithmTag.Sha1;
			}

			PgpSecretKey                    pgpSecKey = ReadSecretKey(keyIn);
            PgpPrivateKey                   pgpPrivKey = pgpSecKey.ExtractPrivateKey(pass);
            PgpSignatureGenerator           sGen = new PgpSignatureGenerator(pgpSecKey.PublicKey.Algorithm, digest);
            PgpSignatureSubpacketGenerator  spGen = new PgpSignatureSubpacketGenerator();

			sGen.InitSign(PgpSignature.CanonicalTextDocument, pgpPrivKey);

			IEnumerator enumerator = pgpSecKey.PublicKey.GetUserIds().GetEnumerator();
            if (enumerator.MoveNext())
            {
                spGen.SetSignerUserId(false, (string) enumerator.Current);
                sGen.SetHashedSubpackets(spGen.Generate());
            }

			// TODO Close this at some point
            FileStream fIn = File.OpenRead(fileName);
            int ch = 0;
			int lastCh = 0;

			ArmoredOutputStream aOut = new ArmoredOutputStream(outputStream);

			aOut.BeginClearText(digest);

			bool newLine = false;

			//
            // note the last \n in the file is ignored
            //
            while ((ch = fIn.ReadByte()) >= 0)
            {
                aOut.WriteByte((byte)ch);

				if (lastCh == '\r' && ch == '\n')
				{
					continue;
				}

				if (newLine)
                {
                    sGen.Update((byte)lastCh);
                    newLine = false;
                }

				if (ch == '\r' || ch == '\n')
				{
					lastCh = ch;
                    newLine = true;
                    continue;
                }

				sGen.Update((byte)ch);
				lastCh = ch;
			}

            aOut.EndClearText();

			BcpgOutputStream bOut = new BcpgOutputStream(aOut);

            sGen.Generate().Encode(bOut);

            aOut.Close();
        }

		public static void Main(
            string[] args)
        {
            if (args[0].Equals("-s"))
            {
				FileStream fis = File.OpenRead(args[2]);
				FileStream fos = File.Create(args[1] + ".asc");

				Stream keyIn = PgpUtilities.GetDecoderStream(fis);

				if (args.Length == 4)
				{
					SignFile(args[1], keyIn, fos, args[3].ToCharArray(), "SHA1");
				}
				else
				{
					SignFile(args[1], keyIn, fos, args[3].ToCharArray(), args[4]);
				}

				fis.Close();
				fos.Close();
            }
            else if (args[0].Equals("-v"))
            {
                FileStream fin = File.OpenRead(args[1]);
				FileStream fis = File.OpenRead(args[2]);
                Stream keyIn = PgpUtilities.GetDecoderStream(fis);

                VerifyFile(fin, keyIn);

				fin.Close();
				fis.Close();
            }
            else
            {
                Console.Error.WriteLine("usage: ClearSignedFileProcessor [-s file keyfile passPhrase]|[-v sigFile keyFile]");
            }
        }
    }
}
