using System;
using System.IO;

using NUnit.Core;
using NUnit.Framework;

namespace Org.BouncyCastle.Bcpg.OpenPgp.Examples.Tests
{
	[TestFixture]
	public class AllTests
	{
		private TextWriter _oldOut;
		private TextWriter _oldErr;

		private MemoryStream _currentOut;
		private MemoryStream _currentErr;

		[SetUp]
		public void SetUp()
		{
			_oldOut = Console.Out;
			_oldErr = Console.Error;
			_currentOut = new MemoryStream();
			_currentErr = new MemoryStream();

			Console.SetOut(new StreamWriter(_currentOut));
			Console.SetError(new StreamWriter(_currentErr));
		}

		[TearDown]
		public void TearDown()
		{
			Console.SetOut(_oldOut);
			Console.SetError(_oldErr);
		}

		[Test]
		public void TestRsaKeyGeneration() 
		{
			RsaKeyRingGenerator.Main(new string[] { "test", "password" });

			CreateSmallTestInput();
			CreateLargeTestInput();

			CheckSigning("bpg");
			CheckKeyBasedEncryption("bpg");
			CheckLargeKeyBasedEncryption("bpg");

			RsaKeyRingGenerator.Main(new string[] { "-a", "test", "password" });

			CheckSigning("asc");
			CheckKeyBasedEncryption("asc");
			CheckLargeKeyBasedEncryption("asc");
		}

		[Test]
		public void TestDsaElGamalKeyGeneration() 
		{
			DsaElGamalKeyRingGenerator.Main(new string[] { "test", "password" });

			CreateSmallTestInput();
			CreateLargeTestInput();

			CheckSigning("bpg");
			CheckKeyBasedEncryption("bpg");
			CheckLargeKeyBasedEncryption("bpg");

			DsaElGamalKeyRingGenerator.Main(new string[] { "-a", "test", "password" });

			CheckSigning("asc");
			CheckKeyBasedEncryption("asc");
			CheckLargeKeyBasedEncryption("asc");
		}

		[Test]
		public void TestPbeEncryption() 
		{
			Console.Error.Flush();
			_currentErr.SetLength(0);

			PbeFileProcessor.Main(new string[] { "-e", "test.txt", "password" });

			PbeFileProcessor.Main(new string[] { "-d", "test.txt.bpg", "password" });

			Console.Error.Flush();
			Assert.AreEqual("no message integrity check", GetLine(_currentErr));

			PbeFileProcessor.Main(new string[] { "-e", "-i", "test.txt", "password" });

			PbeFileProcessor.Main(new string[] { "-d", "test.txt.bpg", "password" });

			Console.Error.Flush();
			Assert.AreEqual("message integrity check passed", GetLine(_currentErr));

			PbeFileProcessor.Main(new string[] { "-e", "-ai", "test.txt", "password" });

			PbeFileProcessor.Main(new string[] { "-d", "test.txt.asc", "password" });

			Console.Error.Flush();
			Assert.AreEqual("message integrity check passed", GetLine(_currentErr));
		}

		private void CheckSigning(
			string type) 
		{
			Console.Out.Flush();
			_currentOut.SetLength(0);

			SignedFileProcessor.Main(new string[] { "-s", "test.txt", "secret." + type, "password" });

			SignedFileProcessor.Main(new string[] { "-v", "test.txt.bpg", "pub." + type });

			Console.Out.Flush();
			Assert.AreEqual("signature verified.", GetLine(_currentOut));

			SignedFileProcessor.Main(new string[] { "-s", "-a", "test.txt", "secret." + type, "password" });

			SignedFileProcessor.Main(new string[] { "-v", "test.txt.asc", "pub." + type });

			Console.Out.Flush();
			Assert.AreEqual("signature verified.", GetLine(_currentOut));
		}

		private void CheckKeyBasedEncryption(
			string type) 
		{
			Console.Error.Flush();
			_currentErr.SetLength(0);

			KeyBasedFileProcessor.Main(new string[] { "-e", "test.txt", "pub." + type });

			KeyBasedFileProcessor.Main(new string[] { "-d", "test.txt.bpg", "secret." + type, "password" });

			Console.Error.Flush();
			Assert.AreEqual("no message integrity check", GetLine(_currentErr));

			KeyBasedFileProcessor.Main(new string[] { "-e", "-i", "test.txt", "pub." + type });

			KeyBasedFileProcessor.Main(new string[] { "-d", "test.txt.bpg", "secret." + type, "password" });

			Console.Error.Flush();
			Assert.AreEqual("message integrity check passed", GetLine(_currentErr));

			KeyBasedFileProcessor.Main(new string[] { "-e", "-ai", "test.txt", "pub." + type });

			KeyBasedFileProcessor.Main(new string[] { "-d", "test.txt.asc", "secret." + type, "password" });

			Console.Error.Flush();
			Assert.AreEqual("message integrity check passed", GetLine(_currentErr));
		}

		private void CheckLargeKeyBasedEncryption(
			string type) 
		{
			Console.Error.Flush();
			_currentErr.SetLength(0);

			KeyBasedLargeFileProcessor.Main(new string[] { "-e", "large.txt", "pub." + type });

			KeyBasedLargeFileProcessor.Main(new string[] { "-d", "large.txt.bpg", "secret." + type, "password" });

			Console.Error.Flush();
			Assert.AreEqual("no message integrity check", GetLine(_currentErr));

			KeyBasedLargeFileProcessor.Main(new string[] { "-e", "-i", "large.txt", "pub." + type });

			KeyBasedLargeFileProcessor.Main(new string[] { "-d", "large.txt.bpg", "secret." + type, "password" });

			Console.Error.Flush();
			Assert.AreEqual("message integrity check passed", GetLine(_currentErr));

			KeyBasedLargeFileProcessor.Main(new string[] { "-e", "-ai", "large.txt", "pub." + type });

			KeyBasedLargeFileProcessor.Main(new string[] { "-d", "large.txt.asc", "secret." + type, "password" });

			Console.Error.Flush();
			Assert.AreEqual("message integrity check passed", GetLine(_currentErr));
		}

		private void CreateSmallTestInput() 
		{
			FileStream fs = File.Create("test.txt");
			TextWriter bfOut = new StreamWriter(fs);

			bfOut.WriteLine("hello world!");

			bfOut.Close();
			fs.Close();
		}

		private void CreateLargeTestInput() 
		{
			FileStream fs = File.Create("large.txt");
			TextWriter bfOut = new StreamWriter(fs);

			for (int i = 0; i != 2000; i++)
			{
				bfOut.WriteLine("hello world!");
			}

			bfOut.Close();
			fs.Close();
		}

		private string GetLine(
			MemoryStream outStr) 
		{
			byte[] b = outStr.ToArray();
			TextReader bRd = new StreamReader(new BufferedStream(new MemoryStream(b, false)));

			outStr.SetLength(0);

			return bRd.ReadLine();
		}

		public static void Main(
			string[] args)
		{
			//junit.textui.TestRunner.run(suite());
			EventListener el = new NullListener();
			suite().Run(el);
		}

		public static TestSuite suite()
		{
			TestSuite suite = new TestSuite("OpenPGP Example Tests");

			suite.Add(new AllTests());

			return suite;
		}

	}
}
