using System;
using System.IO;
using System.Text;

using NUnit.Framework;

using Org.BouncyCastle.Asn1.Nist;
using Org.BouncyCastle.Crypto;
using Org.BouncyCastle.Crypto.IO;
using Org.BouncyCastle.Crypto.Parameters;
using Org.BouncyCastle.Security;
using Org.BouncyCastle.Utilities.Encoders;
using Org.BouncyCastle.Utilities.Test;

namespace Org.BouncyCastle.Tests
{
	/// <remarks>Basic test class for the AES cipher vectors from FIPS-197</remarks>
	[TestFixture]
	public class AesTest
		: SimpleTest
	{
		internal static readonly string[] cipherTests =
		{
			"128",
			"000102030405060708090a0b0c0d0e0f",
			"00112233445566778899aabbccddeeff",
			"69c4e0d86a7b0430d8cdb78070b4c55a",
			"192",
			"000102030405060708090a0b0c0d0e0f1011121314151617",
			"00112233445566778899aabbccddeeff",
			"dda97ca4864cdfe06eaf70a0ec0d7191",
			"256",
			"000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f",
			"00112233445566778899aabbccddeeff",
			"8ea2b7ca516745bfeafc49904b496089",
		};

		public override string Name
		{
			get { return "AES"; }
		}

		[Test]
		public void TestCiphers()
		{
			for (int i = 0; i != cipherTests.Length; i += 4)
			{
				doCipherTest(int.Parse(cipherTests[i]),
					Hex.Decode(cipherTests[i + 1]),
					Hex.Decode(cipherTests[i + 2]),
					Hex.Decode(cipherTests[i + 3]));
			}
		}

		[Test]
		public void TestOids()
		{
			doOidTest();
		}

		[Test]
		public void TestWrap()
		{
			byte[] kek1 = Hex.Decode("000102030405060708090a0b0c0d0e0f");
			byte[] in1 = Hex.Decode("00112233445566778899aabbccddeeff");
			byte[] out1 = Hex.Decode("1fa68b0a8112b447aef34bd8fb5a7b829d3e862371d2cfe5");

			doWrapTest(1, kek1, in1, out1);
		}

		private void doCipherTest(
			int		strength,
			byte[]	keyBytes,
			byte[]	input,
			byte[]	output)
		{
			KeyParameter key = ParameterUtilities.CreateKeyParameter("AES", keyBytes);

			IBufferedCipher inCipher = CipherUtilities.GetCipher("AES/ECB/NoPadding");
			IBufferedCipher outCipher = CipherUtilities.GetCipher("AES/ECB/NoPadding");

			try
			{
				outCipher.Init(true, key);
			}
			catch (Exception e)
			{
				Fail("AES failed initialisation - " + e, e);
			}

			try
			{
				inCipher.Init(false, key);
			}
			catch (Exception e)
			{
				Fail("AES failed initialisation - " + e, e);
			}

			//
			// encryption pass
			//
			MemoryStream bOut = new MemoryStream();

			CipherStream cOut = new CipherStream(bOut, null, outCipher);

			try
			{
				for (int i = 0; i != input.Length / 2; i++)
				{
					cOut.WriteByte(input[i]);
				}
				cOut.Write(input, input.Length / 2, input.Length - input.Length / 2);
				cOut.Close();
			}
			catch (IOException e)
			{
				Fail("AES failed encryption - " + e, e);
			}

			byte[] bytes = bOut.ToArray();

			if (!AreEqual(bytes, output))
			{
				Fail("AES failed encryption - expected "
					+ Encoding.ASCII.GetString(Hex.Encode(output)) + " got "
					+ Encoding.ASCII.GetString(Hex.Encode(bytes)));
			}

			//
			// decryption pass
			//
			MemoryStream bIn = new MemoryStream(bytes, false);

			CipherStream cIn = new CipherStream(bIn, inCipher, null);

			try
			{
//				DataInputStream dIn = new DataInputStream(cIn);
				BinaryReader dIn = new BinaryReader(cIn);

				bytes = new byte[input.Length];

				for (int i = 0; i != input.Length / 2; i++)
				{
//					bytes[i] = (byte)dIn.read();
					bytes[i] = dIn.ReadByte();
				}

				int remaining = bytes.Length - input.Length / 2;
//				dIn.readFully(bytes, input.Length / 2, remaining);
				byte[] extra = dIn.ReadBytes(remaining);
				if (extra.Length < remaining)
					throw new EndOfStreamException();
				extra.CopyTo(bytes, input.Length / 2);
			}
			catch (Exception e)
			{
				Fail("AES failed encryption - " + e, e);
			}

			if (!AreEqual(bytes, input))
			{
				Fail("AES failed decryption - expected "
					+ Encoding.ASCII.GetString(Hex.Encode(input)) + " got "
					+ Encoding.ASCII.GetString(Hex.Encode(bytes)));
			}
		}

		private void doWrapTest(
			int		id,
			byte[]	kek,
			byte[]	inBytes,
			byte[]	outBytes)
		{
			IWrapper wrapper = WrapperUtilities.GetWrapper("AESWrap");

			wrapper.Init(true, ParameterUtilities.CreateKeyParameter("AES", kek));

			try
			{
//				byte[] cText = wrapper.wrap(new SecretKeySpec(inBytes, "AES"));
				byte[] cText = wrapper.Wrap(inBytes, 0, inBytes.Length);
				if (!AreEqual(cText, outBytes))
				{
					Fail("failed wrap test " + id  + " expected "
						+ Encoding.ASCII.GetString(Hex.Encode(outBytes)) + " got "
						+ Encoding.ASCII.GetString(Hex.Encode(cText)));
				}
			}
			catch (Exception e)
			{
				Fail("failed wrap test exception " + e, e);
			}

			wrapper.Init(false, ParameterUtilities.CreateKeyParameter("AES", kek));

			try
			{
//				Key pText = wrapper.unwrap(outBytes, "AES", Cipher.SECRET_KEY);
				byte[] pText = wrapper.Unwrap(outBytes, 0, outBytes.Length);
//				if (!AreEqual(pText.getEncoded(), inBytes))
				if (!AreEqual(pText, inBytes))
				{
					Fail("failed unwrap test " + id  + " expected "
						+ Encoding.ASCII.GetString(Hex.Encode(inBytes)) + " got "
//						+ Encoding.ASCII.GetString(Hex.Encode(pText.getEncoded())));
						+ Encoding.ASCII.GetString(Hex.Encode(pText)));
				}
			}
			catch (Exception e)
			{
				Fail("failed unwrap test exception " + e, e);
			}
		}

		private void doOidTest()
		{
			string[] oids = {
				NistObjectIdentifiers.IdAes128Ecb.Id,
				NistObjectIdentifiers.IdAes128Cbc.Id,
				NistObjectIdentifiers.IdAes128Ofb.Id,
				NistObjectIdentifiers.IdAes128Cfb.Id,
				NistObjectIdentifiers.IdAes192Ecb.Id,
				NistObjectIdentifiers.IdAes192Cbc.Id,
				NistObjectIdentifiers.IdAes192Ofb.Id,
				NistObjectIdentifiers.IdAes192Cfb.Id,
				NistObjectIdentifiers.IdAes256Ecb.Id,
				NistObjectIdentifiers.IdAes256Cbc.Id,
				NistObjectIdentifiers.IdAes256Ofb.Id,
				NistObjectIdentifiers.IdAes256Cfb.Id
			};

			string[] names = {
				"AES/ECB/PKCS7Padding",
				"AES/CBC/PKCS7Padding",
				"AES/OFB/PKCS7Padding",
				"AES/CFB/PKCS7Padding",
				"AES/ECB/PKCS7Padding",
				"AES/CBC/PKCS7Padding",
				"AES/OFB/PKCS7Padding",
				"AES/CFB/PKCS7Padding",
				"AES/ECB/PKCS7Padding",
				"AES/CBC/PKCS7Padding",
				"AES/OFB/PKCS7Padding",
				"AES/CFB/PKCS7Padding"
			};

			byte[] data = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16 };

			for (int i = 0; i != oids.Length; i++)
			{
				IBufferedCipher c1 = CipherUtilities.GetCipher(oids[i]);
				IBufferedCipher c2 = CipherUtilities.GetCipher(names[i]);

				CipherKeyGenerator kg = GeneratorUtilities.GetKeyGenerator(oids[i]);
				KeyParameter k = ParameterUtilities.CreateKeyParameter(oids[i], kg.GenerateKey());

				ICipherParameters parameters = k;

				if (!names[i].StartsWith("AES/ECB"))
				{
					parameters = new ParametersWithIV(parameters, new byte[16]);
				}

				c1.Init(true, parameters);
				c2.Init(false, parameters);



				byte[] result = c2.DoFinal(c1.DoFinal(data));

				if (!AreEqual(data, result))
				{
					Fail("failed OID test");
				}

				if (k.GetKey().Length != (16 + ((i / 4) * 8)))
				{
					Fail("failed key Length test");
				}
			}
		}

		public override void PerformTest()
		{
			TestCiphers();
			TestOids();
			TestWrap();
		}

		public static void Main(
			string[] args)
		{
			RunTest(new AesTest());
		}
	}
}
