using System;
using System.IO;
using System.Text;

using NUnit.Framework;

using Org.BouncyCastle.Crypto;
using Org.BouncyCastle.Crypto.IO;
using Org.BouncyCastle.Crypto.Parameters;
using Org.BouncyCastle.Security;
using Org.BouncyCastle.Utilities.Test;

namespace Org.BouncyCastle.Tests
{
	/// <remarks>Check that cipher input/output streams are working correctly</remarks>
	[TestFixture]
	public class CipherStreamTest
		: SimpleTest
	{
		private void doRunTest(
			string name)
		{
			string lCode = "ABCDEFGHIJKLMNOPQRSTUVWXY0123456789";

			try
			{
				string baseName = name;
				if (name.IndexOf('/') >= 0)
				{
					baseName = name.Substring(0, name.IndexOf('/'));
				}

				CipherKeyGenerator kGen = GeneratorUtilities.GetKeyGenerator(baseName);

				IBufferedCipher inCipher = CipherUtilities.GetCipher(name);
				IBufferedCipher outCipher = CipherUtilities.GetCipher(name);
				KeyParameter key = ParameterUtilities.CreateKeyParameter(baseName, kGen.GenerateKey());
				MemoryStream bIn = new MemoryStream(Encoding.ASCII.GetBytes(lCode), false);
				MemoryStream bOut = new MemoryStream();

				inCipher.Init(true, key);

				// TODO Should we provide GetIV() method on IBufferedCipher?
				//            if (inCipher.getIV() != null)
				//            {
				//                outCipher.Init(false, new ParametersWithIV(key, inCipher.getIV()));
				//            }
				//            else
			{
				outCipher.Init(false, key);
			}

				CipherStream cIn = new CipherStream(bIn, inCipher, null);
				CipherStream cOut = new CipherStream(bOut, null, outCipher);

				int c = 0;

				while ((c = cIn.ReadByte()) >= 0)
				{
					cOut.WriteByte((byte)c);
				}

				cIn.Close();

				cOut.Flush();
				cOut.Close();

				string res = Encoding.ASCII.GetString(bOut.ToArray());

				if (!res.Equals(lCode))
				{
					Fail("Failed - decrypted data doesn't match.");
				}
			}
			catch (Exception e)
			{
				Fail("Failed - exception " + e.ToString());
			}
		}

		private void doTestException(
			string name)
		{
			//string lCode = "ABCDEFGHIJKLMNOPQRSTUVWXY0123456789";

			try
			{
				byte[] rc4Key = { (byte)128, (byte)131, (byte)133, (byte)134,
									(byte)137, (byte)138, (byte)140, (byte)143 };

				KeyParameter cipherKey = ParameterUtilities.CreateKeyParameter(name, rc4Key);
				IBufferedCipher ecipher = CipherUtilities.GetCipher(name);
				ecipher.Init(true, cipherKey);

				byte[] cipherText = new byte[0];
				try
				{
					// According specification Method engineUpdate(byte[] input,
					// int inputOffset, int inputLen, byte[] output, int
					// outputOffset)
					// throws ShortBufferException - if the given output buffer is
					// too
					// small to hold the result
					ecipher.ProcessBytes(new byte[20], 0, 20, cipherText, 0);

//					Fail("failed exception test - no ShortBufferException thrown");
					Fail("failed exception test - no DataLengthException thrown");
				}
//				catch (ShortBufferException e)
				catch (DataLengthException)
				{
					// ignore
				}

				// NB: The lightweight engine doesn't take public/private keys
//				try
//				{
//					IBufferedCipher c = CipherUtilities.GetCipher(name);
//
//					//                Key k = new PublicKey()
//					//                {
//					//
//					//                    public string getAlgorithm()
//					//                    {
//					//                        return "STUB";
//					//                    }
//					//
//					//                    public string getFormat()
//					//                    {
//					//                        return null;
//					//                    }
//					//
//					//                    public byte[] getEncoded()
//					//                    {
//					//                        return null;
//					//                    }
//					//
//					//                };
//					AsymmetricKeyParameter k = new AsymmetricKeyParameter(false);
//					c.Init(true, k);
//
//					Fail("failed exception test - no InvalidKeyException thrown for public key");
//				}
//				catch (InvalidKeyException)
//				{
//					// okay
//				}
//
//				try
//				{
//					IBufferedCipher c = CipherUtilities.GetCipher(name);
//
//					//				Key k = new PrivateKey()
//					//                {
//					//
//					//                    public string getAlgorithm()
//					//                    {
//					//                        return "STUB";
//					//                    }
//					//
//					//                    public string getFormat()
//					//                    {
//					//                        return null;
//					//                    }
//					//
//					//                    public byte[] getEncoded()
//					//                    {
//					//                        return null;
//					//                    }
//					//
//					//                };
//
//					AsymmetricKeyParameter k = new AsymmetricKeyParameter(true);
//					c.Init(false, k);
//
//					Fail("failed exception test - no InvalidKeyException thrown for private key");
//				}
//				catch (InvalidKeyException)
//				{
//					// okay
//				}
			}
			catch (Exception e)
			{
				Fail("unexpected exception.", e);
			}
		}

		[Test]
		public void TestRC4()
		{
			doRunTest("RC4");
		}

		[Test]
		public void TestRC4Exception()
		{
			doTestException("RC4");
		}

		[Test]
		public void TestDesEcbPkcs7()
		{
			doRunTest("DES/ECB/PKCS7Padding");
		}

		[Test]
		public void TestDesCfbNoPadding()
		{
			doRunTest("DES/CFB8/NoPadding");
		}

		public override void PerformTest()
		{
			TestRC4();
			TestRC4Exception();
			TestDesEcbPkcs7();
			TestDesCfbNoPadding();
		}

		public override string Name
		{
			get { return "CipherStreamTest"; }
		}

		public static void Main(
			string[] args)
		{
			RunTest(new CipherStreamTest());
		}
	}
}
