package org.bouncycastle.ocsp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;

import org.bouncycastle.asn1.ASN1OutputStream;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ocsp.OCSPRequest;
import org.bouncycastle.asn1.ocsp.Request;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.X509Extensions;

public class OCSPReq
{
    private OCSPRequest req;

    public OCSPReq(
        OCSPRequest req)
    {
        this.req = req;
    }

	public int getVersion()
	{
		return req.getTbsRequest().getVersion().getValue().intValue() + 1;
	}

	public GeneralName getRequestorName()
	{
		return GeneralName.getInstance(req.getTbsRequest().getRequestorName());
	}

	public Req[] getRequestList()
	{
        ASN1Sequence    seq = req.getTbsRequest().getRequestList();
        Req[]   requests = new Req[seq.size()];

        for (int i = 0; i != requests.length; i++)
        {
            requests[i] = new Req(Request.getInstance(seq.getObjectAt(i)));
        }

		return requests;
	}

	public X509Extensions getRequestExtensions()
	{
		return X509Extensions.getInstance(req.getTbsRequest().getRequestExtensions());
	}

    /**
     * return the object identifier representing the signature algorithm
     */
    public String getSignatureAlgOID()
    {
        if (!this.isSigned())
        {
            return null;
        }

		return req.getOptionalSignature().getSignatureAlgorithm().getObjectId().getId();
    }

    public byte[] getSignature()
    {
        if (!this.isSigned())
        {
            return null;
        }

		return req.getOptionalSignature().getSignature().getBytes();
    }

    public boolean isSigned()
    {
        return req.getOptionalSignature() != null;
    }

    /**
     * verify the signature against the TBSRequest object we contain.
     */
    public boolean verify(
        PublicKey   key,
        String      sigProvider)
        throws OCSPException, NoSuchProviderException
    {
        if (!this.isSigned())
        {
            throw new OCSPException("attempt to verify signature on unsigned object");
        }

        try
        {
            java.security.Signature signature = java.security.Signature.getInstance(this.getSignatureAlgOID(), sigProvider);

            signature.initVerify(key);

            ByteArrayOutputStream   bOut = new ByteArrayOutputStream();
            ASN1OutputStream        aOut = new ASN1OutputStream(bOut);

            aOut.writeObject(req.getTbsRequest());

            signature.update(bOut.toByteArray());

            return signature.verify(this.getSignature());
        }
        catch (NoSuchProviderException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new OCSPException("exception processing sig: " + e, e);
        }
    }

    /**
     * return the ASN.1 encoded representation of this object.
     */
    public byte[] getEncoded()
        throws IOException
    {
        ByteArrayOutputStream   bOut = new ByteArrayOutputStream();
        ASN1OutputStream        aOut = new ASN1OutputStream(bOut);

        aOut.writeObject(req);

        return bOut.toByteArray();
    }
}
