package org.bouncycastle.jce.provider;

import java.lang.reflect.Constructor;
import java.security.InvalidKeyException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;

import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactorySpi;
import javax.crypto.spec.DESKeySpec;
import javax.crypto.spec.DESedeKeySpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;

public class JCESecretKeyFactory
    extends SecretKeyFactorySpi
{
    protected String algName;

    protected JCESecretKeyFactory(
        String  algName)
    {
        this.algName = algName;
    }

    protected SecretKey engineGenerateSecret(
        KeySpec keySpec)
    throws InvalidKeySpecException
    {
		if ( keySpec instanceof SecretKeySpec )
		{
			return (SecretKey)keySpec;
		}

        throw new InvalidKeySpecException("Invalid KeySpec");
    }

    protected KeySpec engineGetKeySpec(
        SecretKey key,
        Class keySpec)
    throws InvalidKeySpecException
    {
        if (SecretKeySpec.class.isAssignableFrom(keySpec))
        {
            return new SecretKeySpec(key.getEncoded(), algName);
        }

        try
        {
            Class[] parameters = { byte[].class };

            Constructor c = keySpec.getConstructor(parameters);
            Object[]    p = new Object[1];

            p[0] = key.getEncoded();

            return (KeySpec)c.newInstance(p);
        }
        catch (Exception e)
        {
            throw new InvalidKeySpecException(e.toString());
        }
    }

    protected SecretKey engineTranslateKey(
        SecretKey key)
    throws InvalidKeyException
    {
        if (!key.getAlgorithm().equalsIgnoreCase(algName))
        {
            throw new InvalidKeyException("Key not of type " + algName + ".");
        }

        return new SecretKeySpec(key.getEncoded(), algName);
    }

    /*
     * classes that inherit from us
     */

    /**
     * PBE password processing for PKCS5 - ascii - single byte per character, no padding.
     */
    static public class PBE_PKCS5
        extends JCESecretKeyFactory
    {
        public PBE_PKCS5()
        {
            super("PBE/PKCS5");
        }

        protected SecretKey engineGenerateSecret(
            KeySpec keySpec)
        throws InvalidKeySpecException
        {
            if (keySpec instanceof PBEKeySpec)
            {
                PBEKeySpec pbeSpec = (PBEKeySpec)keySpec;

                return new JCEPBEKey(false, "PBE/PKCS5", pbeSpec.getPassword());
            }
            throw new InvalidKeySpecException("Invalid KeySpec");
        }
    }

    /**
     * PBE password processing for PKCS12 - unicode, big endian order, with two zero pad bytes.
     */
    static public class PBE_PKCS12
        extends JCESecretKeyFactory
    {
        public PBE_PKCS12()
        {
            super("PBE/PKCS12");
        }

        protected SecretKey engineGenerateSecret(
            KeySpec keySpec)
        throws InvalidKeySpecException
        {
            if (keySpec instanceof PBEKeySpec)
            {
                PBEKeySpec pbeSpec = (PBEKeySpec)keySpec;

                return new JCEPBEKey(true, "PBE/PKCS12", pbeSpec.getPassword());
            }
            throw new InvalidKeySpecException("Invalid KeySpec");
        }
    }

	static public class DES
		extends JCESecretKeyFactory
	{
        public DES()
        {
            super("DES");
        }

		protected SecretKey engineGenerateSecret(
			KeySpec keySpec)
		throws InvalidKeySpecException
		{
			if ( keySpec instanceof DESKeySpec )
			{
				DESKeySpec desKeySpec = (DESKeySpec)keySpec;
				return new SecretKeySpec(desKeySpec.getKey(), "DES");
			}

			return super.engineGenerateSecret(keySpec);
		}
	}

	static public class DESede
		extends JCESecretKeyFactory
	{
        public DESede()
        {
            super("DESede");
        }

        protected KeySpec engineGetKeySpec(
            SecretKey key,
            Class keySpec)
        throws InvalidKeySpecException
        {
            if (SecretKeySpec.class.isAssignableFrom(keySpec))
            {
                return new SecretKeySpec(key.getEncoded(), algName);
            }
            else if (DESedeKeySpec.class.isAssignableFrom(keySpec))
            {
                byte[]  bytes = key.getEncoded();

                try
                {
                    if (bytes.length == 16)
                    {
                        byte[]  longKey = new byte[24];

                        System.arraycopy(bytes, 0, longKey, 0, 16);
                        System.arraycopy(bytes, 0, longKey, 16, 8);

                        return new DESedeKeySpec(longKey);
                    }
                    else
                    {
                        return new DESedeKeySpec(bytes);
                    }
                }
                catch (Exception e)
                {
                    throw new InvalidKeySpecException(e.toString());
                }
            }

            throw new InvalidKeySpecException("Invalid KeySpec");
        }

		protected SecretKey engineGenerateSecret(
			KeySpec keySpec)
		throws InvalidKeySpecException
		{
			if ( keySpec instanceof DESedeKeySpec )
			{
				DESedeKeySpec desKeySpec = (DESedeKeySpec)keySpec;
				return new SecretKeySpec(desKeySpec.getKey(), "DESede");
			}

			return super.engineGenerateSecret(keySpec);
		}
	}
}
