package org.bouncycastle.ocsp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Enumeration;

import org.bouncycastle.asn1.ASN1OutputStream;
import org.bouncycastle.asn1.DEROutputStream;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ocsp.BasicOCSPResponse;

public class BasicOCSPResp
{
    BasicOCSPResponse   resp;
    X509Certificate[]   chain = null;

    public BasicOCSPResp(
        BasicOCSPResponse   resp)
    {
        this.resp = resp;
    }

    public String getSignatureAlgOID()
    {
        return resp.getSignatureAlgorithm().getObjectId().getId();
    }

    public RespData getResponseData()
    {
        return new RespData(resp.getTbsResponseData());
    }

    public byte[] getSignature()
    {
        return resp.getSignature().getBytes();
    }

    public X509Certificate[] getCerts(
        String  provider)
        throws OCSPException, NoSuchProviderException
    {
        if (chain == null)
        {
            ArrayList               certs = new ArrayList();
            ByteArrayOutputStream   bOut = new ByteArrayOutputStream();
            ASN1OutputStream        aOut = new ASN1OutputStream(bOut);
            CertificateFactory      cf;

            try
            {
                cf = CertificateFactory.getInstance("X.509", provider);
            }
            catch (CertificateException ex)
            {
                throw new OCSPException("can't get certificate factory.", ex);
            }

            //
            // load the certificates and revocation lists if we have any
            //
            ASN1Sequence s = resp.getCerts();

            if (s != null)
            {
                Enumeration e = s.getObjects();

                while (e.hasMoreElements())
                {
                    try
                    {
                        aOut.writeObject(e.nextElement());

                        certs.add(cf.generateCertificate(
                            new ByteArrayInputStream(bOut.toByteArray())));
                    }
                    catch (IOException ex)
                    {
                        throw new OCSPException(
                                "can't re-encode certificate!", ex);
                    }
                    catch (CertificateException ex)
                    {
                        throw new OCSPException(
                                "can't re-encode certificate!", ex);
                    }

                    bOut.reset();
                }
            }

            chain = (X509Certificate[])certs.toArray(new X509Certificate[certs.size()]);
        }

        return chain;
    }

    /**
     * verify the signature against the tbsResponseData object we contain.
     */
    public boolean verify(
        PublicKey   key,
        String      sigProvider)
        throws OCSPException, NoSuchProviderException
    {
        try
        {
            java.security.Signature signature = java.security.Signature.getInstance(this.getSignatureAlgOID(), sigProvider);

            signature.initVerify(key);

            ByteArrayOutputStream   bOut = new ByteArrayOutputStream();
            DEROutputStream         dOut = new DEROutputStream(bOut);

            dOut.writeObject(resp.getTbsResponseData());

            signature.update(bOut.toByteArray());

            return signature.verify(this.getSignature());
        }
        catch (NoSuchProviderException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new OCSPException("exception processing sig: " + e, e);
        }
    }

    /**
     * return the ASN.1 encoded representation of this object.
     */
    public byte[] getEncoded()
        throws IOException
    {
        ByteArrayOutputStream   bOut = new ByteArrayOutputStream();
        ASN1OutputStream        aOut = new ASN1OutputStream(bOut);

        aOut.writeObject(resp);

        return bOut.toByteArray();
    }
}
