package org.bouncycastle.openpgp.examples;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.Security;

import org.bouncycastle.bcpg.ArmoredInputStream;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPEncryptedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPUtil;

/**
 * A simple utility class that encrypts/decrypts public key based
 * encryption files.
 * <p>
 * To encrypt a file: KeyBasedFileProcessor -e [-a] fileName publicKeyFile.<br>
 * If -a is specified the output file will be "ascii-armored".
 * <p>
 * To decrypt: KeyBasedFileProcessor -d fileName secretKeyFile passPhrase.
 * <p>
 * Note: this example will silently overwrite files, nor does it pay any attention to
 * the specification of "_CONSOLE" in the filename. It also expects that a single pass phrase
 * will have been used.
 */
public class KeyBasedFileProcessor
{
	private static PGPPublicKey readPublicKey(
		InputStream	in)
		throws IOException
	{
		PGPPublicKeyRing		pgpPub = new PGPPublicKeyRing(in);

		return pgpPub.getPublicKey();
	}
	
	private static PGPPrivateKey readSecretKey(
		InputStream	in,
		char[]			pass)
		throws IOException, PGPException, NoSuchProviderException
	{
		PGPSecretKeyRing		pgpPriv = new PGPSecretKeyRing(in);

		return pgpPriv.getSecretKey().extractPrivateKey(pass, "BC");
	}
	
	/**
	 * decrypt the passed in message stream
	 */
	private static void decryptFile(
		InputStream		in,
		PGPPrivateKey	decKey,
		boolean			armored)
		throws Exception
	{
		if (armored)
		{
			in = new ArmoredInputStream(in);
		}
		
        PGPObjectFactory pgpF = new PGPObjectFactory(in);

		PGPEncryptedDataList	enc = (PGPEncryptedDataList)pgpF.nextObject();
		
        PGPPublicKeyEncryptedData	pbe = (PGPPublicKeyEncryptedData)enc.get(0);

        InputStream clear = pbe.getDataStream(decKey, "BC");
        
        PGPObjectFactory		pgpFact = new PGPObjectFactory(clear);
        
        PGPCompressedData	cData = (PGPCompressedData)pgpFact.nextObject();

		pgpFact = new PGPObjectFactory(cData.getDataStream());
		
		PGPLiteralData				ld = (PGPLiteralData)pgpFact.nextObject();
		
		FileOutputStream			fOut = new FileOutputStream(ld.getFileName());
		
		InputStream	unc = ld.getInputStream();
		int	ch;
		
		while ((ch = unc.read()) >= 0)
		{
			fOut.write(ch);
		}
	}

	private static void encryptFile(
		OutputStream	out,
		String				fileName,
		PGPPublicKey		encKey,
		boolean			armor)
		throws IOException, NoSuchProviderException, PGPException
	{	
		if (armor)
		{
			out = new ArmoredOutputStream(out);
		}
		
		ByteArrayOutputStream	bOut = new ByteArrayOutputStream();
		

		PGPCompressedDataGenerator comData = new PGPCompressedDataGenerator(
																PGPCompressedDataGenerator.ZIP);
																
		PGPUtil.writeFileToLiteralData(comData.open(bOut), PGPLiteralData.BINARY, new File(fileName));
		
		comData.close();
		
		PGPEncryptedDataGenerator	cPk = new PGPEncryptedDataGenerator(PGPEncryptedDataGenerator.CAST5, new SecureRandom(), "BC");
			
		cPk.addMethod(encKey);
		
		byte[]				bytes = bOut.toByteArray();
		
		OutputStream	cOut = cPk.open(out, bytes.length);

		cOut.write(bytes);
		
		cOut.close();
    }

    public static void main(
    	String[] args)
    	throws Exception
    {
        Security.addProvider(new BouncyCastleProvider());

		if (args[0].equals("-e"))
		{
			if (args[1].equals("-a"))
			{
				FileInputStream		keyIn = new FileInputStream(args[3]);
				FileOutputStream	out = new FileOutputStream(args[2] + ".asc");
				encryptFile(out, args[2], readPublicKey(keyIn), true);
			}
			else
			{
				FileInputStream		keyIn = new FileInputStream(args[2]);
				FileOutputStream	out = new FileOutputStream(args[1] + ".bpg");
				encryptFile(out, args[1], readPublicKey(keyIn), false);
			}
		}
		else if (args[0].equals("-d"))
		{
			if (args[1].equals("-a"))
			{
				FileInputStream	keyIn = new FileInputStream(args[3]);
				FileInputStream	in = new FileInputStream(args[2]);
				decryptFile(in, readSecretKey(keyIn, args[4].toCharArray()), true);
			}
			else
			{
				FileInputStream	keyIn = new FileInputStream(args[2]);
				FileInputStream	in = new FileInputStream(args[1]);
				decryptFile(in, readSecretKey(keyIn, args[3].toCharArray()), false);
			}
		}
		else
		{
			System.err.println("usage: KeyBasedFileProcessor -e|-d [-a] file passPhrase");
		}
    }
}
