package org.bouncycastle.openpgp.examples;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.ByteArrayOutputStream;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Security;
import java.security.SignatureException;

import org.bouncycastle.bcpg.ArmoredInputStream;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.BCPGInputStream;
import org.bouncycastle.bcpg.BCPGOutputStream;
import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.PublicKeyAlgorithmTags;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureList;

/**
 * A simple utility class that creates clear signed files and verifies them.
 * <p>
 * To sign a file: ClearSignedFileProcessor -s [-a] fileName outputFile secretKey passPhrase.<br>
 * If -a is specified the output file will be "ascii-armored".
 * <p>
 * To decrypt: ClearSignedFileProcessor -v [-a] fileName signatureFile publicKeyFile.
 * <p>
 * Note: This example does not dash escape the input on signing or look for dash escaping on verification. See section 7 of RFC 2440 for further details.
 */
public class ClearSignedFileProcessor
{
	/**
	 * verify a SHA1 clear text signed file
	 */
	private static void verifyFile(
		InputStream		in,
		InputStream		keyIn)
		throws Exception
	{

        //
        // skip the headers
        //
        int     count = 0;
		int    	ch;

		while ((ch = in.read()) >= 0)
        {
            if (ch == '\n')
            {
                if (++count == 3)
                {
                    break;
                }
            }
        }

        //
        // read the input, making sure it is canonicalised.
        //
        boolean                 newLine = false;
        ByteArrayOutputStream   bOut = new ByteArrayOutputStream();

		while ((ch = in.read()) >= 0 && (!newLine && ch != '-'))
		{
            if (newLine)
            {
			    bOut.write((byte)'\r');
			    bOut.write((byte)'\n');
                newLine = false;
            }
            if (ch == '\r')
            {
                continue;
            }
            if (ch == '\n')
            {
                newLine = true;
                continue;
            }
			bOut.write((byte)ch);
		}
		
        in = new ArmoredInputStream(in);
		
		PGPPublicKeyRing			pgpRing = new PGPPublicKeyRing(new BCPGInputStream(keyIn));

		PGPObjectFactory			pgpFact = new PGPObjectFactory(in);
		PGPSignatureList         	p3 = (PGPSignatureList)pgpFact.nextObject();
        PGPSignature                sig = (PGPSignature)p3.get(0);

        sig.initVerify(pgpRing.getPublicKey(), "BC");

        sig.update(bOut.toByteArray());

		if (sig.verify())
		{
			System.out.println("signature verified.");
		}
		else
		{
			System.out.println("signature verification failed.");
		}
	}

    /**
     * create a clear text signed file.
     */
	private static void signFile(
		String			fileName,
		InputStream		keyIn,
		OutputStream	out,
		char[]			pass)
		throws IOException, NoSuchAlgorithmException, NoSuchProviderException, PGPException, SignatureException
	{	
		PGPSecretKeyRing			pgpPriv = new PGPSecretKeyRing(new BCPGInputStream(keyIn));
		PGPPrivateKey				pgpPrivKey = pgpPriv.getSecretKey().extractPrivateKey(pass, "BC");		
		PGPSignatureGenerator	sGen = new PGPSignatureGenerator(PublicKeyAlgorithmTags.RSA_GENERAL, HashAlgorithmTags.SHA1, "BC");
		
		sGen.initSign(PGPSignature.CANONICAL_TEXT_DOCUMENT, pgpPrivKey);
		
		FileInputStream				fIn = new FileInputStream(fileName);
		int							ch = 0;
		
        out.write("-----BEGIN PGP SIGNED MESSAGE-----\n".getBytes());
        out.write("Hash: SHA1\n\n".getBytes());

        boolean newLine = false;

        //
        // canonicalise the output - note the last \r\n in the file is ignored
        //
		while ((ch = fIn.read()) >= 0)
		{
            out.write(ch);
            if (newLine)
            {
			    sGen.update((byte)'\r');
			    sGen.update((byte)'\n');
                newLine = false;
            }
            if (ch == '\r')
            {
                continue;
            }
            if (ch == '\n')
            {
                newLine = true;
                continue;
            }
			sGen.update((byte)ch);
		}
		
        out = new ArmoredOutputStream(out);

		BCPGOutputStream			bOut = new BCPGOutputStream(out);
		
		sGen.generate().encode(bOut);

		out.close();
    }

    public static void main(
    	String[] args)
    	throws Exception
    {
        Security.addProvider(new BouncyCastleProvider());

		if (args[0].equals("-s"))
		{
            FileInputStream		keyIn = new FileInputStream(args[2]);
            FileOutputStream	out = new FileOutputStream(args[1] + ".asc");
            
            signFile(args[1], keyIn, out, args[3].toCharArray());
		}
		else if (args[0].equals("-v"))
		{
            FileInputStream	in = new FileInputStream(args[1]);
            FileInputStream	keyIn = new FileInputStream(args[2]);
				
            verifyFile(in, keyIn);
		}
		else
		{
			System.err.println("usage: ClearSignedFileProcessor [-s file keyfile passPhrase]|[-v sigFile keyFile]");
		}
    }
}
