package org.bouncycastle.openpgp.examples;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Security;
import java.security.SignatureException;

import org.bouncycastle.bcpg.ArmoredInputStream;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.BCPGInputStream;
import org.bouncycastle.bcpg.BCPGOutputStream;
import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.PublicKeyAlgorithmTags;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureList;

/**
 * A simple utility class that creates seperate signatures for files and verifies them.
 * <p>
 * To sign a file: DetachedSignatureProcessor -s [-a] fileName outputFile secretKey passPhrase.<br>
 * If -a is specified the output file will be "ascii-armored".
 * <p>
 * To decrypt: DetachedSignatureProcessor -v [-a] fileName signatureFile publicKeyFile.
 * <p>
 * Note: this example will silently overwrite files.
 * It also expects that a single pass phrase
 * will have been used.
 */
public class DetachedSignatureProcessor
{
	/**
	 * decrypt the passed in message stream
	 */
	private static void verifySignature(
		String			fileName,
		InputStream		in,
		InputStream		keyIn,
		boolean			armored)
		throws Exception
	{
		if (armored)
		{
			in = new ArmoredInputStream(in);
		}
		
		PGPObjectFactory			pgpFact = new PGPObjectFactory(in);

		PGPCompressedData 			c1 = (PGPCompressedData)pgpFact.nextObject();

		pgpFact = new PGPObjectFactory(c1.getDataStream());
			
		PGPPublicKeyRing			pgpRing = new PGPPublicKeyRing(new BCPGInputStream(keyIn));

		PGPPublicKey				key = pgpRing.getPublicKey();
		InputStream             	dIn = new FileInputStream(fileName);
		int                     	ch;
		PGPSignatureList         	p3 = (PGPSignatureList)pgpFact.nextObject();
        PGPSignature                sig = (PGPSignature)p3.get(0);

        sig.initVerify(key, "BC");

		while ((ch = dIn.read()) >= 0)
		{
			sig.update((byte)ch);
		}


		if (sig.verify())
		{
			System.out.println("signature verified.");
		}
		else
		{
			System.out.println("signature verification failed.");
		}
	}

	private static void createSignature(
		String			fileName,
		InputStream		keyIn,
		OutputStream	out,
		char[]			pass,
		boolean			armor)
		throws IOException, NoSuchAlgorithmException, NoSuchProviderException, PGPException, SignatureException
	{	
		if (armor)
		{
			out = new ArmoredOutputStream(out);
		}
		
		PGPSecretKeyRing			pgpPriv = new PGPSecretKeyRing(new BCPGInputStream(keyIn));
		PGPPrivateKey				pgpPrivKey = pgpPriv.getSecretKey().extractPrivateKey(pass, "BC");		
		PGPSignatureGenerator	sGen = new PGPSignatureGenerator(PublicKeyAlgorithmTags.RSA_GENERAL, HashAlgorithmTags.SHA1, "BC");
		
		sGen.initSign(PGPSignature.BINARY_DOCUMENT, pgpPrivKey);
		
		PGPCompressedDataGenerator cGen = new PGPCompressedDataGenerator(
                                            PGPCompressedDataGenerator.ZLIB);
		
		BCPGOutputStream			bOut = new BCPGOutputStream(cGen.open(out));
		
		FileInputStream				fIn = new FileInputStream(fileName);
		int							ch = 0;
		
		while ((ch = fIn.read()) >= 0)
		{
			sGen.update((byte)ch);
		}
		
		sGen.generate().encode(bOut);

		cGen.close();
		
		out.close();
    }

    public static void main(
    	String[] args)
    	throws Exception
    {
        Security.addProvider(new BouncyCastleProvider());

		if (args[0].equals("-s"))
		{
			if (args[1].equals("-a"))
			{
				FileInputStream		keyIn = new FileInputStream(args[3]);
				FileOutputStream	out = new FileOutputStream(args[2] + ".asc");
				
				createSignature(args[2], keyIn, out, args[4].toCharArray(), true);
			}
			else
			{
				FileInputStream		keyIn = new FileInputStream(args[2]);
				FileOutputStream	out = new FileOutputStream(args[1] + ".bpg");
				
				createSignature(args[1], keyIn, out, args[3].toCharArray(), false);
			}
		}
		else if (args[0].equals("-v"))
		{
			if (args[1].equals("-a"))
			{
				FileInputStream	in = new FileInputStream(args[3]);
				FileInputStream	keyIn = new FileInputStream(args[4]);
				
				verifySignature(args[2], in, keyIn, true);
			}
			else
			{
				FileInputStream	in = new FileInputStream(args[2]);
				FileInputStream	keyIn = new FileInputStream(args[3]);
				
				verifySignature(args[1], in, keyIn, false);
			}
		}
		else
		{
			System.err.println("usage: ClearSignedFileProcessor [-s [-a] file keyfile passPhrase]|[-v [-a] file sigFile keyFile]");
		}
    }
}
