package org.bouncycastle.openpgp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.spec.DSAPublicKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;

import org.bouncycastle.bcpg.*;

import org.bouncycastle.jce.spec.ElGamalParameterSpec;
import org.bouncycastle.jce.spec.ElGamalPublicKeySpec;

/**
 * general class to handle a PGP public key object.
 */
public class PGPPublicKey
	implements PublicKeyAlgorithmTags
{
    private long 	keyID;
	private byte[]	fingerprint;
	private int		keyStrength;
	
	PublicKeyPacket	publicPk;
	ArrayList					ids = new ArrayList();
	ArrayList					idSigs = new ArrayList();
	
	private SignaturePacket	sig = null;

	private void init(
		MessageDigest	sha)
		throws IOException
	{
		BCPGKey				key = publicPk.getKey();
		
		if (publicPk.getVersion() <= 3)
		{
			RSAPublicBCPGKey	rK = (RSAPublicBCPGKey)key;
			
			this.keyID = rK.getModulus().longValue();
				
			sha.update(new MPInteger(rK.getModulus()).getEncoded());
			sha.update(new MPInteger(rK.getPublicExponent()).getEncoded());
			
			this.fingerprint = sha.digest();
			this.keyStrength = rK.getModulus().bitLength();
		}
		else
		{
				byte[]							kBytes = publicPk.getEncodedContents();

				sha.update((byte)0x99);
				sha.update((byte)(kBytes.length >> 8));
				sha.update((byte)kBytes.length);
				sha.update(kBytes);
				
				this.fingerprint = sha.digest();
				this.keyID = ((long)(fingerprint[fingerprint.length - 8] & 0xff) << 56)
								| ((long)(fingerprint[fingerprint.length - 7] & 0xff) << 48)
								| ((long)(fingerprint[fingerprint.length - 6] & 0xff) << 40)
								| ((long)(fingerprint[fingerprint.length - 5] & 0xff) << 32)
								| ((long)(fingerprint[fingerprint.length - 4] & 0xff) << 24)
								| ((long)(fingerprint[fingerprint.length - 3] & 0xff) << 16)
								| ((long)(fingerprint[fingerprint.length - 2] & 0xff) << 8)
								| ((fingerprint[fingerprint.length - 1] & 0xff));
				
				if (key instanceof RSAPublicBCPGKey)
				{
					this.keyStrength = ((RSAPublicBCPGKey)key).getModulus().bitLength();
				}
				else if (key instanceof DSAPublicBCPGKey)
				{
					this.keyStrength = ((DSAPublicBCPGKey)key).getG().bitLength();
				}
				else if (key instanceof ElGamalPublicBCPGKey)
				{
					this.keyStrength = ((ElGamalPublicBCPGKey)key).getG().bitLength();
				}
		}
	}
	
	/**
	  * Constructor for a sub-key.
	  * 
	  * @param pk
	  * @param sha
	  * @param sig
	  */
	PGPPublicKey(
		PublicKeyPacket 	publicPk, 
		MessageDigest 	sha, 
		SignaturePacket 	sig)
		throws IOException
	 {
        this.publicPk = publicPk;
		this.sig = sig;
		
		init(sha);
	 }

	PGPPublicKey(
		PublicKeyPacket 	publicPk,
		MessageDigest	sha,
		ArrayList			ids,
		ArrayList			idSigs)
		throws IOException
	{
		this.publicPk = publicPk;
		this.ids = ids;
		this.idSigs = idSigs;
		
		init(sha);
	}
	
	/**
	 * @return the version of this key.
	 */
	public int getVersion()
	{
		return publicPk.getVersion();
	}
	
	/**
	 * @return creation time of key.
	 */
	public Date getCreationTime()
	{
		return publicPk.getTime();
	}
	
	/**
	 * @return number of valid days from creation time - zero means no
	 * expiry.
	 */
	public int getValidDays()
	{
		return publicPk.getValidDays();
	}
	
	/**
	 * Return the keyID associated with the public key.
	 * 
	 * @return long
	 */
	public long getKeyID()
	{
		return keyID;
	}
	
	/**
	 * Return the 16 byte fingerprint of the key.
	 * 
	 * @return key fingerprint.
	 */
	public byte[] getFingerprint()
	{
		byte[]	tmp = new byte[fingerprint.length];
		
		System.arraycopy(fingerprint, 0, tmp, 0, tmp.length);
		
		return tmp;
	}
	
	/**
	 * Return the algorithm code associated with the public key.
	 * 
	 * @return int
	 */
	public int getAlgorithm()
	{
		return publicPk.getAlgorithm();
	}
	
	/**
	 * Return the strength of the key in bits.
	 * 
	 * @return bit strenght of key.
	 */
	public int getBitStrength()
	{
		return keyStrength;
	}
	
	/**
	 * Return the public key contained in the object.
	 * 
	 * @param provider provider to construct the key for.
	 * 
	 * @return PublicKey
	 * @throws PGPException
	 * @throws NoSuchProviderException
	 */
	public PublicKey getKey(
		String				provider)
		throws PGPException, NoSuchProviderException
	{
		KeyFactory						fact;
		
		try
		{
			switch (publicPk.getAlgorithm())
			{
			case RSA_ENCRYPT:
			case RSA_GENERAL:
				RSAPublicBCPGKey     rsaK = (RSAPublicBCPGKey)publicPk.getKey();
				RSAPublicKeySpec    rsaSpec = new RSAPublicKeySpec(rsaK.getModulus(), rsaK.getPublicExponent());
	
				fact = KeyFactory.getInstance("RSA", provider);
				
				return fact.generatePublic(rsaSpec);
			case DSA:
				DSAPublicBCPGKey	dsaK = (DSAPublicBCPGKey)publicPk.getKey();
				DSAPublicKeySpec 	dsaSpec = new DSAPublicKeySpec(dsaK.getY(), dsaK.getP(), dsaK.getQ(), dsaK.getG());
			
				fact = KeyFactory.getInstance("DSA", provider);
				
				return fact.generatePublic(dsaSpec);
			case ELGAMAL_ENCRYPT:
			case ELGAMAL_GENERAL:
				ElGamalPublicBCPGKey	elK = (ElGamalPublicBCPGKey)publicPk.getKey();
				ElGamalPublicKeySpec	elSpec = new ElGamalPublicKeySpec(elK.getY(), new ElGamalParameterSpec(elK.getP(), elK.getG()));
				
				fact = KeyFactory.getInstance("ElGamal", provider);
				
				return fact.generatePublic(elSpec);
			default:
				throw new PGPException("unknown public key algorithm encountered");
			}
		}
		catch (PGPException e)
		{
			throw e;
		}
		catch (Exception e)
		{
			throw new PGPException("exception constructing public key", e);
		}
	}
	
	/**
	 * Return any userIDs associated with the key.
	 * 
	 * @return an iterator of Strings.
	 */
	public Iterator getUserIDs()
	{
		ArrayList	temp = new ArrayList();
		
		for (int i = 0; i != ids.size(); i++)
		{
			if (ids.get(i) instanceof String)
			{
				temp.add(ids.get(i));
			}
		}
		
		return temp.iterator();
	}
	
	/**
	 * Return any user attribute vectors associated with the key.
	 * 
	 * @return an iterator of Strings.
	 */
	public Iterator getUserAttributes()
	{
		ArrayList	temp = new ArrayList();
		
		for (int i = 0; i != ids.size(); i++)
		{
			if (ids.get(i) instanceof PGPUserAttributeSubpacketVector)
			{
				temp.add(ids.get(i));
			}
		}
		
		return temp.iterator();
	}
	
	public Iterator getSignaturesForID(
		String	id)
	{
		for (int i = 0; i != ids.size(); i++)
		{
			if (id.equals(ids.get(i)))
			{
				return ((ArrayList)idSigs.get(i)).iterator();
			}
		}
		
		return null;
	}
	
	public byte[] getEncoded() 
		throws IOException
	{
		ByteArrayOutputStream	bOut = new ByteArrayOutputStream();
		
		this.encode(bOut);
		
		return bOut.toByteArray();
	}
	
	public void encode(
		OutputStream	outStream) 
		throws IOException
	{
		BCPGOutputStream	out;
		
		if (outStream instanceof BCPGOutputStream)
		{
			out = (BCPGOutputStream)outStream;
		}
		else
		{
			out = new BCPGOutputStream(outStream);
		}
		
		out.writePacket(publicPk);
        
        if (sig == null)	// not a sub-key
        {
			for (int i = 0; i != ids.size(); i++)
			{
				if (ids.get(i) instanceof String)
				{
					String	id = (String)ids.get(i);
		        	
					out.writePacket(new UserIDPacket(id));
				}
				else
				{
					PGPUserAttributeSubpacketVector	v = (PGPUserAttributeSubpacketVector)ids.get(i);

				    out.writePacket(new UserAttributePacket(v.toSubpacketArray()));
				}
	        	
				ArrayList	sigs = (ArrayList)idSigs.get(i);
				for (int j = 0; j != sigs.size(); j++)
				{
					((PGPSignature)sigs.get(j)).encode(out);
				}
			}
        }
        else
        {
        	sig.encode(out);
        }
	}
}
