package org.bouncycastle.x509.extension;

import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.x509.AuthorityKeyIdentifier;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jce.PrincipalUtil;

/**
 * A high level authority key identifier.
 */
public class AuthorityKeyIdentifierStructure
    extends AuthorityKeyIdentifier
{
    private AuthorityKeyIdentifier authKeyID;
    
    public AuthorityKeyIdentifierStructure(
        byte[]  encodedValue)
        throws IOException
    {
        super((ASN1Sequence)X509ExtensionUtil.fromExtensionValue(encodedValue));
    }
    
    private static ASN1Sequence fromCertificate(
        X509Certificate certificate)
        throws CertificateParsingException
    {
        try
        {
            SubjectPublicKeyInfo info = new SubjectPublicKeyInfo(
                                            (ASN1Sequence)new ASN1InputStream(certificate.getPublicKey().getEncoded()).readObject());
            
            GeneralName             genName = new GeneralName(PrincipalUtil.getSubjectX509Principal(certificate));
            ASN1EncodableVector     v = new ASN1EncodableVector();
            
            v.add(genName);
            
            return (ASN1Sequence)new AuthorityKeyIdentifier(
                                info, new GeneralNames(new DERSequence(v)), certificate.getSerialNumber()).toASN1Object();
        }
        catch (Exception e)
        {
            throw new CertificateParsingException("Exception extracting certificate details: " + e.toString());
        }
    }
    
    private static ASN1Sequence fromKey(
        PublicKey pubKey)
        throws InvalidKeyException
    {
        try
        {
        SubjectPublicKeyInfo info = new SubjectPublicKeyInfo(
                                        (ASN1Sequence)new ASN1InputStream(pubKey.getEncoded()).readObject());
        
        return (ASN1Sequence)new AuthorityKeyIdentifier(info).toASN1Object();
        }
        catch (Exception e)
        {
            throw new InvalidKeyException("can't process key: " + e);
        }

    }
    
    /**
     * Create an AuthorityKeyIdentifier using the passed in certificate's public
     * key, issuer and serial number.
     * 
     * @param certificate the certificate providing the information.
     * @throws CertificateParsingException if there is a problem processing the certificate
     */
    public AuthorityKeyIdentifierStructure(
        X509Certificate certificate)
        throws CertificateParsingException
    {
        super(fromCertificate(certificate));
    }
    
    /**
     * Create an AuthorityKeyIdentifier using just the hash of the 
     * public key.
     * 
     * @param pubKey the key to generate the hash from.
     * @throws InvalidKeyException if there is a problem using the key.
     */
    public AuthorityKeyIdentifierStructure(
        PublicKey pubKey) 
        throws InvalidKeyException
    {
        super(fromKey(pubKey));
    }
}
