/*
 * Decompiled with CFR 0.152.
 */
package org.bouncycastle.openpgp.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.Provider;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Date;
import java.util.Iterator;
import javax.crypto.Cipher;
import org.bouncycastle.bcpg.BCPGOutputStream;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPKeyPair;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPLiteralDataGenerator;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPOnePassSignature;
import org.bouncycastle.openpgp.PGPOnePassSignatureList;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPV3SignatureGenerator;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.SimpleTestResult;
import org.bouncycastle.util.test.Test;
import org.bouncycastle.util.test.TestResult;

public class PGPRSATest
implements Test {
    byte[] testPubKey = Base64.decode((String)"mIsEPz2nJAEEAOTVqWMvqYE693qTgzKv/TJpIj3hI8LlYPC6m1dk0z3bDLwVVk9FFAB+CWS8RdFOWt/FG3tEv2nzcoNdRvjv9WALyIGNawtae4Ml6oAT06/511yUzXHOk+9xK3wkXN5jdzUhf4cA2oGpLSV/pZlocsIDL+jCUQtumUPwFodmSHhzAAYptC9FcmljIEVjaGlkbmEgKHRlc3Qga2V5KSA8ZXJpY0Bib3VuY3ljYXN0bGUub3JnPoi4BBMBAgAiBQI/PackAhsDBQkAg9YABAsHAwIDFQIDAxYCAQIeAQIXgAAKCRA1WGFG/fPzc8WMA/9BbjuB8E48QAlxoiVf9U8SfNelrz/ONJA/bMvWr/JnOGA9PPmFD5Uc+kV/q+i94dEMjsC5CQ1moUHWSP2xlQhbOzBP2+oPXw3z2fBs9XJgnTH6QWMAAvLs3ug9po0loNHLobT/D/XdXvcrb3wvwvPT2FptZqrtonH/OdzT9JdfrA==");
    byte[] testPrivKey = Base64.decode((String)"lQH8BD89pyQBBADk1aljL6mBOvd6k4Myr/0yaSI94SPC5WDwuptXZNM92wy8FVZPRRQAfglkvEXRTlrfxRt7RL9p83KDXUb47/VgC8iBjWsLWnuDJeqAE9Ov+ddclM1xzpPvcSt8JFzeY3c1IX+HANqBqS0lf6WZaHLCAy/owlELbplD8BaHZkh4cwAGKf4DAwKbLeIOVYTEdWD5v/YgW8ERs0pDsSIfBTvsJp2qA798KeFuED6jGsHUzdi1M9906PRtplQgnoYmYQrzEc6DXAiAtBR4Kuxi4XHx0ZR2wpVlVxm2Ypgz7pbBNWcWqzvw33inl7tR4IDsRdJOY8cFlN+1tSCf16sDidtKXUVjRjZNYJytH18VfSPlGXMeYgtw3cSGNTERwKaq5E/SozT2MKTiORO0g0Mtyz+9MEB6XVXFavMun/mXURqbZN/k9BFbz+TadpkihrLD1xw3Hp+tpe4CwPQ2GdWKI9KNo5gEnbkJgLrSMGgWalPhknlNHRyYbSq6lbIMJEE3LoOwvYWwweR1+GrV9farJESdunl1mDr5/d6rKru+FFDwZM3na1IF4Ei4FpqhivZ4zG6pN5XqLy+AK85EiW4XH0yAKX1O4YlbmDU4BjxhiwTdwuVMCjLO5++jkz5BBQWdFX8CCMA4FJl36G70IbGzuFfOj07ly7QvRXJpYyBFY2hpZG5hICh0ZXN0IGtleSkgPGVyaWNAYm91bmN5Y2FzdGxlLm9yZz6IuAQTAQIAIgUCPz2nJAIbAwUJAIPWAAQLBwMCAxUCAwMWAgECHgECF4AACgkQNVhhRv3z83PFjAP/QW47gfBOPEAJcaIlX/VPEnzXpa8/zjSQP2zL1q/yZzhgPTz5hQ+VHPpFf6voveHRDI7AuQkNZqFB1kj9sZUIWzswT9vqD18N89nwbPVyYJ0x+kFjAALy7N7oPaaNJaDRy6G0/w/13V73K298L8Lz09habWaq7aJx/znc0/SXX6w=");
    byte[] testPubKeyV3 = Base64.decode((String)"mQCNAz+zvlEAAAEEAMS22jgXbOZ/D3xWgM2kauSdzrwlU7Ms5hDW05ObqQyOFfQoKKMhfupyoa7J3x04VVBKu6Eomvr1es+VImH0esoeWFFahNOYq/I+jRRBwoOhAGZ5UB2/hRd7rFmxqp6sCXi8wmLO2tAorlTzAiNNvl7xF4cQZpc0z56Fwdi2fBUJAAURtApGSVhDSVRZX1FBiQCVAwUQP7O+UZ6Fwdi2fBUJAQFMwwQAqRnFsdg4xQnB8Y5d4cOpXkIn9AZgYS3cxtuSJB84vG2CgC39nfv4c+nlLkWP4puG+mZuJNgVoE84cuAF4I//1anKjlU7q1M6rFQnt5S4uxPyG3dFXmgyU1b4PBOnA0tIxjPzlIhJAMsPCGGA5+5M2JP0ad6RnzqzE3EENMX+GqY=");
    byte[] testPrivKeyV3 = Base64.decode((String)"lQHfAz+zvlEAAAEEAMS22jgXbOZ/D3xWgM2kauSdzrwlU7Ms5hDW05ObqQyOFfQoKKMhfupyoa7J3x04VVBKu6Eomvr1es+VImH0esoeWFFahNOYq/I+jRRBwoOhAGZ5UB2/hRd7rFmxqp6sCXi8wmLO2tAorlTzAiNNvl7xF4cQZpc0z56Fwdi2fBUJAAURAXWwRBZQHNikA/f0ScLLjrXi4s0hgQecg+dkpDow94eu5+AR0DzZnfurpgfUJCNiDi5W/5c3Zj/xyrfMAgkbCgJ1m6FZqAQh7Mq73l7Kfu4/XIkyDF3tDgRuZNezB+JuElX10tV03xumHepp6M6CfhXqNJ15F33F99TA5hXYCPYD7SiSOpIhQkCOAgDAA63imxbpuKE2W7Y4I1BUHB7WQi8ZdkZd04njNTv+rFUuOPapQVfbWG0Vq8ld3YmJB4QWsa2mmqn+qToXbwufAgBpXkjvqK5yPiHFPx2QbFc1VqoCJB6PO5JRIqEiUZBFGdDlLxt3VSyqz7IZ/zEnxZq+tPCGGGSm/sAGiMvENcHVAfy0kTXU42TxEAYJyyNyqjXOobDJpEV1mKhFskRXt7tbMfOSYf91oX8f6xw6O2Nal+hU8dS0Bmfmk5/enHmvRLHQocO0CkZJWENJVFlfUUE=");
    byte[] sig1 = Base64.decode((String)"owGbwMvMwMRoGpHo9vfz52LGNTJJnBmpOTn5eiUVJfb23JvAHIXy/KKcFEWuToapzKwMIGG4Bqav0SwMy3yParsEKi2LMGI9xhh65sBxb05n5++ZLcWNJ/eLFKdWbm95tHbDV7GMwj/tUctUpFUXWPYFCLdNsDiVNuXbQvZtdXV/5xzY+9w1nCnijH9JoNiJ22n2jo0zo30/TZLo+jDl2vTzIvPeLEsPM3ZUE/1Ytqs4SG2TxIQbH7xf3uzcYXq25Fw9AA==");
    byte[] sig1crc = Base64.decode((String)"+3i0");
    byte[] subKey = Base64.decode((String)"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");
    byte[] enc1 = Base64.decode((String)"hIwDKwfQexPJboABA/4/7prhYYMORTiQ5avQKx0XYpCLujzGefYjnyuWZnx3Iev8Pmsguumm+OLLvtXhhkXQmkJRXbIg6Otj2ubPYWflRPgpJSgOrNOreOl5jeABOrtwbV6TJb9OTtZuB7cTQSCq2gmYiSZkluIiDjNs3R3mEanILbYzOQ3zKSggKpzlv9JQAZUqTyDyJ6/OUbJF5fI5uiv76DCsw1zyMWotUIu5/X01q+AVP5Ly3STzI7xkWg/JAPz4zUHism7kSYz2viAQaJx9/bNnH3AM6qm1Fuyikl4=");
    byte[] enc1crc = Base64.decode((String)"lv4o");
    byte[] enc2 = Base64.decode((String)"hIwDKwfQexPJboABBAC62jcJH8xKnKb1neDVmiovYON04+7VQ2v4BmeHwJrdag1gYa++6PeBlQ2Q9lSGBwLobVuJmQ7cOnPUJP727JeSGWlMyFtMbBSHekOaTenT5lj7Zk7oRHxMp/hByzlMacIDzOn8LPSh515RHM57eDLCOwqnAxGQwk67GRl8f5dFH9JQAa7xx8rjCqPbiIQW6t5LqCNvPZOiSCmftll6+se1XJhFEuq8WS4nXtPfTiJ3vib43soJdHzGB6AOs+BQ6aKmmNTVAxa5owhtSt1Z/6dfSSk=");
    byte[] subPubKey = Base64.decode((String)"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");
    byte[] subPubCrc = Base64.decode((String)"=rikt");
    byte[] pgp8Key = Base64.decode((String)"lQIEBEBXUNMBBADScQczBibewnbCzCswc/9ut8R0fwlltBRxMW0NMdKJY2LF7k2COeLOCIU95loJGV6ulbpDCXEO2Jyq8/qGw1qD3SCZNXxKs3GS8Iyh9UwdVL07nMMYl5NiQRsFB7wOb86+94tYWgvikVA5BRP5y3+O3GItnXnpWSJyREUy6WI2QQAGKf4JAwIVmnRs4jtTX2DD05zy2mepEQ8bsqVAKIx7lEwvMVNcvg4Y8vFLh9Mf/uNciwL4Se/ehfKQ/AT0JmBZduYMqRU2zhiBmxj4cXUQ0s36ysj7fyDngGocDnM3cwPxaTF1ZRBQHSLewP7dqE7M73usFSz8vwD/0xNOHFRLKbsORqDlLA1Cg2Yd0wWPS0o7+qqk9ndqrjjSwMM8ftnzFGjShAdg4Ca7fFkcNePP/rrwIH472FuRb7RbWzwXA4+4ZBdl8D4An0dwtfvAO+jCZSrLjmSpxEOveJxYGduyR4IA4lemvAG51YHTHd4NXheuEqsIkn1yarwaaj47lFPnxNOElOREMdZbnkWQb1jfgqO24imEZgrLMkK9bJfoDnlF4k6r6hZOp5FSFvc5kJB4cVo1QJl4pwCSdoU6luwCggrlZhDnkGCSuQUUW45NE7Br22NGqn4/gHs0KCsWbAezApGjqYUCfX1bcpPzUMzUlBaD5rz2vPeO58CDtBJ0ZXN0ZXIgPHRlc3RAdGVzdD6IsgQTAQIAHAUCQFdQ0wIbAwQLBwMCAxUCAwMWAgECHgECF4AACgkQs8JyyQfH97I1QgP8Cd+35maM2cbWV9iVRO+c5456KDi3oIUSNdPf1NQrCAtJqEUhmMStQbdiaFEkPrORISI/2htXruYn0aIpkCfbUheHOu0sef7s6pHmI2kOQPzR+C/j8D9QvWsPOOso81KU2axUY8zIer64Uzqc4szMIlLw06c8vea27RfgjBpSCrywAgAA");
    char[] pgp8Pass = "2002 Buffalo Sabres".toCharArray();
    char[] pass = new char[]{'h', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd'};
    byte[] fingerprintKey = Base64.decode((String)"mQEPA0CiJdUAAAEIAMI+znDlPd2kQoEcnxqxLcRz56Z7ttFKHpnYp0UkljZdquVcBy1jMfXGVV64xN1IvMcyenLXUE0IUeUBCQs6tHunFRAPSeCxJ3FdFe1B5MpqQG8ABnEpAds/hAUfRDZD5y/lolk1hjvFMrRh6WXckaA/QQ2t00NmTrJ1pYUpkw9tnVQbLUjWJhfZDBBcN0ADtATzgkugxMtcDxR6I5x8Ndn+IilqIm23kxGIcmMd/BHOec4cjRwJXXDb7u8tl+2knAf9cwhPHp3+Zy4uGSQPdzQnXOhBlA+4WDa0RROOevWgq8uq8/9Xp/OlTVL+OoIzjsI6mJP1Joa4qmqAnaHAmXcAEQEAAbQoQk9BM1JTS1kgPEJPQSBNb25pdG9yaW5nIEAgODg4LTI2OS01MjY2PokBFQMFEECiJdWqaoCdocCZdwEB0RsH/3HPxoUZ3G3K7T3jgOnJUckTSHWU3XspHzMVgqOxjTrcexi5IsAM5M+BulfWT2aO+Kqf5w8cKTKgW02DNpHUiPjHx0nzDE+Do95zbIErGeK+Twkc4O/aVsvU9GGO81VFI6WMvDQ4CUAUnAdk03MRrzI2nAuhn4NJ5LQS+uJrnqUJ4HmFAz6CQZQKd/kSXgq+A6i7aI1LG80YxWa9ooQgaCrb9dwY/kPQ+yC22zQ3FExtv+Fv3VtAKTilO3vnBA4Y9uTHuObHfI+1yxUS2PrlRUX0m48ZjpIX+cEN3QblGBJudI/A1QSd6P0LZeBr7F1Z1aF7ZDo0KzgiAIBvgXkeTpw=");
    byte[] fingerprintCheck = Base64.decode((String)"=CTv2");

    private boolean notEqual(byte[] byArray, byte[] byArray2) {
        if (byArray.length != byArray2.length) {
            return true;
        }
        for (int i = 0; i != byArray2.length; ++i) {
            if (byArray[i] == byArray2[i]) continue;
            return true;
        }
        return false;
    }

    private TestResult fingerPrintTest() throws Exception {
        PGPPublicKeyRing pGPPublicKeyRing = new PGPPublicKeyRing(this.fingerprintKey);
        PGPPublicKey pGPPublicKey = pGPPublicKeyRing.getPublicKey();
        if (this.notEqual(pGPPublicKey.getFingerprint(), Hex.decode((String)"4FFB9F0884266C715D1CEAC804A3BBFA"))) {
            return new SimpleTestResult(false, this.getName() + ": version 3 fingerprint test failed");
        }
        pGPPublicKeyRing = new PGPPublicKeyRing(this.testPubKey);
        pGPPublicKey = pGPPublicKeyRing.getPublicKey();
        if (this.notEqual(pGPPublicKey.getFingerprint(), Hex.decode((String)"3062363c1046a01a751946bb35586146fdf3f373"))) {
            return new SimpleTestResult(false, this.getName() + ": version 4 fingerprint test failed");
        }
        return new SimpleTestResult(true, this.getName() + ": Okay");
    }

    public TestResult perform() {
        try {
            Object object;
            int n;
            KeyFactory keyFactory = KeyFactory.getInstance("RSA", "BC");
            PublicKey publicKey = null;
            Object var3_4 = null;
            PGPPublicKeyRing pGPPublicKeyRing = new PGPPublicKeyRing(this.testPubKey);
            publicKey = pGPPublicKeyRing.getPublicKey().getKey("BC");
            Iterator iterator = pGPPublicKeyRing.getPublicKey().getUserIDs();
            String string = (String)iterator.next();
            iterator = pGPPublicKeyRing.getPublicKey().getSignaturesForID(string);
            PGPSignature pGPSignature = (PGPSignature)iterator.next();
            pGPSignature.initVerify(pGPPublicKeyRing.getPublicKey(), "BC");
            if (!pGPSignature.verifyCertification(string, pGPPublicKeyRing.getPublicKey())) {
                return new SimpleTestResult(false, this.getName() + ": failed to verify certification");
            }
            ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
            BCPGOutputStream bCPGOutputStream = new BCPGOutputStream(byteArrayOutputStream);
            pGPPublicKeyRing.encode(bCPGOutputStream);
            if (this.notEqual(byteArrayOutputStream.toByteArray(), this.testPubKey)) {
                return new SimpleTestResult(false, this.getName() + ": public key rewrite failed");
            }
            PGPPublicKeyRing pGPPublicKeyRing2 = new PGPPublicKeyRing(this.testPubKeyV3);
            PublicKey publicKey2 = pGPPublicKeyRing.getPublicKey().getKey("BC");
            byteArrayOutputStream = new ByteArrayOutputStream();
            bCPGOutputStream = new BCPGOutputStream(byteArrayOutputStream);
            pGPPublicKeyRing2.encode(bCPGOutputStream);
            char[] cArray = "FIXCITY_QA".toCharArray();
            PGPSecretKeyRing pGPSecretKeyRing = new PGPSecretKeyRing(this.testPrivKeyV3);
            PGPPrivateKey pGPPrivateKey = pGPSecretKeyRing.getSecretKey().extractPrivateKey(cArray, "BC");
            byteArrayOutputStream = new ByteArrayOutputStream();
            bCPGOutputStream = new BCPGOutputStream(byteArrayOutputStream);
            pGPSecretKeyRing.encode(bCPGOutputStream);
            if (this.notEqual(byteArrayOutputStream.toByteArray(), this.testPrivKeyV3)) {
                return new SimpleTestResult(false, this.getName() + ": private key V3 rewrite failed");
            }
            pGPSecretKeyRing = new PGPSecretKeyRing(this.testPrivKey);
            pGPPrivateKey = pGPSecretKeyRing.getSecretKey().extractPrivateKey(this.pass, "BC");
            byteArrayOutputStream = new ByteArrayOutputStream();
            bCPGOutputStream = new BCPGOutputStream(byteArrayOutputStream);
            pGPSecretKeyRing.encode(bCPGOutputStream);
            if (this.notEqual(byteArrayOutputStream.toByteArray(), this.testPrivKey)) {
                return new SimpleTestResult(false, this.getName() + ": private key rewrite failed");
            }
            Cipher cipher = Cipher.getInstance("RSA", "BC");
            cipher.init(1, publicKey);
            byte[] byArray = "hello world".getBytes();
            byte[] byArray2 = cipher.doFinal(byArray);
            cipher.init(2, pGPPrivateKey.getKey());
            byArray2 = cipher.doFinal(byArray2);
            if (this.notEqual(byArray, byArray2)) {
                return new SimpleTestResult(false, this.getName() + ": decryption failed.");
            }
            PGPObjectFactory pGPObjectFactory = new PGPObjectFactory(this.sig1);
            PGPCompressedData pGPCompressedData = (PGPCompressedData)pGPObjectFactory.nextObject();
            pGPObjectFactory = new PGPObjectFactory(pGPCompressedData.getDataStream());
            PGPOnePassSignatureList pGPOnePassSignatureList = (PGPOnePassSignatureList)pGPObjectFactory.nextObject();
            PGPOnePassSignature pGPOnePassSignature = pGPOnePassSignatureList.get(0);
            PGPLiteralData pGPLiteralData = (PGPLiteralData)pGPObjectFactory.nextObject();
            InputStream inputStream = pGPLiteralData.getInputStream();
            pGPOnePassSignature.initVerify(pGPPublicKeyRing.getPublicKey(pGPOnePassSignature.getKeyID()), "BC");
            while ((n = inputStream.read()) >= 0) {
                pGPOnePassSignature.update((byte)n);
            }
            PGPSignatureList pGPSignatureList = (PGPSignatureList)pGPObjectFactory.nextObject();
            if (!pGPOnePassSignature.verify(pGPSignatureList.get(0))) {
                return new SimpleTestResult(false, this.getName() + ": Failed signature check");
            }
            pGPSecretKeyRing = new PGPSecretKeyRing(this.subKey);
            byte[] byArray3 = new byte[]{104, 101, 108, 108, 111, 32, 119, 111, 114, 108, 100, 33, 10};
            PGPObjectFactory pGPObjectFactory2 = new PGPObjectFactory(this.enc1);
            PGPEncryptedDataList pGPEncryptedDataList = (PGPEncryptedDataList)pGPObjectFactory2.nextObject();
            PGPPublicKeyEncryptedData pGPPublicKeyEncryptedData = (PGPPublicKeyEncryptedData)pGPEncryptedDataList.get(0);
            pGPPrivateKey = pGPSecretKeyRing.getSecretKey(pGPPublicKeyEncryptedData.getKeyID()).extractPrivateKey(this.pass, "BC");
            InputStream inputStream2 = pGPPublicKeyEncryptedData.getDataStream(pGPPrivateKey, "BC");
            pGPObjectFactory = new PGPObjectFactory(inputStream2);
            pGPCompressedData = (PGPCompressedData)pGPObjectFactory.nextObject();
            pGPObjectFactory = new PGPObjectFactory(pGPCompressedData.getDataStream());
            PGPLiteralData pGPLiteralData2 = (PGPLiteralData)pGPObjectFactory.nextObject();
            byteArrayOutputStream = new ByteArrayOutputStream();
            if (!pGPLiteralData2.getFileName().equals("test.txt")) {
                throw new RuntimeException("wrong filename in packet");
            }
            InputStream inputStream3 = pGPLiteralData2.getDataStream();
            while ((n = inputStream3.read()) >= 0) {
                byteArrayOutputStream.write(n);
            }
            if (this.notEqual(byteArrayOutputStream.toByteArray(), byArray3)) {
                return new SimpleTestResult(false, this.getName() + ": wrong plain text in decrypted packet");
            }
            byte[] byArray4 = new byte[]{104, 101, 108, 108, 111};
            ByteArrayOutputStream byteArrayOutputStream2 = new ByteArrayOutputStream();
            PGPEncryptedDataGenerator pGPEncryptedDataGenerator = new PGPEncryptedDataGenerator(3, new SecureRandom(), "BC");
            PGPPublicKey pGPPublicKey = pGPSecretKeyRing.getSecretKey(pGPPublicKeyEncryptedData.getKeyID()).getPublicKey();
            pGPEncryptedDataGenerator.addMethod(pGPPublicKey);
            OutputStream outputStream = pGPEncryptedDataGenerator.open((OutputStream)byteArrayOutputStream2, byArray4.length);
            outputStream.write(byArray4);
            outputStream.close();
            pGPObjectFactory2 = new PGPObjectFactory(byteArrayOutputStream2.toByteArray());
            pGPEncryptedDataList = (PGPEncryptedDataList)pGPObjectFactory2.nextObject();
            pGPPublicKeyEncryptedData = (PGPPublicKeyEncryptedData)pGPEncryptedDataList.get(0);
            pGPPrivateKey = pGPSecretKeyRing.getSecretKey(pGPPublicKeyEncryptedData.getKeyID()).extractPrivateKey(this.pass, "BC");
            inputStream2 = pGPPublicKeyEncryptedData.getDataStream(pGPPrivateKey, "BC");
            byteArrayOutputStream.reset();
            while ((n = inputStream2.read()) >= 0) {
                byteArrayOutputStream.write(n);
            }
            byArray2 = byteArrayOutputStream.toByteArray();
            if (this.notEqual(byArray2, byArray4)) {
                return new SimpleTestResult(false, this.getName() + ": wrong plain text in generated short text packet");
            }
            byteArrayOutputStream2 = new ByteArrayOutputStream();
            pGPEncryptedDataGenerator = new PGPEncryptedDataGenerator(3, new SecureRandom(), "BC");
            pGPPublicKey = pGPSecretKeyRing.getSecretKey(pGPPublicKeyEncryptedData.getKeyID()).getPublicKey();
            pGPEncryptedDataGenerator.addMethod(pGPPublicKey);
            outputStream = pGPEncryptedDataGenerator.open((OutputStream)byteArrayOutputStream2, byArray3.length);
            outputStream.write(byArray3);
            outputStream.close();
            pGPObjectFactory2 = new PGPObjectFactory(byteArrayOutputStream2.toByteArray());
            pGPEncryptedDataList = (PGPEncryptedDataList)pGPObjectFactory2.nextObject();
            pGPPublicKeyEncryptedData = (PGPPublicKeyEncryptedData)pGPEncryptedDataList.get(0);
            pGPPrivateKey = pGPSecretKeyRing.getSecretKey(pGPPublicKeyEncryptedData.getKeyID()).extractPrivateKey(this.pass, "BC");
            inputStream2 = pGPPublicKeyEncryptedData.getDataStream(pGPPrivateKey, "BC");
            byteArrayOutputStream.reset();
            while ((n = inputStream2.read()) >= 0) {
                byteArrayOutputStream.write(n);
            }
            byArray2 = byteArrayOutputStream.toByteArray();
            if (this.notEqual(byArray2, byArray3)) {
                return new SimpleTestResult(false, this.getName() + ": wrong plain text in generated packet");
            }
            pGPObjectFactory2 = new PGPObjectFactory(this.subPubKey);
            while ((object = pGPObjectFactory.nextObject()) != null) {
            }
            char[] cArray2 = "hello".toCharArray();
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("RSA", "BC");
            keyPairGenerator.initialize(1024);
            KeyPair keyPair = keyPairGenerator.generateKeyPair();
            PGPSecretKey pGPSecretKey = new PGPSecretKey(16, 1, keyPair.getPublic(), keyPair.getPrivate(), new Date(), "fred", 3, cArray2, null, null, new SecureRandom(), "BC");
            PGPPublicKey pGPPublicKey2 = pGPSecretKey.getPublicKey();
            iterator = pGPPublicKey2.getUserIDs();
            string = (String)iterator.next();
            iterator = pGPPublicKey2.getSignaturesForID(string);
            pGPSignature = (PGPSignature)iterator.next();
            pGPSignature.initVerify(pGPPublicKey2, "BC");
            if (!pGPSignature.verifyCertification(string, pGPPublicKey2)) {
                return new SimpleTestResult(false, this.getName() + ": failed to verify certification");
            }
            pGPPrivateKey = pGPSecretKey.extractPrivateKey(cArray2, "BC");
            if ((pGPPublicKey2 = PGPPublicKey.removeCertification(pGPPublicKey2, string, pGPSignature)) == null) {
                return new SimpleTestResult(false, this.getName() + ": failed certification removal");
            }
            byte[] byArray5 = pGPPublicKey2.getEncoded();
            pGPPublicKey2 = PGPPublicKey.addCertification(pGPPublicKey2, string, pGPSignature);
            byArray5 = pGPPublicKey2.getEncoded();
            PGPSignatureGenerator pGPSignatureGenerator = new PGPSignatureGenerator(1, 2, "BC");
            pGPSignatureGenerator.initSign(32, pGPSecretKey.extractPrivateKey(cArray2, "BC"));
            pGPSignature = pGPSignatureGenerator.generateCertification(pGPPublicKey2);
            pGPPublicKey2 = PGPPublicKey.addCertification(pGPPublicKey2, pGPSignature);
            byArray5 = pGPPublicKey2.getEncoded();
            PGPPublicKeyRing pGPPublicKeyRing3 = new PGPPublicKeyRing(byArray5);
            pGPPublicKey2 = pGPPublicKeyRing3.getPublicKey();
            Iterator iterator2 = pGPPublicKey2.getSignaturesOfType(32);
            pGPSignature = (PGPSignature)iterator2.next();
            pGPSignature.initVerify(pGPPublicKey2, "BC");
            if (!pGPSignature.verifyCertification(pGPPublicKey2)) {
                return new SimpleTestResult(false, this.getName() + ": failed to verify revocation certification");
            }
            PGPKeyPair pGPKeyPair = new PGPKeyPair(1, keyPair.getPublic(), keyPair.getPrivate(), new Date(), "BC");
            PGPPublicKey pGPPublicKey3 = pGPKeyPair.getPublicKey();
            PGPPrivateKey pGPPrivateKey2 = pGPKeyPair.getPrivateKey();
            pGPPublicKey3.getEncoded();
            keyPair = keyPairGenerator.generateKeyPair();
            pGPSecretKey = new PGPSecretKey(16, 1, keyPair.getPublic(), keyPair.getPrivate(), new Date(), "fred", 9, cArray2, null, null, new SecureRandom(), "BC");
            pGPSecretKey.extractPrivateKey(cArray2, "BC");
            pGPSecretKey.encode(new ByteArrayOutputStream());
            String string2 = "newPass";
            pGPSecretKey = PGPSecretKey.copyWithNewPassword(pGPSecretKey, cArray2, string2.toCharArray(), pGPSecretKey.getKeyEncryptionAlgorithm(), new SecureRandom(), "BC");
            pGPSecretKey.extractPrivateKey(string2.toCharArray(), "BC");
            pGPSecretKey.encode(new ByteArrayOutputStream());
            pGPPublicKey2 = pGPSecretKey.getPublicKey();
            pGPPublicKey2.encode(new ByteArrayOutputStream());
            iterator = pGPPublicKey2.getUserIDs();
            string = (String)iterator.next();
            iterator = pGPPublicKey2.getSignaturesForID(string);
            pGPSignature = (PGPSignature)iterator.next();
            pGPSignature.initVerify(pGPPublicKey2, "BC");
            if (!pGPSignature.verifyCertification(string, pGPPublicKey2)) {
                return new SimpleTestResult(false, this.getName() + ": failed to verify certification");
            }
            pGPPrivateKey = pGPSecretKey.extractPrivateKey(string2.toCharArray(), "BC");
            String string3 = "hello world!";
            byteArrayOutputStream = new ByteArrayOutputStream();
            ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(string3.getBytes());
            pGPSignatureGenerator = new PGPSignatureGenerator(1, 2, "BC");
            pGPSignatureGenerator.initSign(0, pGPPrivateKey);
            PGPCompressedDataGenerator pGPCompressedDataGenerator = new PGPCompressedDataGenerator(1);
            BCPGOutputStream bCPGOutputStream2 = new BCPGOutputStream(pGPCompressedDataGenerator.open(byteArrayOutputStream));
            pGPSignatureGenerator.generateOnePassVersion(false).encode(bCPGOutputStream2);
            PGPLiteralDataGenerator pGPLiteralDataGenerator = new PGPLiteralDataGenerator();
            OutputStream outputStream2 = pGPLiteralDataGenerator.open((OutputStream)bCPGOutputStream2, 'b', "_CONSOLE", string3.getBytes().length, new Date());
            while ((n = byteArrayInputStream.read()) >= 0) {
                outputStream2.write(n);
                pGPSignatureGenerator.update((byte)n);
            }
            pGPLiteralDataGenerator.close();
            pGPSignatureGenerator.generate().encode(bCPGOutputStream2);
            pGPCompressedDataGenerator.close();
            pGPObjectFactory = new PGPObjectFactory(byteArrayOutputStream.toByteArray());
            pGPCompressedData = (PGPCompressedData)pGPObjectFactory.nextObject();
            pGPObjectFactory = new PGPObjectFactory(pGPCompressedData.getDataStream());
            pGPOnePassSignatureList = (PGPOnePassSignatureList)pGPObjectFactory.nextObject();
            pGPOnePassSignature = pGPOnePassSignatureList.get(0);
            pGPLiteralData = (PGPLiteralData)pGPObjectFactory.nextObject();
            inputStream = pGPLiteralData.getInputStream();
            pGPOnePassSignature.initVerify(pGPSecretKey.getPublicKey(), "BC");
            while ((n = inputStream.read()) >= 0) {
                pGPOnePassSignature.update((byte)n);
            }
            pGPSignatureList = (PGPSignatureList)pGPObjectFactory.nextObject();
            if (!pGPOnePassSignature.verify(pGPSignatureList.get(0))) {
                return new SimpleTestResult(false, this.getName() + ": Failed generated signature check");
            }
            byteArrayOutputStream = new ByteArrayOutputStream();
            byteArrayInputStream = new ByteArrayInputStream(string3.getBytes());
            PGPV3SignatureGenerator pGPV3SignatureGenerator = new PGPV3SignatureGenerator(1, 2, "BC");
            pGPSignatureGenerator.initSign(0, pGPPrivateKey);
            pGPCompressedDataGenerator = new PGPCompressedDataGenerator(1);
            bCPGOutputStream2 = new BCPGOutputStream(pGPCompressedDataGenerator.open(byteArrayOutputStream));
            pGPSignatureGenerator.generateOnePassVersion(false).encode(bCPGOutputStream2);
            pGPLiteralDataGenerator = new PGPLiteralDataGenerator();
            outputStream2 = pGPLiteralDataGenerator.open((OutputStream)bCPGOutputStream2, 'b', "_CONSOLE", string3.getBytes().length, new Date());
            while ((n = byteArrayInputStream.read()) >= 0) {
                outputStream2.write(n);
                pGPSignatureGenerator.update((byte)n);
            }
            pGPSignatureGenerator.generate().encode(bCPGOutputStream2);
            pGPLiteralDataGenerator.close();
            pGPCompressedDataGenerator.close();
            pGPObjectFactory = new PGPObjectFactory(byteArrayOutputStream.toByteArray());
            pGPCompressedData = (PGPCompressedData)pGPObjectFactory.nextObject();
            pGPObjectFactory = new PGPObjectFactory(pGPCompressedData.getDataStream());
            pGPOnePassSignatureList = (PGPOnePassSignatureList)pGPObjectFactory.nextObject();
            pGPOnePassSignature = pGPOnePassSignatureList.get(0);
            pGPLiteralData = (PGPLiteralData)pGPObjectFactory.nextObject();
            inputStream = pGPLiteralData.getInputStream();
            pGPOnePassSignature.initVerify(pGPSecretKey.getPublicKey(), "BC");
            while ((n = inputStream.read()) >= 0) {
                pGPOnePassSignature.update((byte)n);
            }
            pGPSignatureList = (PGPSignatureList)pGPObjectFactory.nextObject();
            if (!pGPOnePassSignature.verify(pGPSignatureList.get(0))) {
                return new SimpleTestResult(false, this.getName() + ": Failed v3 generated signature check");
            }
            pGPSecretKeyRing = new PGPSecretKeyRing(this.pgp8Key);
            pGPSecretKey = pGPSecretKeyRing.getSecretKey();
            pGPPrivateKey = pGPSecretKey.extractPrivateKey(this.pgp8Pass, "BC");
            TestResult testResult = this.testSig(8, pGPSecretKey.getPublicKey(), pGPPrivateKey);
            if (!testResult.isSuccessful()) {
                return testResult;
            }
            testResult = this.testSig(9, pGPSecretKey.getPublicKey(), pGPPrivateKey);
            if (!testResult.isSuccessful()) {
                return testResult;
            }
            testResult = this.testSig(10, pGPSecretKey.getPublicKey(), pGPPrivateKey);
            if (!testResult.isSuccessful()) {
                return testResult;
            }
            return this.fingerPrintTest();
        }
        catch (Exception exception) {
            return new SimpleTestResult(false, this.getName() + ": exception - " + exception.toString(), exception);
        }
    }

    private TestResult testSig(int n, PGPPublicKey pGPPublicKey, PGPPrivateKey pGPPrivateKey) throws Exception {
        int n2;
        String string = "hello world!";
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(string.getBytes());
        PGPSignatureGenerator pGPSignatureGenerator = new PGPSignatureGenerator(1, n, "BC");
        pGPSignatureGenerator.initSign(0, pGPPrivateKey);
        PGPCompressedDataGenerator pGPCompressedDataGenerator = new PGPCompressedDataGenerator(1);
        BCPGOutputStream bCPGOutputStream = new BCPGOutputStream(pGPCompressedDataGenerator.open(byteArrayOutputStream));
        pGPSignatureGenerator.generateOnePassVersion(false).encode(bCPGOutputStream);
        PGPLiteralDataGenerator pGPLiteralDataGenerator = new PGPLiteralDataGenerator();
        OutputStream outputStream = pGPLiteralDataGenerator.open((OutputStream)bCPGOutputStream, 'b', "_CONSOLE", string.getBytes().length, new Date());
        while ((n2 = byteArrayInputStream.read()) >= 0) {
            outputStream.write(n2);
            pGPSignatureGenerator.update((byte)n2);
        }
        pGPLiteralDataGenerator.close();
        pGPSignatureGenerator.generate().encode(bCPGOutputStream);
        pGPCompressedDataGenerator.close();
        PGPObjectFactory pGPObjectFactory = new PGPObjectFactory(byteArrayOutputStream.toByteArray());
        PGPCompressedData pGPCompressedData = (PGPCompressedData)pGPObjectFactory.nextObject();
        pGPObjectFactory = new PGPObjectFactory(pGPCompressedData.getDataStream());
        PGPOnePassSignatureList pGPOnePassSignatureList = (PGPOnePassSignatureList)pGPObjectFactory.nextObject();
        PGPOnePassSignature pGPOnePassSignature = pGPOnePassSignatureList.get(0);
        PGPLiteralData pGPLiteralData = (PGPLiteralData)pGPObjectFactory.nextObject();
        InputStream inputStream = pGPLiteralData.getInputStream();
        pGPOnePassSignature.initVerify(pGPPublicKey, "BC");
        while ((n2 = inputStream.read()) >= 0) {
            pGPOnePassSignature.update((byte)n2);
        }
        PGPSignatureList pGPSignatureList = (PGPSignatureList)pGPObjectFactory.nextObject();
        if (!pGPOnePassSignature.verify(pGPSignatureList.get(0))) {
            return new SimpleTestResult(false, this.getName() + ": Failed generated signature check - " + n);
        }
        return new SimpleTestResult(true, this.getName() + ": Okay");
    }

    public String getName() {
        return "PGPRSATest";
    }

    public static void main(String[] stringArray) {
        Security.addProvider((Provider)new BouncyCastleProvider());
        PGPRSATest pGPRSATest = new PGPRSATest();
        TestResult testResult = pGPRSATest.perform();
        System.out.println(testResult.toString());
    }
}

