// $Id: $
//
// $Log: $
// Revision 1.0  1998/04/04  raif
// + original version.
//
// $Endlog$
/*
 * Copyright (c) 1997, 1998 Systemics Ltd on behalf of
 * the Cryptix Development Team. All rights reserved.
 */
package NIST;

import java.security.InvalidKeyException;
import java.security.InvalidParameterException;

//...........................................................................
/**
 * An interface that defines the level-1 API for calling NIST_<mode>
 * classes, where NIST_<mode> are the concrete objects that exercise
 * an AES candidate algorithm implementation in the required NIST modes
 * of operation.<p>
 *
 * <b>Copyright</b> &copy; 1997, 1998
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: $</b>
 * @author  Raif S. Naffah
 */
public interface NIST_CipherSpi
{
    /**
     * Initialize this cipher for encryption/decryption, using designated
     * key material.
     *
     * @param state  Desired operation state of this instance. Valid values
     *          are NIST.NIST_Cipher.ENCRYPT_STATE or
     *          NIST.NIST_Cipher.DECRYPT_STATE.
     * @param key   User key material to use for the session.
     * @exception  InvalidKeyException  If the key is invalid.
     * @see NIST.NIST_Cipher#ENCRYPT_STATE
     * @see NIST.NIST_Cipher#DECRYPT_STATE
     */
    public abstract void init (int state, byte[] key)
    throws InvalidKeyException;

    /**
     * Set the starting value of the IV.<p>
     *
     * @param  iv   The user-supplied IV value.
     * @exception IllegalStateException If the cipher is already initialized
     *      in an operation mode and the IV has already been set.
     * @exception InvalidParameterException  If the supplied IV value
     *      is of the wrong length.
     */
    public abstract void setIV (byte[] iv) throws InvalidParameterException;

    /**
     * Continue a multi-part encryption/decryption operation.<p>
     *
     * @param in  Input buffer containing the plaintext/ciphertext (depending
     *      on the current cipher's operational state).
     * @param inOff  Offset into <code>in</code> specifying where data starts.
     *      Caller must ensure it's a positive value that will not cause an
     *      ArrayIndexOutOfBoundsException when accessing <code>in</code>.
     * @param inLen  Length of the subarray. Caller must ensure inLen is a
     *      non-zero positive value.
     * @return The encrypted/decrypted input text.
     * @exception  NullPointerException  If the IV has not been set.
     * @exception  IllegalStateException  If the cipher's operational state is
     *      not properly set.
     */
    public abstract byte[] update (byte[] in, int inOff, int inLen);

    /**
     * Encrypt/decrypt data in a single-part operation or finish a
     * multi-part operation.<p>
     *
     * @param in  Input buffer containing the plaintext/ciphertext (depending
     *      on the current cipher's operational state).
     * @param inOff  Offset into <code>in</code> specifying where data starts.
     *      Caller must ensure it's a positive value that will not cause an
     *      ArrayIndexOutOfBoundsException when accessing <code>in</code>.
     * @param inLen  Length of the subarray. Caller must ensure inLen is
     *      zero or a positive value.
     * @return The encrypted/decrypted input text including any other data
     *      that may have been buffered in one or more earlier invocations
     *      to <code>update</code>.
     * @exception  NullPointerException  If the IV has not been set.
     * @exception  IllegalStateException  If the cipher is not properly
     *      initialized.
     * @exception  InvalidParameterException  If the operation when carried
     *      out, will cause an illegal block size exception; eg. the total
     *      length of data to encrypt is not a multiple of the cipher's block
     *      size.
     */
    public abstract byte[] doFinal (byte[] in, int inOff, int inLen);

    /**
     * Basic simple self test(s).
     *
     * @return True iff all tests pass; false otherwise.
     */
    public abstract boolean self_test();
}
