// $Id: $
//
// $Log: $
// Revision 1.0  1998/03/30  raif
// + original version based on cryptix.tools.CompileKit
//
// $Endlog$
/*
 * Copyright (c) 1997, 1998 Systemics Ltd on behalf of
 * the Cryptix Development Team. All rights reserved.
 */
package NIST;

import java.io.File;
import java.io.IOException;
import java.util.Properties;
import java.util.Vector;

/**
 * Compile the AES-Kit source code.<p>
 *
 * <b>Copyright</b> &copy; 1998
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: $</b>
 * @author  Raif S. Naffah
 */
public final class CompileKit
{
// Constants and variables
//...........................................................................

    private static final String NAME = "CompileKit";

    private static final String NL = System.getProperty("line.separator");
    private static final String VERSION = "$Revision: 1.0$";
    private static final String COPYRIGHT =
        "Copyright (c) 1998 Systemics Ltd. on behalf of" + NL +
        "the Cryptix Development Team.  All rights reserved.";
    private static final String ID =
        "Cryptix AES-Kit javac tool" + NL +
        VERSION + NL +
        COPYRIGHT;

    private int exitValue = 0; // anything but 0 means error

    // current values of switches as set from the command line arguments
    private boolean mute = true;       // -v if true become verbose
    private boolean optimal = false;   // -o if true use -O javac switch
    private String dirName;            // -s  source directory if != user.dir
    private String algorithm = null;   // candidate algorithm name == provider

    private File dir;                  // source directory File object


// main
//...........................................................................

    public static final void main (String[] args) {
        System.out.println(ID + NL + NL);
        CompileKit cmd = new CompileKit();
        cmd.run(args);
    }


// methods
//...........................................................................

    /** main action. */
    private boolean run (String[] args) {
        long time = System.currentTimeMillis();
        if (!processOptions(args))
            return false;
        notify("AES Candidate Algorithm: \""+algorithm+"\"");
        notify("         Home directory: \""+dir+"\"");
        try {
            // javac  <dir>/src/*.java
            dir = new File(dir, algorithm);

            Vector argv = new Vector();

            argv.addElement((optimal) ? "-O" : "-g");

            argv.addElement("-deprecation");

            argv.addElement("-depend");

            argv.addElement("-d");
            argv.addElement(new File(dir, "classes").getAbsolutePath());

            dir = new File(dir, "src");
            String[] files = dir.list();
            for (int i = 0; i < files.length; i++)
                if (files[i].endsWith(".java"))
                    argv.addElement(new File(dir, files[i]).getAbsolutePath());

            args = new String[argv.size()];
            argv.copyInto(args);

            // invoke the javac tool
            notify("About to invoke javac" );
            if (! new sun.tools.javac.Main(System.out, "javac").compile(args))
                throw new RuntimeException("Javac invocation failed");
        }
        catch (Exception x) {
            x.printStackTrace();
            error("Exception encountered: " + x.getMessage());
            return false;
        }
        notify("Java interpreter used: Version " + System.getProperty("java.version"));
        notify("Java Just-In-Time (JIT) compiler: " + System.getProperty("java.compiler"));
        notify("Total execution time (ms): " + (System.currentTimeMillis() - time));
        return true;
    }
    
    /** Process command line arguments and initialise instance fields. */
    private boolean processOptions (String[] args) {
        int argc = args.length;
        if (argc == 0) {
            printHelp();
            return false;
        }
        System.out.println(
            "(type \"java NIST."+NAME+"\" with no arguments for help)"+NL+NL);
        int i = -1;
        String cmd = "";
        boolean next = true;
        while (true) {
            if (next) {
                i++;
                if (i >= argc)
                    break;
                else
                    cmd = args[i];
            } else
                cmd = "-" + cmd.substring(2);

            if (cmd.startsWith("-v")) {
                mute = false;
                next = (cmd.length() == 2);
            } else if (cmd.startsWith("-o")) {
                optimal = true;
                next = (cmd.length() == 2);
            } else if (cmd.startsWith("-h")) { // home directory
                dirName = args[i + 1];
                i++;
                next = true;
            } else // it's the algorithm
                algorithm = cmd;
        }
        // fill the blanks
        if (algorithm == null) {
            error("Missing candidate Algorithm name");
            return false;
        }
        if (dirName == null)
            dirName =  System.getProperty("user.dir");
        dir = new File(dirName);
        if (! dir.isDirectory()) {
            error("Source \"" + dir.getAbsolutePath() + "\" is not a directory");
            return false;
        }
        return true;
    }

    /** Print an error message to System.err. */
    private void error (String s) {
        System.err.println(NL + "*** "+s+"...");
        exitValue = -1;
    }

    /** @return The exeucution result code: 0 means OK. */
    public int getExitValue() { return exitValue; }

    /** Write a notification message to System.out. */
    private void notify (String s) {
        if (!mute)
            System.out.println(NAME+": "+s+"...");
    }
    
    /** write help to System.out. */
    void printHelp () {
        System.out.println(
        "NAME" + NL +
        "  "+NAME+": Compiles a specified AES candidate algorithm source" + NL +
        "  code files." + NL + NL +
        "SYNTAX" + NL +
        "  java NIST."+NAME + NL +
        "    [ -v ]" + NL +
        "    [ -o ]" + NL +
        "    [ -h <home-directory>]" + NL +
        "    <algorithm>" + NL + NL +
        "DESCRIPTION" + NL +
        "  Invokes the javac command to compile all .java source files in a" + NL +
        "  designated directory according to the conventions of the AES-Kit." + NL + NL +
        "OPTIONS" + NL +
        "  -v   Verbose." + NL + NL +
        "  -o   Optimize compiled code." + NL + NL +
        "  -h <home-directory>" + NL +
        "       Pathname of the algorithm home directory containing the"+NL+
        "       'src' and 'classes' directories. If unspecified, the current"+NL+
        "       user directory is considered." + NL + NL +
        "  <algorithm>" + NL +
        "       Candidate algorithm name (<Foo> in the documentation)." + NL + NL +
        COPYRIGHT);
    }
}
